/*
 * Copyright (c) 2004-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */


#include <sys/fcntl.h>
#include <netinet/if_ether.h>
#include <netinet/in.h>
#include <net/if.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <linux/sockios.h>
#include <linux/if_packet.h>

#include "dhcpc.h"
#include "packet.h"
#include "script.h"
#include "zeroconf.h"
#include "debug.h"

extern struct client_config_t client_config;

int zc_arp_socket(const char* inf)
{
    struct sockaddr_ll ll_from;
    struct ifreq ifr;
    int fd;

    fd = socket(AF_PACKET, SOCK_RAW, htons(ETH_P_ARP));
    if (fd < 0)
        return -1;

    memset(ifr.ifr_name, 0, IFNAMSIZ);
    strncpy(ifr.ifr_name, inf, IFNAMSIZ-1);
    if (ioctl(fd, SIOCGIFHWADDR, &ifr) < 0) {
        close(fd);
        return -1;
    }

    memset(ll_from.sll_addr, 0, ETH_ALEN);
    memcpy(ll_from.sll_addr, ifr.ifr_hwaddr.sa_data, ETH_ALEN);

    if (ioctl(fd, SIOCGIFINDEX, &ifr) < 0) {
        close(fd);
        return -1;
    }

    ll_from.sll_family = AF_PACKET;
    ll_from.sll_protocol = htons(ETH_P_ARP);
    ll_from.sll_ifindex = ifr.ifr_ifindex;
    ll_from.sll_hatype = ARPHRD_ETHER;
    ll_from.sll_pkttype = PACKET_HOST;
    ll_from.sll_halen = ETH_ALEN;
    
    if (bind(fd, (struct sockaddr*)&ll_from, sizeof(ll_from)) < 0) {
        close(fd);
        return -1;
    }

    return fd;    
}

void zc_handle_input(struct zc_config* zc, time_t now)
{
    if (zc->zc_state != ZEROCONF_DISABLED) {
        unsigned char buf[2048];

        int len = recv(zc->zc_fd, buf, 2048, 0);
        if (len >= ETH_HLEN + (int)sizeof(struct ether_arp)) {
            const unsigned char* pch = buf;
        
            pch += ETH_HLEN;

            const struct ether_arp* ap = (const struct ether_arp*)pch;

            if ((ap->arp_pro != htons(ETHERTYPE_IP) &&
                 ap->arp_pro != htons(ETHERTYPE_TRAIL)) 
                || ap->arp_hln != sizeof(ap->arp_sha)
                || ap->arp_pln != sizeof(ap->arp_spa))
                return;

            if (ap->arp_op == htons(ARPOP_REQUEST) ||
                ap->arp_op == htons(ARPOP_REPLY)) {
                if (0 == memcmp(ap->arp_spa, zc->zc_addr, 4))
                    goto collision;
            }

            if (zc->zc_state == ZEROCONF_PROBING) {
                if (ap->arp_op == htons(ARPOP_REQUEST)) {
                    if (0 == memcmp(ap->arp_tpa, zc->zc_addr, 4) &&
                        0 != memcmp(ap->arp_sha, zc->zc_hw_addr, ETH_ALEN))
                        goto collision;
                }
            }
        }
    }

    return;

collision:
    LOG(LOG_DEBUG, "Detected zc collision...");
    zc_start_probing(zc, now);
}

void background(void);

void zc_handle_timer(struct zc_config* zc, time_t now)
{
    switch(zc->zc_state) {
    case ZEROCONF_DISABLED:
    case ZEROCONF_RUNNING:
    default:
        zc->zc_timeout = 0x7fffffff;
        break;
    case ZEROCONF_PROBING:
        if (zc->zc_count == 0) {
            zc->zc_timeout = now + 2;
            zc->zc_count++;
        } else if (zc->zc_count <= 4) {
            zc_send_arp_probe(zc);

            zc->zc_timeout = now + 2;
            zc->zc_count++;
        } else {
            char rgch[32];
            sprintf(rgch, "%u.%u.%u.%u", zc->zc_addr[0], zc->zc_addr[1],
                    zc->zc_addr[2], zc->zc_addr[3]);
            run_script_zeroconf(rgch, "255.255.0.0", "bound");

            LOG(LOG_INFO, "zc announcing %s", rgch);

            zc->zc_state = ZEROCONF_ANNOUNCING;
            zc->zc_count = 0;

            if (!client_config.foreground)
                background();

            zc->zc_timeout = now;
        }

        break;
    case ZEROCONF_ANNOUNCING:
        if (zc->zc_count == 0) {
            zc_send_arp_announce(zc);

            zc->zc_timeout = now + 2;
            zc->zc_count++;
        } else {
            zc_send_arp_announce(zc);            

            zc->zc_state = ZEROCONF_RUNNING;
            zc->zc_count = 0;

            zc->zc_timeout = now;
        }

        break;
    };
}

void zc_init(struct zc_config* zc, const unsigned char* hw_addr)
{
    zc->zc_fd = -1;
    zc->zc_state = ZEROCONF_DISABLED;
    zc->zc_timeout = 0x7ffffff;
    zc->zc_count = 0;

    memset(zc->zc_addr, 0, 4);
    
    memcpy(zc->zc_hw_addr, hw_addr, ETH_ALEN);
}

void zc_reset(struct zc_config* zc)
{
    if (zc->zc_state != ZEROCONF_DISABLED) {
        close(zc->zc_fd);
        zc->zc_fd = -1;
        zc->zc_state = ZEROCONF_DISABLED;
        zc->zc_timeout = 0x7fffffff;
        zc->zc_count = 0;
        memset(zc->zc_addr, 0, 4);
        
        if (zc->zc_state == ZEROCONF_ANNOUNCING ||
            zc->zc_state == ZEROCONF_RUNNING)
            run_script(NULL, "deconfig");
    }
}

void zc_start_probing(struct zc_config* zc, time_t now)
{
    zc->zc_addr[0] = 169;
    zc->zc_addr[1] = 254;

    int fd = open("/dev/urandom", O_RDONLY);

    do {
        read(fd, &zc->zc_addr[2], 1);
    } while (zc->zc_addr[2] < 1 || zc->zc_addr[2] > 254);
    
    read(fd, &zc->zc_addr[3], 1);

    close(fd);

    if (zc->zc_state == ZEROCONF_ANNOUNCING || 
        zc->zc_state == ZEROCONF_RUNNING) {
        run_script(NULL, "deconfig");
    }

    LOG(LOG_DEBUG, "Trying zc probe of %u.%u.%u.%u",
        zc->zc_addr[0], zc->zc_addr[1], zc->zc_addr[2], zc->zc_addr[3]);
    zc->zc_state = ZEROCONF_PROBING;
    zc->zc_timeout = now;
    zc->zc_count = 0;
}

void zc_send_arp_probe(struct zc_config* zc)
{
    unsigned char buf[2048];
    struct ether_header* packethdr = (struct ether_header*)buf;
    struct ether_arp* ap = (struct ether_arp*)(packethdr + 1);

    memcpy(&(packethdr->ether_shost), zc->zc_hw_addr, ETH_ALEN);
    memset(&(packethdr->ether_dhost), 0xFF, ETH_ALEN);
    packethdr->ether_type = htons(ETHERTYPE_ARP);    

    ap->arp_hrd = htons(ARPHRD_ETHER);
    ap->arp_pro = htons(ETHERTYPE_IP);
    ap->arp_hln = ETH_ALEN;
    ap->arp_pln = 4;
    ap->arp_op = htons(ARPOP_REQUEST);

    memcpy(ap->arp_sha, zc->zc_hw_addr, ETH_ALEN);
    memset(ap->arp_tha, 0, ETH_ALEN);
    memset(ap->arp_spa, 0, 4);
    memcpy(ap->arp_tpa, zc->zc_addr, 4);

    LOG(LOG_DEBUG, "Sending zc arp probe...");
    send(zc->zc_fd, buf, sizeof(*packethdr) + sizeof(*ap), 0);
}

void zc_send_arp_announce(struct zc_config* zc)
{
    unsigned char buf[2048];
    struct ether_header* packethdr = (struct ether_header*)buf;
    struct ether_arp* ap = (struct ether_arp*)(packethdr + 1);

    memcpy(&(packethdr->ether_shost), zc->zc_hw_addr, ETH_ALEN);
    memset(&(packethdr->ether_dhost), 0xFF, ETH_ALEN);
    packethdr->ether_type = htons(ETHERTYPE_ARP);    

    ap->arp_hrd = htons(ARPHRD_ETHER);
    ap->arp_pro = htons(ETHERTYPE_IP);
    ap->arp_hln = ETH_ALEN;
    ap->arp_pln = 4;
    ap->arp_op = htons(ARPOP_REQUEST);

    memcpy(ap->arp_sha, zc->zc_hw_addr, ETH_ALEN);
    memset(ap->arp_tha, 0, ETH_ALEN);
    memcpy(ap->arp_spa, zc->zc_addr, 4);
    memcpy(ap->arp_tpa, zc->zc_addr, 4);

    LOG(LOG_DEBUG, "Sending zc announce...");
    send(zc->zc_fd, buf, sizeof(*packethdr) + sizeof(*ap), 0);
}
