/*
 * Copyright (c) 2011-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:	GPL-2.0
 */

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <assert.h>
#include "syslog_client.h"

#ifdef DBG_SYSLOG

#define SYSLOG_DEF_LOCATION     "/opt/log/"
#define SYSLOG_DEF_FILESIZE     (32 * 1024)
#define SYSLOG_MIN_FILESIZE     (2 * 1024)
#define SYSLOG_TRUNC_FACTOR     (2)

#ifdef ALLOW_KERNEL_LOG_FORMAT
#undef SYSLOG_KERNEL_FORMAT
#endif

#ifndef SYSLOG_KERNEL_FORMAT
#undef SYSLOG_OVERRIDE_WITH_MICROSECONDS
#endif

#ifdef SYSLOG_KERNEL_FORMAT
#include "sonos_time.h"
#endif

#define DBG_SYSLOG_PRINTF(...)

static void write_log_msg(int fd, const char* msg, int len);

static char syslog_ident[64];
static char syslog_location[128];

static int syslog_fd = -1;
static int syslog_mask = LOG_UPTO(LOG_INFO);
static int syslog_max_filesize = SYSLOG_DEF_FILESIZE;
static int syslog_option = 0;
static pid_t syslog_pid;

void __syslog_chk(int level, int flag, const char *format, ...)
{
    (void) flag;
    va_list ap;

    va_start(ap, format);
    vsyslog(level, format, ap);
    va_end(ap);
}

void openlog(const char* ident, int option, int facility)
{
    (void)facility;

    if (ident == NULL) {
        return;
    }

    syslog_option = option;
    strncpy(syslog_ident, ident, sizeof(syslog_ident));
    syslog_ident[sizeof(syslog_ident) - 1] = '\0';
    syslog_pid = getpid();

    closelog();

    if ((syslog_option & LOG_SONOS_STDOUT) == 0) {
        snprintf(syslog_location, sizeof(syslog_location), "%s%s.log",
                 SYSLOG_DEF_LOCATION, syslog_ident);

        syslog_fd = open(syslog_location, O_CREAT | O_RDWR, DEFFILEMODE);
        if (syslog_fd >= 0) {
            lseek(syslog_fd, 0, SEEK_END);
        }
    }

    return;
}

void closelog()
{
    if (syslog_fd != -1) {
        close(syslog_fd);
        syslog_fd = -1;
    }
}

int setlogmask(int mask)
{
    int prev = syslog_mask;

    syslog_mask = mask;

    return prev;
}

void setmaxlogsize(int size)
{
    if (size < SYSLOG_MIN_FILESIZE * 2) {
        size = SYSLOG_MIN_FILESIZE * 2;
    }
    syslog_max_filesize = size;
}

void syslog(int level, const char* format, ...)
{
    va_list ap;

    va_start(ap, format);
    vsyslog(level, format, ap);
    va_end(ap);
}

void vsyslog(int level, const char* format, va_list ap)
{
    char tmp[1024];
    int len;
    int pfx_len;

#ifdef SYSLOG_KERNEL_FORMAT
    struct timeval timeval_now;
#else
    unsigned char pri;
    struct tm *localtm;
    char timestamp[32];
#ifdef SYSLOG_OVERRIDE_WITH_MICROSECONDS
    struct timeval timeval_now;
    char usec_buf[14];
    usec_buf[0] = 0;
#else
    time_t now;
#endif
#endif

    if (syslog_fd == -1 && (syslog_option & LOG_SONOS_STDOUT) == 0) {
        return;
    }

    if ((syslog_mask & LOG_MASK(LOG_PRI(level))) == 0) {
        return;
    }

#ifdef SYSLOG_KERNEL_FORMAT
    sonosGetBootTimeval(&timeval_now);
    pfx_len = snprintf(tmp, sizeof(tmp), "[%5ld.%06ld] %s_%d:",
                       timeval_now.tv_sec, timeval_now.tv_usec, syslog_ident, syslog_pid);
#else
#ifdef SYSLOG_OVERRIDE_WITH_MICROSECONDS
    gettimeofday(&timeval_now, NULL);
    localtm = localtime(&timeval_now.tv_sec);
    snprintf(usec_buf, sizeof(usec_buf), ".%06ld", timeval_now.tv_usec);
#else
    now = time(NULL);
    localtm = localtime(&now);
#endif

    if (!localtm) {
        return;
    }
    len = strftime(timestamp, sizeof(timestamp), "%b %d %H:%M:%S", localtm);
    if (len == 0) {
        return;
    }
    if (timestamp[4] == '0') {
        timestamp[4] = ' ';
    }

    pri = LOG_USER | LOG_PRI(level);

#ifdef SYSLOG_OVERRIDE_WITH_MICROSECONDS
    pfx_len = snprintf(tmp, sizeof(tmp), "<%d>%s%s none %s_%d:", pri, timestamp, usec_buf,
                       syslog_ident, syslog_pid);
#else
    pfx_len = snprintf(tmp, sizeof(tmp), "<%d>%s none %s_%d:", pri, timestamp,
                       syslog_ident, syslog_pid);
#endif
#endif

    len = vsnprintf(tmp + pfx_len, sizeof(tmp) - pfx_len, format, ap);

    if (len < 0) {
        return;
    }
    len += pfx_len;

    if (len >= (int)sizeof(tmp)) {
        len = sizeof(tmp) - 1;
    }

    if (len > 0) {
        if (tmp[len - 1] != '\n') {
            tmp[len++] = '\n';
        }

        if (syslog_option & LOG_SONOS_STDOUT) {
            write(STDOUT_FILENO, tmp, len);
        } else {
            write_log_msg(syslog_fd, tmp, len);
        }
    }
}


static void write_log_msg(int fd, const char* msg, int len)
{

    off_t pos = lseek(fd, 0, SEEK_CUR);

    if (pos == (off_t)-1) {
        return;
    }

    if (pos >= syslog_max_filesize) {
        int i;

        off_t truncated_size = syslog_max_filesize / SYSLOG_TRUNC_FACTOR;

        if (truncated_size < SYSLOG_MIN_FILESIZE) {
            truncated_size = SYSLOG_MIN_FILESIZE;
        }

        char* buf = malloc(truncated_size);
        if (!buf) {
            truncated_size = 0;
            goto cleanup;
        }

        if (lseek(fd, -truncated_size, SEEK_END) == (off_t)-1) {
            truncated_size = 0;
            goto cleanup;
        }
        if (read(fd, buf, truncated_size) != truncated_size) {
            truncated_size = 0;
            goto cleanup;
        }

        for (i = 0; i < truncated_size; i++) {
            if (buf[i] == '\n') {
                i++;
                break;
            }
        }

        truncated_size -= i;

        if (truncated_size > 0) {
            if (lseek(fd, 0, SEEK_SET) == (off_t)-1) {
                truncated_size = 0;
                goto cleanup;
            }
            if (write(fd, &buf[i], truncated_size) != truncated_size) {
                truncated_size = 0;
                goto cleanup;
            }
        }
cleanup:
        free(buf);
        if (ftruncate(fd, truncated_size) != 0) {
            return;
        }
    }

    write(fd, msg, len);

    fsync(fd);
}


#endif
