/*
 * socket.c -- DHCP server client/server socket creation
 *
 * udhcp client/server
 * Copyright (C) 1999 Matthew Ramsay <matthewr@moreton.com.au>
 *			Chris Trew <ctrew@moreton.com.au>
 *
 * Rewrite by Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <unistd.h>
#include <string.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <errno.h>
#include <features.h>
#if __GLIBC__ >=2 && __GLIBC_MINOR >= 1
#include <netpacket/packet.h>
#include <net/ethernet.h>
#else
#include <asm/types.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>
#endif
#if defined(__SONOS_LINUX__)
#include <linux/if_bridge.h>
#include <linux/sockios.h>

/* Interface used to see if the driver is connected */
#define ATH_CTL_STA_GET_STATE        302

#define STA_BEACON_IES_LEN           1000
#define STA_SSID_LEN                 32
#define STA_ADDR_LEN                 6

struct sta_state_info {
    unsigned long flags;
    int  state;
    int  freq;
    int  ap_freq;

    int  ssid_len;
    char ssid[STA_SSID_LEN+1];
    char bssid[STA_ADDR_LEN];

    unsigned short beacon_int;
    unsigned short beacon_capa;
    unsigned int   beacon_ies_len;
    char           beacon_ies[STA_BEACON_IES_LEN];
};
#define STA_STATE_FLAG_CONNECTED (1 << 1)

// Copy KLUDGE from enet_port_stats.cxx: 
// No good place to pull these typedefs from, needed by ethtool.h.
typedef unsigned long long u64;
typedef unsigned int       u32;
typedef unsigned short     u16;
typedef unsigned char       u8;

#include <linux/ethtool.h>
#endif

#include "debug.h"

int read_interface(const char *interface, int *ifindex, u_int32_t *addr, unsigned char *arp)
{
	int fd;
	struct ifreq ifr;
	struct sockaddr_in *our_ip;

	memset(&ifr, 0, sizeof(struct ifreq));
	if((fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW)) >= 0) {
		ifr.ifr_addr.sa_family = AF_INET;
		strcpy(ifr.ifr_name, interface);

		if (addr) { 
			if (ioctl(fd, SIOCGIFADDR, &ifr) == 0) {
				our_ip = (struct sockaddr_in *) &ifr.ifr_addr;
				*addr = our_ip->sin_addr.s_addr;
				DEBUG(LOG_INFO, "%s (our ip) = %s", ifr.ifr_name, inet_ntoa(our_ip->sin_addr));
			} else {
				LOG(LOG_ERR, "SIOCGIFADDR failed, is the interface up and configured?: %s", 
						strerror(errno));
				close(fd);
				return -1;
			}
		}
		
		if (ioctl(fd, SIOCGIFINDEX, &ifr) == 0) {
			DEBUG(LOG_INFO, "adapter index %d", ifr.ifr_ifindex);
			*ifindex = ifr.ifr_ifindex;
		} else {
			LOG(LOG_ERR, "SIOCGIFINDEX failed!: %s", strerror(errno));
			close(fd);
			return -1;
		}
		if (ioctl(fd, SIOCGIFHWADDR, &ifr) == 0) {
			memcpy(arp, ifr.ifr_hwaddr.sa_data, 6);
			LOG(LOG_INFO, "Interface: %s Address: %02x:%02x:%02x:%02x:%02x:%02x",
                            interface, arp[0], arp[1], arp[2], arp[3], arp[4], arp[5]);
		} else {
			LOG(LOG_ERR, "SIOCGIFHWADDR failed!: %s", strerror(errno));
			close(fd);
			return -1;
		}
	} else {
		LOG(LOG_ERR, "socket failed!: %s", strerror(errno));
		return -1;
	}
	close(fd);
	return 0;
}


int listen_socket(unsigned int ip, int port, const char *inf)
{
	struct ifreq interface;
	int fd;
	struct sockaddr_in addr;
	int n = 1;

	DEBUG(LOG_INFO, "Opening listen socket on 0x%08x:%d %s\n", ip, port, inf);
	if ((fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
		DEBUG(LOG_ERR, "socket call failed: %s", strerror(errno));
		return -1;
	}
	
	memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);
	addr.sin_addr.s_addr = ip;

	if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *) &n, sizeof(n)) == -1) {
		close(fd);
		return -1;
	}
	if (setsockopt(fd, SOL_SOCKET, SO_BROADCAST, (char *) &n, sizeof(n)) == -1) {
		close(fd);
		return -1;
	}

	strncpy(interface.ifr_ifrn.ifrn_name, inf, IFNAMSIZ-1);
	interface.ifr_ifrn.ifrn_name[IFNAMSIZ-1] = '\0';
	if (setsockopt(fd, SOL_SOCKET, SO_BINDTODEVICE,(char *)&interface, sizeof(interface)) < 0) {
		close(fd);
		return -1;
	}

	if (bind(fd, (struct sockaddr *)&addr, sizeof(struct sockaddr)) == -1) {
		close(fd);
		return -1;
	}
	
	return fd;
}


int raw_socket(int ifindex)
{
	int fd;
	struct sockaddr_ll sock;

	DEBUG(LOG_INFO, "Opening raw socket on ifindex %d\n", ifindex);
	/*
	 * For ZPs, open raw socket, passing up link level header so
	 * source address (SA) of BOOTP response can be logged.
	 * Otherwise, pass up cooked packets with link level header
	 * removed. (SWPBL-74226)
	 */
#ifdef SONOS_ARCH_ATTR_IS_ZONE_PLAYER
	if ((fd = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_IP))) < 0)
#else
	if ((fd = socket(PF_PACKET, SOCK_DGRAM, htons(ETH_P_IP))) < 0)
#endif
	{
		DEBUG(LOG_ERR, "socket call failed: %s", strerror(errno));
		return -1;
	}
	
	sock.sll_family = AF_PACKET;
	sock.sll_protocol = htons(ETH_P_IP);
	sock.sll_ifindex = ifindex;
	if (bind(fd, (struct sockaddr *) &sock, sizeof(sock)) < 0) {
		DEBUG(LOG_ERR, "bind call failed: %s", strerror(errno));
		close(fd);
		return -1;
	}

	return fd;
}


#if defined(__SONOS_LINUX__)
static int do_ioctl(const char *interface, unsigned long *args)
{
    int fd;
    int rc = 0;

    if ((fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW)) < 0) {
        return -1;
    }

    struct ifreq ifr;
    memset(&ifr, 0, sizeof(struct ifreq));
    strcpy(ifr.ifr_name, interface);
    ((unsigned long *)(&ifr.ifr_data))[0] = (unsigned long)args;

    if (ioctl(fd, SIOCDEVPRIVATE, &ifr) != 0) {
        DEBUG(LOG_ERR, "ioctl failed for %s", interface);
        rc = -2;
    }
    close(fd);

    return rc;
}

int bridge_ready(const char *interface)
{
    char valid = 0;

    unsigned long args[4];
    args[0] = (unsigned long)BRCTL_GET_ANY_FORWARDING;
    args[1] = (unsigned long)&valid;

    /* Treat failure as success so we don't wait forever */
    if (do_ioctl(interface, args)) {
        valid = 1;
    }

    return valid;
}

#if defined(SONOS_ARCH_ATTR_USES_DRIVER_NL80211)
int wifi_ready(const char *interface)
{
    int ready = 0;
    int fd;
    struct ifreq ifr;

    fd = socket(AF_INET, SOCK_DGRAM, 0);
    if (fd < 0) {
        /* Treat failure as success so we don't wait forever */
        return 1;
    }

    memset(&ifr, 0, sizeof(ifr));
    (void) strncpy(ifr.ifr_name, interface, sizeof(ifr.ifr_name) - 1);

    /* Treat failure as success so we don't wait forever */
    if (ioctl(fd, SIOCGIFFLAGS, &ifr) || (ifr.ifr_flags & IFF_RUNNING)) {
        ready = 1;
    }
    (void) close(fd);
    return ready;
}
#else
int wifi_ready(const char *interface)
{
    int ready = 0;

    struct sta_state_info info;
    memset(&info, 0, sizeof(info));

    unsigned long args[4];
    args[0] = (unsigned long)ATH_CTL_STA_GET_STATE;
    args[1] = (unsigned long)&info;
    args[2] = (unsigned long)sizeof(info);

    /* Treat failure as success so we don't wait forever */
    if (do_ioctl(interface, args) || (info.flags & STA_STATE_FLAG_CONNECTED)) {
        ready = 1;
    }

    return ready;
}
#endif

/* shamelessly borrowed from netstartd */
static int get_carrier(int device)
{
    int has_carrier = 0;
    int fd;

    if ((fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW)) >= 0) {
        struct ifreq ifr;
        struct ethtool_cmd cmd;

        memset(&cmd, 0, sizeof(cmd));
        cmd.cmd = ETHTOOL_GSET;

        memset(&ifr, 0, sizeof(struct ifreq));
        ifr.ifr_data = (char *)&cmd;
        sprintf((char *)(&ifr.ifr_name), "eth%d", device);

        /* driver sets speed to 0 if link is down... */
        if (ioctl(fd, SIOCETHTOOL, &ifr) == 0) {
            has_carrier = (cmd.speed != 0);
        }

        close(fd);
    }

    return has_carrier;
}

int ethernet_available()
{
    unsigned int i;

    // we only support 8 ports
    for (i = 0; i < 8; i++) {
        if (get_carrier(i) > 0) {
            return 1;
        }
    }

    return 0;
}
#endif

