/*
 * Copyright (c) 2003-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */

#include "nlhelp.h"

#include <asm/types.h>
#include <netinet/if_ether.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <linux/if.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <linux/sockios.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "sonos_time.h"
#include "zeroconf.h"
#include "debug.h"

void rtnl_close(struct rtnl_handle *rth)
{
    close(rth->fd);
}

int rtnl_open(struct rtnl_handle *rth, unsigned subscriptions)
{
    socklen_t addr_len;

    memset(rth, 0, sizeof(*rth));

    rth->fd = socket(PF_NETLINK, SOCK_RAW, NETLINK_ROUTE);
    if (rth->fd < 0)
        return -1;

    memset(&rth->local, 0, sizeof(rth->local));
    rth->local.nl_family = AF_NETLINK;
    rth->local.nl_groups = subscriptions;

    if (bind(rth->fd, (struct sockaddr*)&rth->local, sizeof(rth->local)) < 0) {
        close(rth->fd);
        return -1;
    }

    addr_len = sizeof(rth->local);
    if (getsockname(rth->fd, (struct sockaddr*)&rth->local, &addr_len) < 0) {
        close(rth->fd);
        return -1;
    }

    if (addr_len != sizeof(rth->local)) {
        close(rth->fd);
        return -1;
    }

    if (rth->local.nl_family != AF_NETLINK) {
        close(rth->fd);
        return -1;
    }

    rth->seq = sonosGetBootTime();

    return 0;
}

#if defined(__SONOS_LINUX__)

static int __get_arp_ip(const char* if_name,
                        in_addr_t* out_addr)
{
    int ret = 0;
    int fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (fd != -1) {
        char buffer[16384];
        struct ifconf ifConf;
        ifConf.ifc_len = sizeof(buffer);
        ifConf.ifc_ifcu.ifcu_buf = (caddr_t)buffer;
        if (ioctl(fd, SIOCGIFCONF, &ifConf) >= 0) {
            int i;
            for (i = 0; i < ifConf.ifc_len; ) {
                struct ifreq *pifReq =
                    (struct ifreq *)((caddr_t)ifConf.ifc_req + i);
                i += sizeof(*pifReq);

                if ((pifReq->ifr_addr.sa_family == AF_INET) &&
                    (0 == strcmp(if_name, pifReq->ifr_name))) {
                    struct sockaddr_in addr;
                    memcpy(&addr, &pifReq->ifr_addr, sizeof(pifReq->ifr_addr));
                    *out_addr = addr.sin_addr.s_addr;
                    ret = 1;
                    break;
                }
            }
        }

        close(fd);
    }

    return ret;
}

int rtnl_send_gratuitous_arp(const char* if_name,
                             const unsigned char* if_hw_addr)
{
    in_addr_t addr;

    if (__get_arp_ip(if_name, &addr)) {
        int fd = zc_arp_socket(if_name);
        if (fd != -1) {
            unsigned char buf[2048];
            struct ether_header* packethdr = (struct ether_header*)buf;
            struct ether_arp* ap = (struct ether_arp*)(packethdr + 1);

            memcpy(&(packethdr->ether_shost), if_hw_addr, ETH_ALEN);
            memset(&(packethdr->ether_dhost), 0xFF, ETH_ALEN);
            packethdr->ether_type = htons(ETHERTYPE_ARP);    

            ap->arp_hrd = htons(ARPHRD_ETHER);
            ap->arp_pro = htons(ETHERTYPE_IP);
            ap->arp_hln = ETH_ALEN;
            ap->arp_pln = 4;
            ap->arp_op = htons(ARPOP_REQUEST);

            memcpy(ap->arp_sha, if_hw_addr, ETH_ALEN);
            memset(ap->arp_tha, 0, ETH_ALEN);
            memcpy(ap->arp_spa, &addr, 4);
            memcpy(ap->arp_tpa, &addr, 4);

            send(fd, buf, sizeof(*packethdr) + sizeof(*ap), 0);

            close(fd);

            return 1;
        }
    }

    return 0;
}

static int __check_dev_name_attr(struct nlmsghdr* hdr,
                                 const char* wifi_if_name)
{
    void* pv = NLMSG_DATA(hdr);

    char rgchDev[IFNAMSIZ + 1];

    struct rtattr* attr = (struct rtattr *)pv;
    int attrlen = NLMSG_PAYLOAD(hdr, 0);

    for (; RTA_OK(attr, attrlen); attr = RTA_NEXT(attr, attrlen)) {
        if (attr->rta_type == RWA_DEV_NAME) {
            if (RTA_PAYLOAD(attr) <= IFNAMSIZ) {
                memset(rgchDev, 0, sizeof(rgchDev));
                memcpy(rgchDev, RTA_DATA(attr), RTA_PAYLOAD(attr));
                if (0 == strcmp(rgchDev, wifi_if_name))
                    return 1;
            }
        }
    }

    return 0;
}

static void __arp_if_assoc_changed(struct nlmsghdr* hdr,
                                   const char* wifi_if_name,
                                   const char* if_name,
                                   const unsigned char* if_hw_addr)
{
    void* pv = NLMSG_DATA(hdr);

    int our_dev = 0;

    unsigned char assoc_mac[ETH_ALEN] = 
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
    static const unsigned char empty_mac[ETH_ALEN] = 
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    char rgchDev[IFNAMSIZ + 1];

    struct rtattr* attr = (struct rtattr *)pv;
    int attrlen = NLMSG_PAYLOAD(hdr, 0);

    for (; RTA_OK(attr, attrlen); attr = RTA_NEXT(attr, attrlen)) {
        if (attr->rta_type == RWA_DEV_NAME) {
            if (RTA_PAYLOAD(attr) <= IFNAMSIZ) {
                memset(rgchDev, 0, sizeof(rgchDev));
                memcpy(rgchDev, RTA_DATA(attr), RTA_PAYLOAD(attr));
                if (0 == strcmp(rgchDev, wifi_if_name))
                    our_dev = 1;
                else
                    break;
            }
        } else if (attr->rta_type == RWA_ASSOC_MAC) {
            if (RTA_PAYLOAD(attr) == ETH_ALEN) {
                memcpy(assoc_mac, RTA_DATA(attr), ETH_ALEN);
            }
        }
    }

    if (our_dev && 0 != memcmp(assoc_mac, empty_mac, ETH_ALEN))
        rtnl_send_gratuitous_arp(if_name, if_hw_addr);
}

#endif

int rtnl_msg_triggers_dhcp_renew(struct rtnl_handle *rth,
                                 const char* wifi_if_name,
                                 const char* if_name,
                                 const unsigned char* if_hw_addr)
{
#if defined(__SONOS_LINUX__)
    struct sockaddr_nl sanl;
    socklen_t sanllen = sizeof(sanl);
    
    int amt;
    char buf[2048];
    amt = recvfrom(rth->fd, buf, sizeof(buf),
                   0, (struct sockaddr*)&sanl, &sanllen);
    if (amt < 0) {
        return 0;
    }

    if (sanl.nl_family != AF_NETLINK) {
        return 0;
    }

    struct nlmsghdr* hdr = (struct nlmsghdr*)buf;
    size_t cb = (size_t)amt;

    for (; NLMSG_OK(hdr, cb); hdr = NLMSG_NEXT(hdr, cb)) {
        if (sanl.nl_groups == RTMGRP_Rincon) {
            if (hdr->nlmsg_type == RWM_HHID) {
                if (__check_dev_name_attr(hdr, wifi_if_name)) {
                    LOG(LOG_INFO, "Event: HHID changed - renew DHCP");
                    return 1;
                }
            } else if (hdr->nlmsg_type == RWM_MII) {
                if((sonosGetBootTime() - rth->seq) < 3) {
                    return 0;
                }
                LOG(LOG_INFO, "Event: Link changed - renew DHCP");
                return 1;
            } else if (hdr->nlmsg_type == RWM_RENEW_DHCP) {
                LOG(LOG_INFO, "Event: Renew DHCP");
                return 1;
            } else if (hdr->nlmsg_type == RWM_ASSOC) {
                __arp_if_assoc_changed(hdr, wifi_if_name, if_name, if_hw_addr);
                LOG(LOG_INFO, "Event: ASSOC");
            }
        } else {
            break;
        }
    }
#else
    (void)rth;
    (void)wifi_if_name;
    (void)if_name;
    (void)if_hw_addr;
#endif

    return 0;
}
