/*
 * Copyright (c) 2015-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:	GPL-2.0
 *
 * linux/fs/proc/sonos_lock.c
 *
 * /proc entry support to allow control over unlock authorization
 * functionality on secured Sonos products.
 */

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/module.h>
#include <asm/uaccess.h>
#include "mdp.h"
#include <linux/regmap.h>
#include <linux/mfd/syscon.h>
#include "sonos_rollback.h"
#define SONOS_LOCK_C
#include "sonos_lock.h"
#ifdef CONFIG_SONOS_SECBOOT
#include <sonos/firmware_allowlist.h>
#include <linux/sonos_sec_lock.h>
#endif


static int proc_show_allow(struct seq_file *m, void *v);
static ssize_t proc_write_allow(struct file *file, const char __user * buf,
				size_t length, loff_t *offset);

#define IMPLEMENT_ALLOW_PROCFILE(s)					\
									\
static bool allow_##s = true;						\
									\
bool sonos_allow_##s(void)						\
{									\
	return allow_##s;						\
}									\
EXPORT_SYMBOL(sonos_allow_##s);						\
									\
static int proc_open_allow_##s(struct inode *inode, struct file *file)	\
{									\
	return single_open(file, proc_show_allow, &allow_##s);		\
}									\
									\
static const struct file_operations allow_##s##_proc_fops = {		\
	.open		= proc_open_allow_##s,				\
	.read		= seq_read,					\
	.write		= proc_write_allow,				\
	.llseek		= seq_lseek,					\
	.release	= single_release,				\
};

IMPLEMENT_ALLOW_PROCFILE(insmod)
IMPLEMENT_ALLOW_PROCFILE(mount_dev)
IMPLEMENT_ALLOW_PROCFILE(mount_exec)

static int proc_show_allow(struct seq_file *m, void *v)
{
	bool *allow_flag = m->private;
	seq_printf(m, "%d\n", *allow_flag ? 1 : 0);
	return 0;
}

struct procfile_devunlock_map_entry {
	const struct file_operations *fops;
	const char *file_name;
	bool *allow_flag;
	u32 devunlock_perm;
};

#define MAP_ENTRY(s, perm) { &allow_##s##_proc_fops, "allow_" #s, &allow_##s, perm }
static const struct procfile_devunlock_map_entry devunlock_map[] =
{
	MAP_ENTRY(insmod, MDP_AUTH_FLAG_INSMOD_CTRL),
	MAP_ENTRY(mount_dev, MDP_AUTH_FLAG_NODEV_CTRL),
	MAP_ENTRY(mount_exec, MDP_AUTH_FLAG_EXEC_ENABLE),
};

static ssize_t proc_write_allow(struct file *file,
				const char __user * buf,
				size_t length,
				loff_t *offset)
{
	char c;
	ssize_t result = -EINVAL;

	if (length == 0) {
		return 0;
	}

	if (copy_from_user(&c, buf, 1)) {
		return -EFAULT;
	}

	if (c == '0') {
		const struct procfile_devunlock_map_entry *map_entry;
		size_t i;
		for (i = 0; i < ARRAY_SIZE(devunlock_map); i++) {
			map_entry = &devunlock_map[i];
			if (PDE_DATA(file->f_inode) == map_entry->fops) {
				break;
			}
		}

		BUG_ON(i == ARRAY_SIZE(devunlock_map));

		if (is_mdp_authorized(map_entry->devunlock_perm)) {
			printk(KERN_INFO "%s not set to 0 due to DevUnlock\n",
			       map_entry->file_name);
		}
		else {
			printk(KERN_INFO "%s set to 0\n",
			       map_entry->file_name);
			*(map_entry->allow_flag) = false;
		}
		result = length;
	}

	return result;
}


static int proc_show_enabled(struct seq_file *m, void *v);

#define IMPLEMENT_ENABLED_PROCFILE(s, perm)				\
									\
static const u32 devunlock_perm_enabled_##s = perm;			\
									\
static int proc_open_enabled_##s(struct inode *inode, struct file *file)\
{									\
	return single_open(file, proc_show_enabled,			\
			   (void *)&devunlock_perm_enabled_##s);	\
}									\
									\
static const struct file_operations s##_enable_proc_fops = {		\
	.open		= proc_open_enabled_##s,			\
	.read		= seq_read,					\
	.llseek		= seq_lseek,					\
	.release	= single_release,				\
};

IMPLEMENT_ENABLED_PROCFILE(console, MDP_AUTH_FLAG_CONSOLE_ENABLE)
IMPLEMENT_ENABLED_PROCFILE(telnet, MDP_AUTH_FLAG_TELNET_ENABLE)
IMPLEMENT_ENABLED_PROCFILE(exec, MDP_AUTH_FLAG_EXEC_ENABLE)
IMPLEMENT_ENABLED_PROCFILE(kernel_debug, MDP_AUTH_FLAG_KERNEL_DEBUG_ENABLE)

static int proc_show_enabled(struct seq_file *m, void *v)
{
	const u32 *devunlock_perm = m->private;
	seq_printf(m, "%d\n", is_mdp_authorized(*devunlock_perm));
	return 0;
}

#if defined(CONFIG_UBIFS_FS)

static int current_ubifs_type = 0;
void sonos_set_proc_crypt(int type)
{
	current_ubifs_type = type;
}
EXPORT_SYMBOL(sonos_set_proc_crypt);

static int ubifs_crypt_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "%d\n", current_ubifs_type);
	return 0;
}

static int ubifs_crypt_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ubifs_crypt_proc_show, NULL);
}

static const struct file_operations ubifs_crypt_proc_fops = {
	.open		= ubifs_crypt_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};
#endif

extern void wdt2_set_service_state(int);
extern int  wdt2_get_service_state(void);


void __attribute__((weak)) wdt2_set_service_state(int new_state)
{
}

int __attribute__((weak)) wdt2_get_service_state(void)
{
	return KERNEL_DEFAULT;
}

static int wdog_state_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "%d\n", wdt2_get_service_state());
	return 0;
}

static int wdog_state_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, wdog_state_proc_show, NULL);
}

static ssize_t wdog_state_write_data(struct file *file, const char __user * buf, size_t length, loff_t * offset)
{
	char	buffer[64];
	__s32	new_state;

        memset(buffer, 0, sizeof(buffer));
        if (length > sizeof(buffer) - 1)
                length = sizeof(buffer) - 1;
        if (copy_from_user(buffer, buf, length))
                return -EFAULT;

	sscanf(buffer, "%d", &new_state);

	if ( new_state < 0 || new_state > 2 ) {
		printk(KERN_INFO "illegal state - must be 0, 1 or 2\n");
	} else {
		wdt2_set_service_state(new_state);
	}
	return length;
}

static const struct file_operations wdog_state_proc_fops = {
	.open		= wdog_state_proc_open,
	.read		= seq_read,
	.write		= wdog_state_write_data,
	.llseek		= seq_lseek,
	.release	= single_release,
};



static int allowlist_flags_ctrl_proc_show(struct seq_file *m, void *v)
{
	uint32_t flags = 0;
#ifdef CONFIG_SONOS_SECBOOT
	if (__be32_to_cpu(SONOS_FIRMWARE_ALLOWLIST.header.numEntries) > 0) {
		flags |= 0x2;
	}
#endif
	seq_printf(m, "0x%x\n", flags);
	return 0;
}

static int allowlist_flags_ctrl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, allowlist_flags_ctrl_proc_show, NULL);
}

static const struct file_operations allowlist_flags_proc_fops = {
	.open		= allowlist_flags_ctrl_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};


static int __init proc_unlock_auth_init(void)
{
	proc_create_data("sonos-lock/allow_insmod", 0, NULL,
			 &allow_insmod_proc_fops,
			 (void *)&allow_insmod_proc_fops);
	proc_create_data("sonos-lock/allow_mount_dev", 0, NULL,
			 &allow_mount_dev_proc_fops,
			 (void *)&allow_mount_dev_proc_fops);
	proc_create_data("sonos-lock/allow_mount_exec", 0, NULL,
			 &allow_mount_exec_proc_fops,
			 (void *)&allow_mount_exec_proc_fops);
	proc_create("sonos-lock/console_enable", 0, NULL,
		    &console_enable_proc_fops);
	proc_create("sonos-lock/telnet_enable", 0, NULL,
		    &telnet_enable_proc_fops);
	proc_create("sonos-lock/exec_enable", 0, NULL,
		    &exec_enable_proc_fops);
	proc_create("sonos-lock/kernel_debug_enable", 0, NULL,
		    &kernel_debug_enable_proc_fops);
#if defined(CONFIG_UBIFS_FS)
	proc_create("sonos-lock/ubifs_crypt", 0, NULL, &ubifs_crypt_proc_fops);
#endif
	proc_create("sonos-lock/watchdog_service_state", 0, NULL, &wdog_state_proc_fops);
	proc_create("sonos-lock/allowlist_flags", 0, NULL, &allowlist_flags_proc_fops);
	return 0;
}
module_init(proc_unlock_auth_init);
