


/* Standard Linux headers */
#include <linux/types.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/amlogic/media/amvecm/amvecm.h>
#include "set_hdr2_v0.h"
#include "arch/vpp_hdr_regs.h"
#include "arch/vpp_regs.h"

//#define HDR2_MODULE
//#define HDR2_PRINT

#ifndef HDR2_MODULE

// sdr to hdr table  12bit
int cgain_lut0[65] = {
	0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400,
	0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400,
	0x4c0, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400,
	0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x40e,
	0x429, 0x444, 0x45f, 0x479, 0x492, 0x4ab, 0x4c3, 0x4db, 0x4f2,
	0x509, 0x520, 0x536, 0x54c, 0x561, 0x576, 0x58b, 0x59f, 0x5b3,
	0x5c0, 0x5d0, 0x5f2, 0x609, 0x620, 0x636, 0x64c, 0x661, 0x676,
	0x68b, 0x69f
};

// hdr10 to gamma lut 12bit (hdr to sdr)
int cgain_lut1[65] = {
	0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400,
	0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400,
	0x4c0, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400,
	0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x400, 0x40e,
	0x429, 0x444, 0x45f, 0x479, 0x492, 0x4ab, 0x4c3, 0x4db, 0x4f2,
	0x509, 0x520, 0x536, 0x54c, 0x561, 0x576, 0x58b, 0x59f, 0x5b3,
	0x5c0, 0x5d0, 0x5f2, 0x609, 0x620, 0x636, 0x64c, 0x661, 0x676,
	0x68b, 0x69f
};

// sdr to hdr 10bit (gamma to peak)
int cgain_lut2[65] = {
	0xc00, 0xc00, 0xc00, 0xc00, 0xc00, 0xc00, 0xc00, 0xc00, 0xc00,
	0xc00, 0xc00, 0xc0e, 0xc79, 0xcdb, 0xd36, 0xd8b, 0xdda, 0xe25,
	0xe6b, 0xead, 0xeec, 0xf28, 0xf61, 0xf98, 0xfcc, 0xfff, 0x102f,
	0x105d, 0x108a, 0x10b5, 0x10df, 0x1107, 0x112e, 0x1154, 0x1178, 0x119c,
	0x11bf, 0x11e0, 0x1201, 0x1221, 0x1240, 0x125e, 0x127c, 0x1299, 0x12b5,
	0x12d1, 0x12ec, 0x1306, 0x1320, 0x1339, 0x1352, 0x136b, 0x1383, 0x139a,
	0x13b1, 0x13c7, 0x13de, 0x13f3, 0x1409, 0x141e, 0x1432, 0x1447, 0x145b,
	0x146e, 0x1482
};

int eo_y_lut_hdr[143] = {
	61440, 66560, 94208, 110592, 121984, 132160, 138816, 146432, 151264,
	156096, 161440, 165568, 168768, 172224, 175952, 179968, 182240, 200680,
	215102, 226400, 235271, 244625, 250984, 258029, 264312, 269323, 275208,
	280295, 284260, 288817, 294028, 297434, 300794, 304586, 308851, 312465,
	315139, 318120, 321437, 325119, 328439, 330693, 333181, 335922, 338938,
	342251, 344974, 346965, 349143, 351524, 354124, 356960, 360050, 361931,
	363762, 365751, 367912, 370258, 372802, 375559, 377689, 379306, 381056,
	382948, 384994, 387204, 389591, 392167, 394081, 395581, 397197, 398940,
	400818, 402840, 405018, 407363, 409743, 411100, 412561, 414132, 415820,
	417636, 419588, 421685, 423939, 426172, 427472, 428869, 430370, 431980,
	433710, 435567, 437561, 439701, 441999, 443416, 444740, 446160, 447685,
	449321, 451078, 452962, 454986, 457157, 459120, 460370, 461713, 463154,
	464701, 466363, 468146, 470061, 472118, 474326, 475917, 477191, 478560,
	480031, 481611, 483309, 485135, 487098, 489208, 491477, 492719, 494032,
	495444, 496965, 498601, 500363, 502261, 504304, 506506, 508391, 509670,
	511049, 512536, 514140, 515870, 517739, 519756, 521934, 524287
};

int eo_y_lut_hlg[143] = {
	0, 333141, 349525, 360448, 365909, 371370, 376832, 379562, 382293,
	385024, 387754, 390485, 393216, 394581, 395946, 397312, 398677, 409600,
	415061, 420522, 425984, 428714, 431445, 434176, 436906, 439637, 442368,
	443733, 445098, 446464, 447829, 449194, 450560, 451925, 453290, 454656,
	456021, 457386, 458752, 459434, 460117, 460800, 461482, 462165, 462848,
	463530, 464213, 464896, 465578, 466261, 466944, 467626, 468309, 468992,
	469674, 470357, 471040, 471722, 472405, 473088, 473770, 474453, 475136,
	475477, 475818, 476160, 476501, 476842, 477184, 477525, 477866, 478208,
	478549, 478890, 479232, 479573, 479914, 480256, 480597, 464911, 465640,
	466401, 467197, 468028, 468896, 469803, 470750, 471740, 472774, 473854,
	474982, 475648, 476264, 476907, 477579, 478281, 479014, 479780, 480580,
	481416, 482289, 483201, 484154, 485150, 486190, 487276, 488411, 489597,
	490835, 491824, 492500, 493206, 493944, 494714, 495519, 496360, 497238,
	498156, 499114, 500115, 501161, 502254, 503396, 504588, 505834, 507135,
	508199, 508909, 509651, 510426, 511236, 512081, 512965, 513888, 514852,
	515859, 516911, 518010, 519158, 520358, 521611, 522920, 524287
};

int eo_y_lut_sdr[143] = {
	0, 131072, 169072, 192440, 208748, 220974, 231863, 240311, 248196,
	254344, 261528, 265967, 270660, 275934, 280168, 283414, 286977, 311340,
	327680, 339286, 349357, 359062, 365682, 372745, 378914, 383660, 389051,
	394162, 397536, 401264, 405358, 409712, 412138, 414760, 417582, 420609,
	423845, 426638, 428471, 430413, 432468, 434636, 436919, 439320, 441839,
	443423, 444804, 446246, 447751, 449320, 450952, 452649, 454412, 456240,
	458136, 459425, 460441, 461491, 462575, 463695, 464851, 466041, 467268,
	468532, 469832, 471168, 472542, 473954, 475269, 476013, 476776, 477558,
	478359, 479181, 480022, 480882, 481763, 482664, 483585, 484526, 485488,
	486471, 487474, 488499, 489544, 490610, 491609, 492163, 492728, 493304,
	493891, 494489, 495097, 495717, 496348, 496990, 497643, 498307, 498983,
	499670, 500369, 501079, 501800, 502533, 503278, 504034, 504802, 505582,
	506373, 507177, 507948, 508362, 508781, 509207, 509639, 510077, 510521,
	510971, 511427, 511890, 512359, 512834, 513315, 513802, 514296, 514796,
	515303, 515815, 516335, 516860, 517392, 517931, 518476, 519027, 519586,
	520150, 520721, 521299, 521883, 522474, 523072, 523676, 524287
};

int oe_y_lut_hdr[149] = {0, 3, 5, 8, 12, 19, 28, 41, 60, 67, 74, 80, 85,
	96, 105, 113, 120, 134, 146, 157, 167, 184, 200, 214, 227, 250, 270,
	288, 304, 332, 357, 380, 400, 435, 465, 492, 517, 559, 595, 628, 656,
	706, 749, 787, 820, 850, 878, 903, 927, 949, 970, 989, 1008, 1042, 1073,
	1102, 1129, 1154, 1177, 1199, 1219, 1258, 1292, 1324, 1354, 1381, 1407,
	1431, 1453, 1495, 1533, 1568, 1600, 1630, 1657, 1683, 1708, 1753, 1794,
	1831, 1865, 1897, 1926, 1954, 1980, 2028, 2071, 2110, 2146, 2179, 2210,
	2239, 2267, 2317, 2361, 2402, 2440, 2474, 2506, 2536, 2564, 2616, 2662,
	2704, 2742, 2778, 2810, 2841, 2870, 2922, 2969, 3011, 3050, 3086, 3119,
	3150, 3179, 3231, 3278, 3321, 3360, 3396, 3429, 3459, 3488, 3540, 3587,
	3629, 3668, 3703, 3736, 3766, 3795, 3821, 3846, 3870, 3892, 3913, 3934,
	3953, 3971, 3989, 4006, 4022, 4038, 4053, 4068, 4082, 4095
};

int oe_y_lut_hlg[149] = {0, 0, 0, 0, 1, 1, 2, 3, 4, 5, 6, 6, 6, 7, 8, 9,
	9, 10, 12, 12, 13, 15, 16, 18, 19, 21, 24, 25, 27, 30, 33, 36, 39,
	43, 48, 51, 55, 61, 67, 73, 78, 87, 96, 103, 110, 117, 123, 129, 135,
	141, 146, 151, 156, 166, 175, 183, 192, 199, 207, 214, 221, 235, 247,
	259, 271, 282, 293, 303, 313, 332, 350, 367, 384, 399, 414, 429, 443,
	470, 495, 519, 543, 565, 586, 607, 627, 665, 701, 735, 768, 799, 829,
	858, 886, 940, 991, 1039, 1086, 1130, 1173, 1214, 1254, 1330, 1402,
	1470, 1536, 1598, 1659, 1717, 1773, 1881, 1982, 2079, 2165, 2243,
	2313, 2377, 2436, 2541, 2633, 2714, 2788, 2855, 2916, 2972, 3025,
	3119, 3203, 3279, 3347, 3409, 3467, 3520, 3570, 3616, 3660, 3701,
	3740, 3778, 3813, 3847, 3879, 3910, 3939, 3968, 3995, 4022, 4047,
	4072, 4095
};

int oe_y_lut_sdr[149] = {0, 1, 1, 2, 2, 3, 5, 7, 9, 10, 11, 12, 12, 13,
	15, 16, 16, 18, 20, 21, 22, 24, 26, 28, 30, 33, 35, 38, 40, 44, 47,
	50, 53, 59, 63, 67, 71, 78, 85, 90, 95, 105, 113, 121, 127, 134, 140,
	146, 151, 156, 161, 166, 170, 179, 187, 195, 202, 209, 215, 222, 228,
	239, 250, 260, 270, 279, 287, 296, 304, 319, 334, 347, 360, 372, 384,
	395, 406, 426, 445, 464, 481, 497, 513, 528, 542, 569, 595, 619, 642,
	664, 684, 704, 724, 760, 794, 826, 857, 886, 914, 940, 966, 1015, 1060,
	1103, 1144, 1183, 1220, 1255, 1290, 1355, 1415, 1473, 1527, 1579, 1628,
	1676, 1722, 1808, 1889, 1966, 2039, 2108, 2174, 2237, 2298, 2414, 2522,
	2624, 2721, 2814, 2902, 2987, 3068, 3147, 3222, 3296, 3367, 3436, 3503,
	3569, 3633, 3695, 3756, 3816, 3874, 3931, 3987, 4042, 4095
};

int oo_y_lut_hdr_hlg[149] = {
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	3917, 3776, 3662, 3565, 3410, 3288, 3188, 3104, 2968, 2862, 2775,
	2702, 2639, 2584, 2535, 2491, 2452, 2416, 2383, 2352, 2297, 2249,
	2207, 2169, 2134, 2103, 2074, 2047, 2000, 1958, 1921, 1888, 1858,
	1831, 1806, 1782, 1741, 1705, 1672, 1644, 1617, 1594, 1572, 1552,
	1515, 1484, 1456, 1431, 1408, 1387, 1368, 1351, 1319, 1292, 1267,
	1245, 1226, 1208, 1191, 1176, 1148, 1124, 1103, 1084, 1067, 1051,
	1037, 1023, 1000, 979, 960, 944, 929, 915, 903, 891, 870, 852,
	836, 822, 808, 797, 786, 776, 757, 742, 728, 715, 704, 693, 684,
	675, 659, 646, 633, 622, 613, 604, 595, 588, 581, 574, 568, 562,
	557, 551, 546, 542, 537, 533, 529, 525, 522, 518, 515, 512
};

int oo_y_lut_hdr_sdr[149] = {
	3261, 3261, 3261, 3261, 3261, 3261, 3261, 3261, 3261, 3261, 3261,
	3261, 3261, 3261, 3261, 3261, 3261, 3261, 3261, 3261, 3261, 3261,
	3261, 3261, 3261, 3261, 3261, 3261, 3261, 3261, 3261, 3260, 3260,
	3260, 3260, 3260, 3260, 3260, 3259, 3259, 3259, 3259, 3258, 3258,
	3257, 3257, 3256, 3256, 3255, 3255, 3254, 3254, 3253, 3252, 3251,
	3251, 3250, 3249, 3248, 3247, 3246, 3244, 3242, 3240, 3238, 3236,
	3235, 3233, 3231, 3227, 3223, 3220, 3216, 3212, 3208, 3205, 3201,
	3194, 3186, 3179, 3172, 3164, 3157, 3150, 3143, 3128, 3114, 3100,
	3086, 3072, 3058, 3044, 3030, 3003, 2976, 2950, 2924, 2898, 2873,
	2848, 2823, 2774, 2727, 2680, 2635, 2591, 2549, 2507, 2466, 2388,
	2314, 2243, 2176, 2112, 2051, 1993, 1938, 1835, 1742, 1658, 1581,
	1511, 1447, 1388, 1334, 1240, 1159, 1090, 1029, 977, 930, 889,
	852, 818, 787, 759, 733, 708, 685, 664, 644, 624, 606, 589, 572,
	556, 541, 526, 512
};

int oo_y_lut_hlg_hdr[149] = {
	4, 8, 9, 11, 12, 14, 16, 19, 22, 23, 24, 24, 25, 26, 27, 28, 29,
	30, 31, 32, 33, 35, 36, 37, 38, 40, 42, 43, 44, 46, 48, 49, 51,
	53, 55, 57, 58, 61, 63, 65, 67, 70, 73, 75, 77, 79, 81, 82, 84,
	85, 86, 88, 89, 91, 93, 95, 96, 98, 99, 101, 102, 104, 107, 109,
	111, 112, 114, 116, 117, 120, 122, 125, 127, 129, 131, 133, 135,
	138, 141, 144, 146, 148, 151, 153, 155, 158, 162, 165, 168, 171,
	173, 176, 178, 182, 186, 190, 193, 196, 199, 202, 204, 209, 214,
	218, 222, 225, 229, 232, 235, 240, 245, 250, 255, 259, 263, 266,
	270, 276, 282, 288, 293, 297, 302, 306, 310, 317, 324, 330, 336,
	342, 347, 352, 356, 360, 365, 369, 372, 376, 380, 383, 386, 389,
	392, 395, 398, 401, 404, 407, 409
};

int oo_y_lut_sdr_hdr[149] = {
	643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643,
	643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643,
	643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643,
	643, 643, 643, 643, 643, 643, 643, 643, 644, 644, 644, 644, 644,
	644, 644, 644, 645, 645, 645, 645, 645, 646, 646, 646, 647, 647,
	647, 648, 648, 649, 649, 650, 651, 652, 652, 653, 654, 655, 656,
	658, 659, 661, 662, 664, 665, 667, 670, 673, 676, 679, 682, 685,
	688, 691, 698, 704, 710, 717, 723, 729, 736, 742, 755, 768, 782,
	795, 809, 822, 836, 850, 878, 906, 934, 963, 992, 1022, 1052,
	1081, 1142, 1203, 1264, 1326, 1387, 1449, 1510, 1571, 1691, 1808,
	1923, 2036, 2145, 2252, 2357, 2460, 2562, 2662, 2761, 2860, 2959,
	3057, 3156, 3256, 3356, 3457, 3559, 3663, 3768, 3875, 3984, 4095
};

int oo_y_lut_1[149] = {
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095, 4095,
	4095, 4095, 4095, 4095, 4095, 4095
};

#else //HDR2_MODULE
int64_t FloatRev(int64_t ia)
{
	int64_t tmp;
	int64_t iA_exp;
	int64_t iA_val;

	iA_exp = ia >> precision;
	iA_val = ia & 0x3fff;

	if (iA_exp == 0x3f)
		tmp = 0;
	else if (iA_exp >= precision)
		tmp = ((int64_t)(iA_val + (1ULL << precision)))
			<< (iA_exp - precision);
	else
		tmp = ((int64_t)(iA_val + (1ULL << precision) +
			(1ULL << (precision - iA_exp - 1)))) >>
			(precision - iA_exp);

	return tmp;
}



int64_t FloatCon(int64_t iA, int MOD)
{
	int64_t oxs;
	int64_t oXs_exp;
	int64_t oXs_val;
	int64_t exp;
	int64_t val;
	int64_t iA_tmp;
	int64_t diff;

	exp = LOG2(iA);
	/*exp = iA==0 ? 0 : LOG2(iA);*/

	oXs_exp = exp;
	if (exp >= precision)
		val = iA >> (exp - precision);
	else
		val = iA << (precision - exp);

	oXs_val = val & ((1 << precision) - 1);

	if (iA == 0) {
		oXs_exp = 0x3f;
		oXs_val = 0;
	}

	if (exp >= MOD) {
		oXs_exp = MOD - 1;
		oXs_val = 0x3fff;
	}

	oxs = (oXs_exp << 14) + oXs_val;
	iA_tmp = FloatRev(oxs);

	return oxs;
}

int64_t pq_eotf(int64_t e)
{
	double m1 = 2610.0/4096/4;
	double m2 = 2523.0/4096*128;
	double c3 = 2392.0/4096*32;
	double c2 = 2413.0/4096*32;
	double c1 = c3-c2+1;
	int64_t o;
	double e_v = ((double)e) / pow(2, IE_BW);

	o = (int64_t)(pow(((MAX((pow(e_v, (1 / m2)) - c1), 0)) /
		(c2 - c3 * pow(e_v, (1 / m2)))),
		(1 / m1)) * (pow(2, O_BW)) + 0.5);
	return o;
}

int64_t pq_oetf(int64_t o)
{
	double m1 = 2610.0/4096/4;
	double m2 = 2523.0/4096*128;
	double c3 = 2392.0/4096*32;
	double c2 = 2413.0/4096*32;
	double c1 = c3 - c2 + 1;
	int64_t e;
	double o_v = o / pow(2, O_BW);

	if (o == pow(2, O_BW))
		e = (int64_t)pow(2, OE_BW);
	else
		e = (int64_t)(pow(((c1 + c2 * pow(o_v, m1)) /
			(1 + c3 * pow(o_v, m1))),
			m2) * (pow(2, OE_BW)) + 0 .5);
	return e;
}

int64_t gm_eotf(int64_t e)
{
	int64_t o;
	double e_v = ((double)e) / pow(2, IE_BW);

	o = (int64_t)(pow(e_v, 2.4) * (pow(2, O_BW)) + 0.5);
	return o;
}

int64_t gm_oetf(int64_t o)
{
	int64_t e;
	double o_v = o / pow(2, O_BW);

	e = (int64_t)(pow(o_v, 1 / 2.4) * (pow(2, OE_BW)) + 0.5);
	return e;
}

int64_t sld_eotf(int64_t e)
{
	double m = 0.15;
	double p = 2.0;
	int64_t o;
	double e_v = ((double)e) / pow(2, 12);
	double tmp = pow((e_v), -(1.0 / m));

	o = (int64_t)(1.0 / (pow((e_v), -(1.0 / m)) * p - p + 1.0) *
			(pow(2, 32)) + 0.5);
	return o;
}

int64_t sld_oetf(int64_t o)
{
	double m = 0.15;
	double p = 2.0;
	int64_t e;
	double o_v = o / pow(2, 32);

	e = (int64_t)(pow((p*o_v) / ((p - 1)*o_v + 1.0), m) *
			(pow(2, OE_BW)) + 0.5);
	return e;
}

int64_t hlg_eotf(int64_t e)
{
	double a = 0.17883277;
	double b = 0.02372241;
	double c = 1.00429347;
	double e_v = ((double)e) / pow(2, IE_BW);
	double o_v;
	int64_t o;

	if (e_v < 0.5)
		o_v = pow(e_v, 2) / 3;
	else
		o_v = exp((e_v - c) / a) + b;
	o = (int64_t)(o_v * (pow(2, O_BW)) + 0.5);
	return o;
}


int64_t hlg_oetf(int64_t o)
{
	double a = 0.17883277;
	double b = 0.02372241;
	double c = 1.00429347;
	double tmp = 0.08333333;
	int64_t e;
	double e_v;
	double o_v = o / pow(2, O_BW);

	if (o_v < tmp)
		e_v = pow((3 * o_v), 0.5);
	else
		e_v = a * log(o_v - b) + c;

	e = (int64_t)(e_v * (pow(2, OE_BW)) + 0.5);
	return e;
}

int64_t ootf_gain(int64_t o)
{
	double p1 = fmt_io == 1;
	double p2 = 1;
	double p3 = fmt_io == 2;
	double p4 = 1;
	double p5 = 0;
	double o_v = o / pow(2, O_BW);

	double y = 4 * o_v * pow((1 - o_v), 3) * p1 +
			6 * pow(o_v, 2) * pow((1 - o_v), 2) * p2 +
			4 * pow(o_v, 3) * (1 - o_v) * p3 + pow(o_v, 4);

	double gain = o_v == 0 ?  1 : y / o_v;
	int64_t gain_t;

	gain_t = (int64_t)(gain * (pow(2, OGAIN_BW - 2)) + 0.5);

	return gain_t;
}

int64_t hlg_gain(int64_t o)
{
	double p = 1.2 - 1 + 0.42 * (LOG2(peak_out / 1000) / LOG2(10));
	int64_t gain;
	double o_v = o / pow(2, O_BW);

	gain = (int64_t)(pow(o_v, p) * (pow(2, OGAIN_BW)) + 0.5);
	return gain;
}

int64_t nolinear_cgain(int64_t i)
{
	int64_t out;
	double ColorSaturationWeight = 1.2;
	double fscsm = 3 + ColorSaturationWeight *
			MAX((log(((double)i) / pow(2, OE_BW-2)) - 1), 0);

	out = (int64_t)(pow(2, 10) * fscsm);
	return out;
}

/*146bins*/
void eotf_float_gen(int64_t *o_out, MenuFun eotf)
{
	int64_t tmp_o, tmp_e;
	int i;

	for (i = 0; i < 16; i++) {
		tmp_e = (int64_t)((1ULL << (IE_BW - 10)) * i);
		tmp_o = eotf(tmp_e);
		o_out[i] = FloatCon(tmp_o, maxbit);
	}

	for (i = 2; i <= 128; i++) {
		tmp_e = (int64_t)((1ULL << (IE_BW - 7)) * i);
		if (tmp_e == (1 << IE_BW))
			tmp_o = 0xffffffff;
		else
			tmp_o = eotf(tmp_e);
		o_out[i + 14] = FloatCon(tmp_o, maxbit);
	}
}

/*149 bins piece wise lut*/
void oetf_float_gen(int64_t *bin_e, MenuFun oetf)
{
	/*int64_t bin_e[1024]; = zeros(1024);*/
	int64_t bin_o[1025];/* = zeros(1024);*/

	int i = 0, j;
	int bin_num = 0;

	bin_o[i] = 0;
	i++;
	/*bin1~bin8*/
	for (; pow(2, i - 1) * pow(2, 4) < pow(2, 11); i++)
		bin_o[i] = POW(2, i - 1) * POW(2, 4);

	bin_num = i;

	/*bin9~bin44*/
	for (j = 11; j < 20; j++) {/* bin_o< 2^20*/
		for (; i < bin_num + 4; i++)
			bin_o[i] = (i - bin_num) * (POW(2, j - 2)) + POW(2, j);
		bin_num = i;
	}
	bin_num = i;

	/*bin45~bin132*/
	for (j = 20; j < 31; j++) {/*bin_o<2 ^31*/
		for (; i < bin_num + 8; i++)
			bin_o[i] = (i - bin_num) * (POW(2, j - 3)) + POW(2, j);
		bin_num = i;
	}
	/*bin133~bin148*/
	for (; i < bin_num + 16; i++)
		bin_o[i] = (i - bin_num) * (POW(2, 31 - 4)) + POW(2, 31);

	bin_o[i] = 0x100000000;

	for (j = 0; j <= i; j++) {
		bin_e[j] = oetf(bin_o[j]);
		if (bin_e[j] >= (1 << OE_BW))
			bin_e[j] = (1 << OE_BW) - 1;
	}
}

void nolinear_lut_gen(int64_t *bin_c, MenuFun cgain)
{
	/*int bin_c[65]; = zeros(1024);*/
	/*int max_in = 1 << 12;*/
	/*bin_c : 4.10*/
	/*c gain input :OE_BW*/
	int j;

	for (j = 0; j <= 64; j++)
		bin_c[j] = cgain(j * 64);
}

#endif /*HDR2_MODULE*/

static uint force_din_swap = 0xff;
module_param(force_din_swap, uint, 0664);
MODULE_PARM_DESC(force_din_swap, "\n force_din_swap\n");

static uint force_mtrxo_en = 0xff;
module_param(force_mtrxo_en, uint, 0664);
MODULE_PARM_DESC(force_mtrxo_en, "\n force_mtrxo_en\n");

static uint force_mtrxi_en = 0xff;
module_param(force_mtrxi_en, uint, 0664);
MODULE_PARM_DESC(force_mtrxi_en, "\n force_mtrxi_en\n");

static uint force_eo_enable = 0xff;
module_param(force_eo_enable, uint, 0664);
MODULE_PARM_DESC(force_eo_enable, "\n force_eo_enable\n");

static uint force_oe_enable = 0xff;
module_param(force_oe_enable, uint, 0664);
MODULE_PARM_DESC(force_oe_enable, "\n force_oe_enable\n");

static uint force_ogain_enable = 0xff;
module_param(force_ogain_enable, uint, 0664);
MODULE_PARM_DESC(force_ogain_enable, "\n force_ogain_enable\n");

static uint force_cgain_enable = 0xff;
module_param(force_cgain_enable, uint, 0664);
MODULE_PARM_DESC(force_cgain_enable, "\n force_cgain_enable\n");

static uint out_luma = 8;
module_param(out_luma, uint, 0664);
MODULE_PARM_DESC(out_luma, "\n out_luma\n");

static uint in_luma = 1;/*1 as 100luminance*/
module_param(in_luma, uint, 0664);
MODULE_PARM_DESC(in_luma, "\n in_luma\n");

static uint adp_scal_shift = 10;/*1 as 100luminance*/
module_param(adp_scal_shift, uint, 0664);
MODULE_PARM_DESC(adp_scal_shift, "\n adp_scal_shift\n");

static uint alpha_oe_a = 0x1;
module_param(alpha_oe_a, uint, 0664);
MODULE_PARM_DESC(alpha_oe_a, "\n alpha_oe_a\n");

static uint alpha_oe_b = 0x1;
module_param(alpha_oe_b, uint, 0664);
MODULE_PARM_DESC(alpha_oe_b, "\n alpha_oe_b\n");

static uint hdr2_debug;
module_param(hdr2_debug, uint, 0664);
MODULE_PARM_DESC(hdr2_debug, "\n hdr2_debug\n");

/* gamut 3x3 matrix*/
int ncl_2020_709[9] = {
	3401, -1204, -149, -255, 2320, -17, -37, -206, 2291};
/*int cl_2020_709[9] =*/
	/*{-1775, 3867, -44, 3422, -1154, -220 ,-304,	43, 2309}; */
int ncl_709_2020[9] = {1285, 674, 89, 142, 1883, 23, 34, 180, 1834};
/*int cl_709_2020[9] =*/
	/*{436, 1465, 148, 1285, 674, 89, 34, 180, 1834}; */
/*int yrb2ycbcr_cl2020[15] =*/
	/*{876, 0, 0, -566, 0, 566, -902, 902, 0, -462, 0, 462, -521, 521, 0};*/

/* matrix coef */
int rgb2yuvpre[3]	= {0, 0, 0};
int rgb2yuvpos[3]	= {64, 512, 512};
int yuv2rgbpre[3]	= {-64, -512, -512};
int yuv2rgbpos[3]	= {0, 0, 0};

/*matrix coef BT709*/
int yuv2rgbmat[15] = {
	1197, 0, 0, 1197, 1851, 0, 1197, 0, 1163, 1197, 2271, 0, 1197, 0, 2011};
int rgb2ycbcr[15] = {
	230, 594, 52, -125, -323, 448, 448, -412, -36, 0, 0, 0, 0, 0, 0};
int rgb2ycbcr_ncl2020[15] = {
	230, 594, 52, -125, -323, 448, 448, -412, -36, 0, 0, 0, 0, 0, 0};
int rgb2ycbcr_709[15] = {
	186, 627, 63, -103, -345, 448, 448, -407, -41, 0, 0, 0, 0, 0, 0};
int ycbcr2rgb_709[15]  = {
	1192, 0, 1836, 1192, -217, -546, 1192, 2166, 0, 0, 0, 0, 0, 0, 0};
/*int yrb2ycbcr_cl2020[15] =*/
	/*{876, 0, 0, -566, 0, 566, -902, 902, 0, -462, 0, 462, -521, 521, 0};*/
int ycbcr2rgb_ncl2020[15] = {
	1197, 0, 1726, 1197, -193, -669, 1197, 2202, 0, 0, 0, 0, 0, 0, 0};
/*int ycbcr2yrb_cl2020[15] =*/
	/*{1197,0,0,1197,0,1163,1197,1851,0, 1197, 0, 2011, 1197, 2271, 0};*/

static int bypass_coeff[15] = {
	1024, 0, 0,
	0, 1024, 0,
	0, 0, 1024,
	0, 0, 0,
	0, 0, 0,
};

/*in/out matrix*/
void set_hdr_matrix(
	enum hdr_module_sel module_sel,
	enum hdr_matrix_sel mtx_sel,
	struct hdr_proc_mtx_param_s *hdr_mtx_param)
{
	unsigned int MATRIXI_COEF00_01 = 0;
	unsigned int MATRIXI_COEF02_10 = 0;
	unsigned int MATRIXI_COEF11_12 = 0;
	unsigned int MATRIXI_COEF20_21 = 0;
	unsigned int MATRIXI_COEF22 = 0;
	unsigned int MATRIXI_COEF30_31 = 0;
	unsigned int MATRIXI_COEF32_40 = 0;
	unsigned int MATRIXI_COEF41_42 = 0;
	unsigned int MATRIXI_OFFSET0_1 = 0;
	unsigned int MATRIXI_OFFSET2 = 0;
	unsigned int MATRIXI_PRE_OFFSET0_1 = 0;
	unsigned int MATRIXI_PRE_OFFSET2 = 0;
	unsigned int MATRIXI_CLIP = 0;
	unsigned int MATRIXI_EN_CTRL = 0;

	unsigned int MATRIXO_COEF00_01 = 0;
	unsigned int MATRIXO_COEF02_10 = 0;
	unsigned int MATRIXO_COEF11_12 = 0;
	unsigned int MATRIXO_COEF20_21 = 0;
	unsigned int MATRIXO_COEF22 = 0;
	unsigned int MATRIXO_COEF30_31 = 0;
	unsigned int MATRIXO_COEF32_40 = 0;
	unsigned int MATRIXO_COEF41_42 = 0;
	unsigned int MATRIXO_OFFSET0_1 = 0;
	unsigned int MATRIXO_OFFSET2 = 0;
	unsigned int MATRIXO_PRE_OFFSET0_1 = 0;
	unsigned int MATRIXO_PRE_OFFSET2 = 0;
	unsigned int MATRIXO_CLIP = 0;
	unsigned int MATRIXO_EN_CTRL = 0;

	unsigned int CGAIN_OFFT = 0;
	unsigned int CGAIN_COEF0 = 0;
	unsigned int CGAIN_COEF1 = 0;
	unsigned int ADPS_CTRL = 0;
	unsigned int ADPS_ALPHA0 = 0;
	unsigned int ADPS_ALPHA1 = 0;
	unsigned int ADPS_BETA0 = 0;
	unsigned int ADPS_BETA1 = 0;
	unsigned int ADPS_BETA2 = 0;
	unsigned int ADPS_COEF0 = 0;
	unsigned int ADPS_COEF1 = 0;
	unsigned int GMUT_CTRL = 0;
	unsigned int GMUT_COEF0 = 0;
	unsigned int GMUT_COEF1 = 0;
	unsigned int GMUT_COEF2 = 0;
	unsigned int GMUT_COEF3 = 0;
	unsigned int GMUT_COEF4 = 0;

	unsigned int hdr_ctrl = 0;

	int adpscl_mode = 0;

	int c_gain_lim_coef[3];
	int gmut_coef[3][3];
	int gmut_shift;
	int adpscl_enable[3];
	int adpscl_alpha[3];
	int adpscl_shift[3];
	int adpscl_ys_coef[3];
	int adpscl_beta[3];
	int adpscl_beta_s[3];

	int i = 0;
	int mtx[15] = {
		1024, 0, 0,
		0, 1024, 0,
		0, 0, 1024,
		0, 0, 0,
		0, 0, 0,
	};

	if (module_sel & VD1_HDR) {
		MATRIXI_COEF00_01 = VD1_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF00_01 = VD1_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF02_10 = VD1_HDR2_MATRIXI_COEF02_10;
		MATRIXI_COEF11_12 = VD1_HDR2_MATRIXI_COEF11_12;
		MATRIXI_COEF20_21 = VD1_HDR2_MATRIXI_COEF20_21;
		MATRIXI_COEF22 = VD1_HDR2_MATRIXI_COEF22;
		MATRIXI_COEF30_31 = VD1_HDR2_MATRIXI_COEF30_31;
		MATRIXI_COEF32_40 = VD1_HDR2_MATRIXI_COEF32_40;
		MATRIXI_COEF41_42 = VD1_HDR2_MATRIXI_COEF41_42;
		MATRIXI_OFFSET0_1 = VD1_HDR2_MATRIXI_OFFSET0_1;
		MATRIXI_OFFSET2 = VD1_HDR2_MATRIXI_OFFSET2;
		MATRIXI_PRE_OFFSET0_1 = VD1_HDR2_MATRIXI_PRE_OFFSET0_1;
		MATRIXI_PRE_OFFSET2 = VD1_HDR2_MATRIXI_PRE_OFFSET2;
		MATRIXI_CLIP = VD1_HDR2_MATRIXI_CLIP;
		MATRIXI_EN_CTRL = VD1_HDR2_MATRIXI_EN_CTRL;

		MATRIXO_COEF00_01 = VD1_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF00_01 = VD1_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF02_10 = VD1_HDR2_MATRIXO_COEF02_10;
		MATRIXO_COEF11_12 = VD1_HDR2_MATRIXO_COEF11_12;
		MATRIXO_COEF20_21 = VD1_HDR2_MATRIXO_COEF20_21;
		MATRIXO_COEF22 = VD1_HDR2_MATRIXO_COEF22;
		MATRIXO_COEF30_31 = VD1_HDR2_MATRIXO_COEF30_31;
		MATRIXO_COEF32_40 = VD1_HDR2_MATRIXO_COEF32_40;
		MATRIXO_COEF41_42 = VD1_HDR2_MATRIXO_COEF41_42;
		MATRIXO_OFFSET0_1 = VD1_HDR2_MATRIXO_OFFSET0_1;
		MATRIXO_OFFSET2 = VD1_HDR2_MATRIXO_OFFSET2;
		MATRIXO_PRE_OFFSET0_1 = VD1_HDR2_MATRIXO_PRE_OFFSET0_1;
		MATRIXO_PRE_OFFSET2 = VD1_HDR2_MATRIXO_PRE_OFFSET2;
		MATRIXO_CLIP = VD1_HDR2_MATRIXO_CLIP;
		MATRIXO_EN_CTRL = VD1_HDR2_MATRIXO_EN_CTRL;

		CGAIN_OFFT = VD1_HDR2_CGAIN_OFFT;
		CGAIN_COEF0 = VD1_HDR2_CGAIN_COEF0;
		CGAIN_COEF1 = VD1_HDR2_CGAIN_COEF1;
		ADPS_CTRL = VD1_HDR2_ADPS_CTRL;
		ADPS_ALPHA0 = VD1_HDR2_ADPS_ALPHA0;
		ADPS_ALPHA1 = VD1_HDR2_ADPS_ALPHA1;
		ADPS_BETA0 = VD1_HDR2_ADPS_BETA0;
		ADPS_BETA1 = VD1_HDR2_ADPS_BETA1;
		ADPS_BETA2 = VD1_HDR2_ADPS_BETA2;
		ADPS_COEF0 = VD1_HDR2_ADPS_COEF0;
		ADPS_COEF1 = VD1_HDR2_ADPS_COEF1;
		GMUT_CTRL = VD1_HDR2_GMUT_CTRL;
		GMUT_COEF0 = VD1_HDR2_GMUT_COEF0;
		GMUT_COEF1 = VD1_HDR2_GMUT_COEF1;
		GMUT_COEF2 = VD1_HDR2_GMUT_COEF2;
		GMUT_COEF3 = VD1_HDR2_GMUT_COEF3;
		GMUT_COEF4 = VD1_HDR2_GMUT_COEF4;

		hdr_ctrl = VD1_HDR2_CTRL;
	} else if (module_sel & VD2_HDR) {
		MATRIXI_COEF00_01 = VDIN1_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF00_01 = VDIN1_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF02_10 = VDIN1_HDR2_MATRIXI_COEF02_10;
		MATRIXI_COEF11_12 = VDIN1_HDR2_MATRIXI_COEF11_12;
		MATRIXI_COEF20_21 = VDIN1_HDR2_MATRIXI_COEF20_21;
		MATRIXI_COEF22 = VDIN1_HDR2_MATRIXI_COEF22;
		MATRIXI_COEF30_31 = VDIN1_HDR2_MATRIXI_COEF30_31;
		MATRIXI_COEF32_40 = VDIN1_HDR2_MATRIXI_COEF32_40;
		MATRIXI_COEF41_42 = VDIN1_HDR2_MATRIXI_COEF41_42;
		MATRIXI_OFFSET0_1 = VDIN1_HDR2_MATRIXI_OFFSET0_1;
		MATRIXI_OFFSET2 = VDIN1_HDR2_MATRIXI_OFFSET2;
		MATRIXI_PRE_OFFSET0_1 = VDIN1_HDR2_MATRIXI_PRE_OFFSET0_1;
		MATRIXI_PRE_OFFSET2 = VDIN1_HDR2_MATRIXI_PRE_OFFSET2;
		MATRIXI_CLIP = VDIN1_HDR2_MATRIXI_CLIP;
		MATRIXI_EN_CTRL = VDIN1_HDR2_MATRIXI_EN_CTRL;

		MATRIXO_COEF00_01 = VDIN1_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF00_01 = VDIN1_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF02_10 = VDIN1_HDR2_MATRIXO_COEF02_10;
		MATRIXO_COEF11_12 = VDIN1_HDR2_MATRIXO_COEF11_12;
		MATRIXO_COEF20_21 = VDIN1_HDR2_MATRIXO_COEF20_21;
		MATRIXO_COEF22 = VDIN1_HDR2_MATRIXO_COEF22;
		MATRIXO_COEF30_31 = VDIN1_HDR2_MATRIXO_COEF30_31;
		MATRIXO_COEF32_40 = VDIN1_HDR2_MATRIXO_COEF32_40;
		MATRIXO_COEF41_42 = VDIN1_HDR2_MATRIXO_COEF41_42;
		MATRIXO_OFFSET0_1 = VDIN1_HDR2_MATRIXO_OFFSET0_1;
		MATRIXO_OFFSET2 = VDIN1_HDR2_MATRIXO_OFFSET2;
		MATRIXO_PRE_OFFSET0_1 = VDIN1_HDR2_MATRIXO_PRE_OFFSET0_1;
		MATRIXO_PRE_OFFSET2 = VDIN1_HDR2_MATRIXO_PRE_OFFSET2;
		MATRIXO_CLIP = VDIN1_HDR2_MATRIXO_CLIP;
		MATRIXO_EN_CTRL = VDIN1_HDR2_MATRIXO_EN_CTRL;

		CGAIN_OFFT = VDIN1_HDR2_CGAIN_OFFT;
		CGAIN_COEF0 = VDIN1_HDR2_CGAIN_COEF0;
		CGAIN_COEF1 = VDIN1_HDR2_CGAIN_COEF1;
		ADPS_CTRL = VDIN1_HDR2_ADPS_CTRL;
		ADPS_ALPHA0 = VDIN1_HDR2_ADPS_ALPHA0;
		ADPS_ALPHA1 = VDIN1_HDR2_ADPS_ALPHA1;
		ADPS_BETA0 = VDIN1_HDR2_ADPS_BETA0;
		ADPS_BETA1 = VDIN1_HDR2_ADPS_BETA1;
		ADPS_BETA2 = VDIN1_HDR2_ADPS_BETA2;
		ADPS_COEF0 = VDIN1_HDR2_ADPS_COEF0;
		ADPS_COEF1 = VDIN1_HDR2_ADPS_COEF1;
		GMUT_CTRL = VDIN1_HDR2_GMUT_CTRL;
		GMUT_COEF0 = VDIN1_HDR2_GMUT_COEF0;
		GMUT_COEF1 = VDIN1_HDR2_GMUT_COEF1;
		GMUT_COEF2 = VDIN1_HDR2_GMUT_COEF2;
		GMUT_COEF3 = VDIN1_HDR2_GMUT_COEF3;
		GMUT_COEF4 = VDIN1_HDR2_GMUT_COEF4;

		hdr_ctrl = VDIN1_HDR2_CTRL;
	} else if (module_sel & OSD1_HDR) {
		MATRIXI_COEF00_01 = OSD1_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF00_01 = OSD1_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF02_10 = OSD1_HDR2_MATRIXI_COEF02_10;
		MATRIXI_COEF11_12 = OSD1_HDR2_MATRIXI_COEF11_12;
		MATRIXI_COEF20_21 = OSD1_HDR2_MATRIXI_COEF20_21;
		MATRIXI_COEF22 = OSD1_HDR2_MATRIXI_COEF22;
		MATRIXI_COEF30_31 = OSD1_HDR2_MATRIXI_COEF30_31;
		MATRIXI_COEF32_40 = OSD1_HDR2_MATRIXI_COEF32_40;
		MATRIXI_COEF41_42 = OSD1_HDR2_MATRIXI_COEF41_42;
		MATRIXI_OFFSET0_1 = OSD1_HDR2_MATRIXI_OFFSET0_1;
		MATRIXI_OFFSET2 = OSD1_HDR2_MATRIXI_OFFSET2;
		MATRIXI_PRE_OFFSET0_1 = OSD1_HDR2_MATRIXI_PRE_OFFSET0_1;
		MATRIXI_PRE_OFFSET2 = OSD1_HDR2_MATRIXI_PRE_OFFSET2;
		MATRIXI_CLIP = OSD1_HDR2_MATRIXI_CLIP;
		MATRIXI_EN_CTRL = OSD1_HDR2_MATRIXI_EN_CTRL;

		MATRIXO_COEF00_01 = OSD1_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF00_01 = OSD1_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF02_10 = OSD1_HDR2_MATRIXO_COEF02_10;
		MATRIXO_COEF11_12 = OSD1_HDR2_MATRIXO_COEF11_12;
		MATRIXO_COEF20_21 = OSD1_HDR2_MATRIXO_COEF20_21;
		MATRIXO_COEF22 = OSD1_HDR2_MATRIXO_COEF22;
		MATRIXO_COEF30_31 = OSD1_HDR2_MATRIXO_COEF30_31;
		MATRIXO_COEF32_40 = OSD1_HDR2_MATRIXO_COEF32_40;
		MATRIXO_COEF41_42 = OSD1_HDR2_MATRIXO_COEF41_42;
		MATRIXO_OFFSET0_1 = OSD1_HDR2_MATRIXO_OFFSET0_1;
		MATRIXO_OFFSET2 = OSD1_HDR2_MATRIXO_OFFSET2;
		MATRIXO_PRE_OFFSET0_1 = OSD1_HDR2_MATRIXO_PRE_OFFSET0_1;
		MATRIXO_PRE_OFFSET2 = OSD1_HDR2_MATRIXO_PRE_OFFSET2;
		MATRIXO_CLIP = OSD1_HDR2_MATRIXO_CLIP;
		MATRIXO_EN_CTRL = OSD1_HDR2_MATRIXO_EN_CTRL;

		CGAIN_OFFT = OSD1_HDR2_CGAIN_OFFT;
		CGAIN_COEF0 = OSD1_HDR2_CGAIN_COEF0;
		CGAIN_COEF1 = OSD1_HDR2_CGAIN_COEF1;
		ADPS_CTRL = OSD1_HDR2_ADPS_CTRL;
		ADPS_ALPHA0 = OSD1_HDR2_ADPS_ALPHA0;
		ADPS_ALPHA1 = OSD1_HDR2_ADPS_ALPHA1;
		ADPS_BETA0 = OSD1_HDR2_ADPS_BETA0;
		ADPS_BETA1 = OSD1_HDR2_ADPS_BETA1;
		ADPS_BETA2 = OSD1_HDR2_ADPS_BETA2;
		ADPS_COEF0 = OSD1_HDR2_ADPS_COEF0;
		ADPS_COEF1 = OSD1_HDR2_ADPS_COEF1;
		GMUT_CTRL = OSD1_HDR2_GMUT_CTRL;
		GMUT_COEF0 = OSD1_HDR2_GMUT_COEF0;
		GMUT_COEF1 = OSD1_HDR2_GMUT_COEF1;
		GMUT_COEF2 = OSD1_HDR2_GMUT_COEF2;
		GMUT_COEF3 = OSD1_HDR2_GMUT_COEF3;
		GMUT_COEF4 = OSD1_HDR2_GMUT_COEF4;

		hdr_ctrl = OSD1_HDR2_CTRL;
	} else if (module_sel & DI_HDR) {
		MATRIXI_COEF00_01 = DI_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF00_01 = DI_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF02_10 = DI_HDR2_MATRIXI_COEF02_10;
		MATRIXI_COEF11_12 = DI_HDR2_MATRIXI_COEF11_12;
		MATRIXI_COEF20_21 = DI_HDR2_MATRIXI_COEF20_21;
		MATRIXI_COEF22 = DI_HDR2_MATRIXI_COEF22;
		MATRIXI_COEF30_31 = DI_HDR2_MATRIXI_COEF30_31;
		MATRIXI_COEF32_40 = DI_HDR2_MATRIXI_COEF32_40;
		MATRIXI_COEF41_42 = DI_HDR2_MATRIXI_COEF41_42;
		MATRIXI_OFFSET0_1 = DI_HDR2_MATRIXI_OFFSET0_1;
		MATRIXI_OFFSET2 = DI_HDR2_MATRIXI_OFFSET2;
		MATRIXI_PRE_OFFSET0_1 = DI_HDR2_MATRIXI_PRE_OFFSET0_1;
		MATRIXI_PRE_OFFSET2 = DI_HDR2_MATRIXI_PRE_OFFSET2;
		MATRIXI_CLIP = DI_HDR2_MATRIXI_CLIP;
		MATRIXI_EN_CTRL = DI_HDR2_MATRIXI_EN_CTRL;

		MATRIXO_COEF00_01 = DI_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF00_01 = DI_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF02_10 = DI_HDR2_MATRIXO_COEF02_10;
		MATRIXO_COEF11_12 = DI_HDR2_MATRIXO_COEF11_12;
		MATRIXO_COEF20_21 = DI_HDR2_MATRIXO_COEF20_21;
		MATRIXO_COEF22 = DI_HDR2_MATRIXO_COEF22;
		MATRIXO_COEF30_31 = DI_HDR2_MATRIXO_COEF30_31;
		MATRIXO_COEF32_40 = DI_HDR2_MATRIXO_COEF32_40;
		MATRIXO_COEF41_42 = DI_HDR2_MATRIXO_COEF41_42;
		MATRIXO_OFFSET0_1 = DI_HDR2_MATRIXO_OFFSET0_1;
		MATRIXO_OFFSET2 = DI_HDR2_MATRIXO_OFFSET2;
		MATRIXO_PRE_OFFSET0_1 = DI_HDR2_MATRIXO_PRE_OFFSET0_1;
		MATRIXO_PRE_OFFSET2 = DI_HDR2_MATRIXO_PRE_OFFSET2;
		MATRIXO_CLIP = DI_HDR2_MATRIXO_CLIP;
		MATRIXO_EN_CTRL = DI_HDR2_MATRIXO_EN_CTRL;

		CGAIN_OFFT = DI_HDR2_CGAIN_OFFT;
		CGAIN_COEF0 = DI_HDR2_CGAIN_COEF0;
		CGAIN_COEF1 = DI_HDR2_CGAIN_COEF1;
		ADPS_CTRL = DI_HDR2_ADPS_CTRL;
		ADPS_ALPHA0 = DI_HDR2_ADPS_ALPHA0;
		ADPS_ALPHA1 = DI_HDR2_ADPS_ALPHA1;
		ADPS_BETA0 = DI_HDR2_ADPS_BETA0;
		ADPS_BETA1 = DI_HDR2_ADPS_BETA1;
		ADPS_BETA2 = DI_HDR2_ADPS_BETA2;
		ADPS_COEF0 = DI_HDR2_ADPS_COEF0;
		ADPS_COEF1 = DI_HDR2_ADPS_COEF1;
		GMUT_CTRL = DI_HDR2_GMUT_CTRL;
		GMUT_COEF0 = DI_HDR2_GMUT_COEF0;
		GMUT_COEF1 = DI_HDR2_GMUT_COEF1;
		GMUT_COEF2 = DI_HDR2_GMUT_COEF2;
		GMUT_COEF3 = DI_HDR2_GMUT_COEF3;
		GMUT_COEF4 = DI_HDR2_GMUT_COEF4;

		hdr_ctrl = DI_HDR2_CTRL;
	} else if (module_sel & VDIN0_HDR) {
		MATRIXI_COEF00_01 = VDIN0_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF00_01 = VDIN0_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF02_10 = VDIN0_HDR2_MATRIXI_COEF02_10;
		MATRIXI_COEF11_12 = VDIN0_HDR2_MATRIXI_COEF11_12;
		MATRIXI_COEF20_21 = VDIN0_HDR2_MATRIXI_COEF20_21;
		MATRIXI_COEF22 = VDIN0_HDR2_MATRIXI_COEF22;
		MATRIXI_COEF30_31 = VDIN0_HDR2_MATRIXI_COEF30_31;
		MATRIXI_COEF32_40 = VDIN0_HDR2_MATRIXI_COEF32_40;
		MATRIXI_COEF41_42 = VDIN0_HDR2_MATRIXI_COEF41_42;
		MATRIXI_OFFSET0_1 = VDIN0_HDR2_MATRIXI_OFFSET0_1;
		MATRIXI_OFFSET2 = VDIN0_HDR2_MATRIXI_OFFSET2;
		MATRIXI_PRE_OFFSET0_1 = VDIN0_HDR2_MATRIXI_PRE_OFFSET0_1;
		MATRIXI_PRE_OFFSET2 = VDIN0_HDR2_MATRIXI_PRE_OFFSET2;
		MATRIXI_CLIP = VDIN0_HDR2_MATRIXI_CLIP;
		MATRIXI_EN_CTRL = VDIN0_HDR2_MATRIXI_EN_CTRL;

		MATRIXO_COEF00_01 = VDIN0_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF00_01 = VDIN0_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF02_10 = VDIN0_HDR2_MATRIXO_COEF02_10;
		MATRIXO_COEF11_12 = VDIN0_HDR2_MATRIXO_COEF11_12;
		MATRIXO_COEF20_21 = VDIN0_HDR2_MATRIXO_COEF20_21;
		MATRIXO_COEF22 = VDIN0_HDR2_MATRIXO_COEF22;
		MATRIXO_COEF30_31 = VDIN0_HDR2_MATRIXO_COEF30_31;
		MATRIXO_COEF32_40 = VDIN0_HDR2_MATRIXO_COEF32_40;
		MATRIXO_COEF41_42 = VDIN0_HDR2_MATRIXO_COEF41_42;
		MATRIXO_OFFSET0_1 = VDIN0_HDR2_MATRIXO_OFFSET0_1;
		MATRIXO_OFFSET2 = VDIN0_HDR2_MATRIXO_OFFSET2;
		MATRIXO_PRE_OFFSET0_1 = VDIN0_HDR2_MATRIXO_PRE_OFFSET0_1;
		MATRIXO_PRE_OFFSET2 = VDIN0_HDR2_MATRIXO_PRE_OFFSET2;
		MATRIXO_CLIP = VDIN0_HDR2_MATRIXO_CLIP;
		MATRIXO_EN_CTRL = VDIN0_HDR2_MATRIXO_EN_CTRL;

		CGAIN_OFFT = VDIN0_HDR2_CGAIN_OFFT;
		CGAIN_COEF0 = VDIN0_HDR2_CGAIN_COEF0;
		CGAIN_COEF1 = VDIN0_HDR2_CGAIN_COEF1;
		ADPS_CTRL = VDIN0_HDR2_ADPS_CTRL;
		ADPS_ALPHA0 = VDIN0_HDR2_ADPS_ALPHA0;
		ADPS_ALPHA1 = VDIN0_HDR2_ADPS_ALPHA1;
		ADPS_BETA0 = VDIN0_HDR2_ADPS_BETA0;
		ADPS_BETA1 = VDIN0_HDR2_ADPS_BETA1;
		ADPS_BETA2 = VDIN0_HDR2_ADPS_BETA2;
		ADPS_COEF0 = VDIN0_HDR2_ADPS_COEF0;
		ADPS_COEF1 = VDIN0_HDR2_ADPS_COEF1;
		GMUT_CTRL = VDIN0_HDR2_GMUT_CTRL;
		GMUT_COEF0 = VDIN0_HDR2_GMUT_COEF0;
		GMUT_COEF1 = VDIN0_HDR2_GMUT_COEF1;
		GMUT_COEF2 = VDIN0_HDR2_GMUT_COEF2;
		GMUT_COEF3 = VDIN0_HDR2_GMUT_COEF3;
		GMUT_COEF4 = VDIN0_HDR2_GMUT_COEF4;

		hdr_ctrl = VDIN0_HDR2_CTRL;
	} else if (module_sel & VDIN1_HDR) {
		MATRIXI_COEF00_01 = VDIN1_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF00_01 = VDIN1_HDR2_MATRIXI_COEF00_01;
		MATRIXI_COEF02_10 = VDIN1_HDR2_MATRIXI_COEF02_10;
		MATRIXI_COEF11_12 = VDIN1_HDR2_MATRIXI_COEF11_12;
		MATRIXI_COEF20_21 = VDIN1_HDR2_MATRIXI_COEF20_21;
		MATRIXI_COEF22 = VDIN1_HDR2_MATRIXI_COEF22;
		MATRIXI_COEF30_31 = VDIN1_HDR2_MATRIXI_COEF30_31;
		MATRIXI_COEF32_40 = VDIN1_HDR2_MATRIXI_COEF32_40;
		MATRIXI_COEF41_42 = VDIN1_HDR2_MATRIXI_COEF41_42;
		MATRIXI_OFFSET0_1 = VDIN1_HDR2_MATRIXI_OFFSET0_1;
		MATRIXI_OFFSET2 = VDIN1_HDR2_MATRIXI_OFFSET2;
		MATRIXI_PRE_OFFSET0_1 = VDIN1_HDR2_MATRIXI_PRE_OFFSET0_1;
		MATRIXI_PRE_OFFSET2 = VDIN1_HDR2_MATRIXI_PRE_OFFSET2;
		MATRIXI_CLIP = VDIN1_HDR2_MATRIXI_CLIP;
		MATRIXI_EN_CTRL = VDIN1_HDR2_MATRIXI_EN_CTRL;

		MATRIXO_COEF00_01 = VDIN1_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF00_01 = VDIN1_HDR2_MATRIXO_COEF00_01;
		MATRIXO_COEF02_10 = VDIN1_HDR2_MATRIXO_COEF02_10;
		MATRIXO_COEF11_12 = VDIN1_HDR2_MATRIXO_COEF11_12;
		MATRIXO_COEF20_21 = VDIN1_HDR2_MATRIXO_COEF20_21;
		MATRIXO_COEF22 = VDIN1_HDR2_MATRIXO_COEF22;
		MATRIXO_COEF30_31 = VDIN1_HDR2_MATRIXO_COEF30_31;
		MATRIXO_COEF32_40 = VDIN1_HDR2_MATRIXO_COEF32_40;
		MATRIXO_COEF41_42 = VDIN1_HDR2_MATRIXO_COEF41_42;
		MATRIXO_OFFSET0_1 = VDIN1_HDR2_MATRIXO_OFFSET0_1;
		MATRIXO_OFFSET2 = VDIN1_HDR2_MATRIXO_OFFSET2;
		MATRIXO_PRE_OFFSET0_1 = VDIN1_HDR2_MATRIXO_PRE_OFFSET0_1;
		MATRIXO_PRE_OFFSET2 = VDIN1_HDR2_MATRIXO_PRE_OFFSET2;
		MATRIXO_CLIP = VDIN1_HDR2_MATRIXO_CLIP;
		MATRIXO_EN_CTRL = VDIN1_HDR2_MATRIXO_EN_CTRL;

		CGAIN_OFFT = VDIN1_HDR2_CGAIN_OFFT;
		CGAIN_COEF0 = VDIN1_HDR2_CGAIN_COEF0;
		CGAIN_COEF1 = VDIN1_HDR2_CGAIN_COEF1;
		ADPS_CTRL = VDIN1_HDR2_ADPS_CTRL;
		ADPS_ALPHA0 = VDIN1_HDR2_ADPS_ALPHA0;
		ADPS_ALPHA1 = VDIN1_HDR2_ADPS_ALPHA1;
		ADPS_BETA0 = VDIN1_HDR2_ADPS_BETA0;
		ADPS_BETA1 = VDIN1_HDR2_ADPS_BETA1;
		ADPS_BETA2 = VDIN1_HDR2_ADPS_BETA2;
		ADPS_COEF0 = VDIN1_HDR2_ADPS_COEF0;
		ADPS_COEF1 = VDIN1_HDR2_ADPS_COEF1;
		GMUT_CTRL = VDIN1_HDR2_GMUT_CTRL;
		GMUT_COEF0 = VDIN1_HDR2_GMUT_COEF0;
		GMUT_COEF1 = VDIN1_HDR2_GMUT_COEF1;
		GMUT_COEF2 = VDIN1_HDR2_GMUT_COEF2;
		GMUT_COEF3 = VDIN1_HDR2_GMUT_COEF3;
		GMUT_COEF4 = VDIN1_HDR2_GMUT_COEF4;

		hdr_ctrl = VDIN1_HDR2_CTRL;
	}

	VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, hdr_mtx_param->mtx_on, 13, 1);

	if (mtx_sel & HDR_IN_MTX) {
		if (hdr_mtx_param->mtx_in) {
			for (i = 0; i < 15; i++)
				mtx[i] = hdr_mtx_param->mtx_in[i];
		}
		VSYNC_WR_MPEG_REG(MATRIXI_EN_CTRL, hdr_mtx_param->mtx_on);
		/*yuv in*/
		VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, hdr_mtx_param->mtx_on, 4, 1);

		VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, hdr_mtx_param->mtx_only,
			16, 1);
		VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, 0, 17, 1);
		/*mtx in en*/
		VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, 1, 14, 1);

		VSYNC_WR_MPEG_REG(MATRIXI_COEF00_01,
			(mtx[0 * 3 + 0] << 16) | (mtx[0 * 3 + 1] & 0x1FFF));
		VSYNC_WR_MPEG_REG(MATRIXI_COEF02_10,
			(mtx[0 * 3 + 2] << 16) | (mtx[1 * 3 + 0] & 0x1FFF));
		VSYNC_WR_MPEG_REG(MATRIXI_COEF11_12,
			(mtx[1 * 3 + 1] << 16) | (mtx[1 * 3 + 2] & 0x1FFF));
		VSYNC_WR_MPEG_REG(MATRIXI_COEF20_21,
			(mtx[2 * 3 + 0] << 16) | (mtx[2 * 3 + 1] & 0x1FFF));
		VSYNC_WR_MPEG_REG(MATRIXI_COEF22,
			mtx[2 * 3 + 2]);
		VSYNC_WR_MPEG_REG(MATRIXI_OFFSET0_1,
			(yuv2rgbpos[0] << 16) | (yuv2rgbpos[1] & 0xFFF));
		VSYNC_WR_MPEG_REG(MATRIXI_OFFSET2,
			yuv2rgbpos[2]);
		VSYNC_WR_MPEG_REG(MATRIXI_PRE_OFFSET0_1,
			(yuv2rgbpre[0] << 16)|(yuv2rgbpre[1] & 0xFFF));
		VSYNC_WR_MPEG_REG(MATRIXI_PRE_OFFSET2,
			yuv2rgbpre[2]);

	} else if (mtx_sel & HDR_GAMUT_MTX) {
		if (hdr_mtx_param->mtx_gamut) {
			for (i = 0; i < 9; i++)
				gmut_coef[i/3][i%3] =
					hdr_mtx_param->mtx_gamut[i];
		}
		gmut_shift = 11;

		if (hdr_mtx_param->mtx_cgain) {
			for (i = 0; i < 3; i++)
				c_gain_lim_coef[i] =
					hdr_mtx_param->mtx_cgain[i] << 2;
		}

		adpscl_mode = 1;/*according to test code*/
		for (i = 0; i < 3; i++) {
			adpscl_enable[i] = 0;
			if (hdr_mtx_param->p_sel & HDR_SDR)
				adpscl_alpha[i] = out_luma *
					(1 << adp_scal_shift) / in_luma;
			else if (hdr_mtx_param->p_sel & SDR_HDR)
				adpscl_alpha[i] = in_luma *
					(1 << adp_scal_shift) / out_luma;
			else if (hdr_mtx_param->p_sel & HDR_BYPASS)
				adpscl_alpha[i] = out_luma *
					(1 << adp_scal_shift) / in_luma;
			else if (hdr_mtx_param->p_sel & HLG_SDR)
				adpscl_alpha[i] = out_luma *
					(1 << adp_scal_shift) / in_luma;
			else if (hdr_mtx_param->p_sel & HLG_HDR)
				adpscl_alpha[i] = 1 *
					(1 << adp_scal_shift) / in_luma;
			adpscl_shift[i] = adp_scal_shift;
			if (hdr_mtx_param->mtx_ogain)
				adpscl_ys_coef[i] =
					hdr_mtx_param->mtx_ogain[i] << 1;
			adpscl_beta_s[i] = 0;
			adpscl_beta[i] = FLTZERO;
		}

		/*gamut mode: 1->gamut before ootf*/
					/*2->gamut after ootf*/
					/*other->disable gamut*/
		VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, 1, 6, 2);

	    VSYNC_WR_MPEG_REG(GMUT_CTRL, gmut_shift);
	    VSYNC_WR_MPEG_REG(GMUT_COEF0,
			gmut_coef[0][1] << 16 | gmut_coef[0][0]);
	    VSYNC_WR_MPEG_REG(GMUT_COEF1,
			gmut_coef[1][0] << 16 | gmut_coef[0][2]);
	    VSYNC_WR_MPEG_REG(GMUT_COEF2,
			gmut_coef[1][2] << 16 | gmut_coef[1][1]);
	    VSYNC_WR_MPEG_REG(GMUT_COEF3,
			gmut_coef[2][1] << 16 | gmut_coef[2][0]);
	    VSYNC_WR_MPEG_REG(GMUT_COEF4, gmut_coef[2][2]);

	    VSYNC_WR_MPEG_REG(CGAIN_COEF0,
			c_gain_lim_coef[1] << 16 |
			c_gain_lim_coef[0]);
	    VSYNC_WR_MPEG_REG(CGAIN_COEF1, c_gain_lim_coef[2]);

	    VSYNC_WR_MPEG_REG(ADPS_CTRL, adpscl_enable[2] << 6 |
							adpscl_enable[1] << 5 |
							adpscl_enable[0] << 4 |
							adpscl_mode);
		VSYNC_WR_MPEG_REG(ADPS_ALPHA0,
				adpscl_alpha[1]<<16 | adpscl_alpha[0]);
		VSYNC_WR_MPEG_REG(ADPS_ALPHA1, adpscl_shift[0] << 24 |
							adpscl_shift[1] << 20 |
							adpscl_shift[2] << 16 |
							adpscl_alpha[2]);
	    VSYNC_WR_MPEG_REG(ADPS_BETA0,
			adpscl_beta_s[0] << 20 | adpscl_beta[0]);
	    VSYNC_WR_MPEG_REG(ADPS_BETA1,
			adpscl_beta_s[1] << 20 | adpscl_beta[1]);
	    VSYNC_WR_MPEG_REG(ADPS_BETA2,
			adpscl_beta_s[2] << 20 | adpscl_beta[2]);
	    VSYNC_WR_MPEG_REG(ADPS_COEF0,
			adpscl_ys_coef[1] << 16 | adpscl_ys_coef[0]);
	    VSYNC_WR_MPEG_REG(ADPS_COEF1, adpscl_ys_coef[2]);

	} else if (mtx_sel & HDR_OUT_MTX) {
		if (hdr_mtx_param->mtx_out) {
			for (i = 0; i < 15; i++)
				mtx[i] = hdr_mtx_param->mtx_out[i];
		}

		VSYNC_WR_MPEG_REG(CGAIN_OFFT,
			(rgb2yuvpos[2] << 16) | rgb2yuvpos[1]);
		VSYNC_WR_MPEG_REG(MATRIXO_EN_CTRL, hdr_mtx_param->mtx_on);
		/*yuv in*/
		VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, hdr_mtx_param->mtx_on, 4, 1);

		VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, hdr_mtx_param->mtx_only,
			16, 1);
		VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, 0, 17, 1);
		/*mtx out en*/
		VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, 1, 15, 1);

		VSYNC_WR_MPEG_REG(MATRIXO_COEF00_01,
			(mtx[0 * 3 + 0] << 16) | (mtx[0 * 3 + 1] & 0x1FFF));
		VSYNC_WR_MPEG_REG(MATRIXO_COEF02_10,
			(mtx[0 * 3 + 2] << 16) | (mtx[1 * 3 + 0] & 0x1FFF));
		VSYNC_WR_MPEG_REG(MATRIXO_COEF11_12,
			(mtx[1 * 3 + 1] << 16) | (mtx[1 * 3 + 2] & 0x1FFF));
		VSYNC_WR_MPEG_REG(MATRIXO_COEF20_21,
			(mtx[2 * 3 + 0] << 16) | (mtx[2 * 3 + 1] & 0x1FFF));
		VSYNC_WR_MPEG_REG(MATRIXO_COEF22,
			mtx[2 * 3 + 2]);
		VSYNC_WR_MPEG_REG(MATRIXO_OFFSET0_1,
			(rgb2yuvpos[0] << 16) | (rgb2yuvpos[1]&0xFFF));
		VSYNC_WR_MPEG_REG(MATRIXO_OFFSET2,
			rgb2yuvpos[2]);
		VSYNC_WR_MPEG_REG(MATRIXO_PRE_OFFSET0_1,
			(rgb2yuvpre[0] << 16)|(rgb2yuvpre[1]&0xFFF));
		VSYNC_WR_MPEG_REG(MATRIXO_PRE_OFFSET2,
			rgb2yuvpre[2]);
	}

}

void set_eotf_lut(
	enum hdr_module_sel module_sel,
	struct hdr_proc_lut_param_s *hdr_lut_param)
{
	unsigned int lut[HDR2_EOTF_LUT_SIZE];
	unsigned int eotf_lut_addr_port = 0;
	unsigned int eotf_lut_data_port = 0;
	unsigned int hdr_ctrl = 0;
	unsigned int i = 0;

	if (module_sel & VD1_HDR) {
		eotf_lut_addr_port = VD1_EOTF_LUT_ADDR_PORT;
		eotf_lut_data_port = VD1_EOTF_LUT_DATA_PORT;
		hdr_ctrl = VD1_HDR2_CTRL;
	} else if (module_sel & VD2_HDR) {
		eotf_lut_addr_port = VD2_EOTF_LUT_ADDR_PORT;
		eotf_lut_data_port = VD2_EOTF_LUT_DATA_PORT;
		hdr_ctrl = VD2_HDR2_CTRL;
	} else if (module_sel & OSD1_HDR) {
		eotf_lut_addr_port = OSD1_EOTF_LUT_ADDR_PORT;
		eotf_lut_data_port = OSD1_EOTF_LUT_DATA_PORT;
		hdr_ctrl = OSD1_HDR2_CTRL;
	} else if (module_sel & DI_HDR) {
		eotf_lut_addr_port = DI_EOTF_LUT_ADDR_PORT;
		eotf_lut_data_port = DI_EOTF_LUT_DATA_PORT;
		hdr_ctrl = DI_HDR2_CTRL;
	} else if (module_sel & VDIN0_HDR) {
		eotf_lut_addr_port = VDIN0_EOTF_LUT_ADDR_PORT;
		eotf_lut_data_port = VDIN0_EOTF_LUT_DATA_PORT;
		hdr_ctrl = VDIN0_HDR2_CTRL;
	} else if (module_sel & VDIN1_HDR) {
		eotf_lut_addr_port = VDIN1_EOTF_LUT_ADDR_PORT;
		eotf_lut_data_port = VDIN1_EOTF_LUT_DATA_PORT;
		hdr_ctrl = VDIN1_HDR2_CTRL;
	}

	for (i = 0; i < HDR2_EOTF_LUT_SIZE; i++)
		lut[i] = hdr_lut_param->eotf_lut[i];

	VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, hdr_lut_param->lut_on, 3, 1);

	if (!hdr_lut_param->lut_on)
		return;

	VSYNC_WR_MPEG_REG(eotf_lut_addr_port, 0x0);
	for (i = 0; i < HDR2_EOTF_LUT_SIZE; i++)
		VSYNC_WR_MPEG_REG(eotf_lut_data_port, lut[i]);
}

void set_ootf_lut(
	enum hdr_module_sel module_sel,
	struct hdr_proc_lut_param_s *hdr_lut_param)
{
	unsigned int lut[HDR2_OOTF_LUT_SIZE];
	unsigned int ootf_lut_addr_port = 0;
	unsigned int ootf_lut_data_port = 0;
	unsigned int hdr_ctrl = 0;
	unsigned int i = 0;

	if (module_sel & VD1_HDR) {
		ootf_lut_addr_port = VD1_OGAIN_LUT_ADDR_PORT;
		ootf_lut_data_port = VD1_OGAIN_LUT_DATA_PORT;
		hdr_ctrl = VD1_HDR2_CTRL;
	} else if (module_sel & VD2_HDR) {
		ootf_lut_addr_port = VD2_OGAIN_LUT_ADDR_PORT;
		ootf_lut_data_port = VD2_OGAIN_LUT_DATA_PORT;
		hdr_ctrl = VD2_HDR2_CTRL;
	} else if (module_sel & OSD1_HDR) {
		ootf_lut_addr_port = OSD1_OGAIN_LUT_ADDR_PORT;
		ootf_lut_data_port = OSD1_OGAIN_LUT_DATA_PORT;
		hdr_ctrl = OSD1_HDR2_CTRL;
	} else if (module_sel & DI_HDR) {
		ootf_lut_addr_port = DI_OGAIN_LUT_ADDR_PORT;
		ootf_lut_data_port = DI_OGAIN_LUT_DATA_PORT;
		hdr_ctrl = DI_HDR2_CTRL;
	} else if (module_sel & VDIN0_HDR) {
		ootf_lut_addr_port = VDIN0_OGAIN_LUT_ADDR_PORT;
		ootf_lut_data_port = VDIN0_OGAIN_LUT_DATA_PORT;
		hdr_ctrl = VDIN0_HDR2_CTRL;
	} else if (module_sel & VDIN1_HDR) {
		ootf_lut_addr_port = VDIN1_OGAIN_LUT_ADDR_PORT;
		ootf_lut_data_port = VDIN1_OGAIN_LUT_DATA_PORT;
		hdr_ctrl = VDIN1_HDR2_CTRL;
	}

	for (i = 0; i < HDR2_OOTF_LUT_SIZE; i++)
		lut[i] = hdr_lut_param->ogain_lut[i];

	VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, hdr_lut_param->lut_on, 1, 1);

	if (!hdr_lut_param->lut_on)
		return;

	VSYNC_WR_MPEG_REG(ootf_lut_addr_port, 0x0);
	for (i = 0; i < HDR2_OOTF_LUT_SIZE / 2; i++)
		VSYNC_WR_MPEG_REG(ootf_lut_data_port,
			(lut[i * 2 + 1] << 16) +
			lut[i * 2]);
	VSYNC_WR_MPEG_REG(ootf_lut_data_port, lut[148]);
}

void set_oetf_lut(
	enum hdr_module_sel module_sel,
	struct hdr_proc_lut_param_s *hdr_lut_param)
{
	unsigned int lut[HDR2_OETF_LUT_SIZE];
	unsigned int oetf_lut_addr_port = 0;
	unsigned int oetf_lut_data_port = 0;
	unsigned int hdr_ctrl = 0;
	unsigned int i = 0;

	if (module_sel & VD1_HDR) {
		oetf_lut_addr_port = VD1_OETF_LUT_ADDR_PORT;
		oetf_lut_data_port = VD1_OETF_LUT_DATA_PORT;
		hdr_ctrl = VD1_HDR2_CTRL;
	} else if (module_sel & VD2_HDR) {
		oetf_lut_addr_port = VD2_OETF_LUT_ADDR_PORT;
		oetf_lut_data_port = VD2_OETF_LUT_DATA_PORT;
		hdr_ctrl = VD2_HDR2_CTRL;
	} else if (module_sel & OSD1_HDR) {
		oetf_lut_addr_port = OSD1_OETF_LUT_ADDR_PORT;
		oetf_lut_data_port = OSD1_OETF_LUT_DATA_PORT;
		hdr_ctrl = OSD1_HDR2_CTRL;
	} else if (module_sel & DI_HDR) {
		oetf_lut_addr_port = DI_OETF_LUT_ADDR_PORT;
		oetf_lut_data_port = DI_OETF_LUT_DATA_PORT;
		hdr_ctrl = DI_HDR2_CTRL;
	} else if (module_sel & VDIN0_HDR) {
		oetf_lut_addr_port = VDIN0_OETF_LUT_ADDR_PORT;
		oetf_lut_data_port = VDIN0_OETF_LUT_DATA_PORT;
		hdr_ctrl = VDIN0_HDR2_CTRL;
	} else if (module_sel & VDIN1_HDR) {
		oetf_lut_addr_port = VDIN1_OETF_LUT_ADDR_PORT;
		oetf_lut_data_port = VDIN1_OETF_LUT_DATA_PORT;
		hdr_ctrl = VDIN1_HDR2_CTRL;
	}

	for (i = 0; i < HDR2_OETF_LUT_SIZE; i++)
		lut[i] = hdr_lut_param->oetf_lut[i];

	VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, hdr_lut_param->lut_on, 2, 1);

	if (!hdr_lut_param->lut_on)
		return;

	VSYNC_WR_MPEG_REG(oetf_lut_addr_port, 0x0);
	for (i = 0; i < HDR2_OETF_LUT_SIZE / 2; i++) {
		if (hdr_lut_param->bitdepth == 10)
			VSYNC_WR_MPEG_REG(oetf_lut_data_port,
				((lut[i * 2 + 1] >> 2) << 16) +
				(lut[i * 2] >> 2));
		else
			VSYNC_WR_MPEG_REG(oetf_lut_data_port,
				(lut[i * 2 + 1] << 16) +
				lut[i * 2]);
		}
		VSYNC_WR_MPEG_REG(oetf_lut_data_port, lut[148]);
}

void set_c_gain(
	enum hdr_module_sel module_sel,
	struct hdr_proc_lut_param_s *hdr_lut_param)
{
	unsigned int lut[HDR2_CGAIN_LUT_SIZE];
	unsigned int cgain_lut_addr_port = 0;
	unsigned int cgain_lut_data_port = 0;
	unsigned int hdr_ctrl = 0;
	unsigned int i = 0;

	if (module_sel & VD1_HDR) {
		cgain_lut_addr_port = VD1_CGAIN_LUT_ADDR_PORT;
		cgain_lut_data_port = VD1_CGAIN_LUT_DATA_PORT;
		hdr_ctrl = VD1_HDR2_CTRL;
	} else if (module_sel & VD2_HDR) {
		cgain_lut_addr_port = VD2_CGAIN_LUT_ADDR_PORT;
		cgain_lut_data_port = VD2_CGAIN_LUT_DATA_PORT;
		hdr_ctrl = VD2_HDR2_CTRL;
	} else if (module_sel & OSD1_HDR) {
		cgain_lut_addr_port = OSD1_CGAIN_LUT_ADDR_PORT;
		cgain_lut_data_port = OSD1_CGAIN_LUT_DATA_PORT;
		hdr_ctrl = OSD1_HDR2_CTRL;
	} else if (module_sel & DI_HDR) {
		cgain_lut_addr_port = DI_CGAIN_LUT_ADDR_PORT;
		cgain_lut_data_port = DI_CGAIN_LUT_DATA_PORT;
		hdr_ctrl = DI_HDR2_CTRL;
	} else if (module_sel & VDIN0_HDR) {
		cgain_lut_addr_port = VDIN0_CGAIN_LUT_ADDR_PORT;
		cgain_lut_data_port = VDIN0_CGAIN_LUT_DATA_PORT;
		hdr_ctrl = VDIN0_HDR2_CTRL;
	} else if (module_sel & VDIN1_HDR) {
		cgain_lut_addr_port = VDIN1_CGAIN_LUT_ADDR_PORT;
		cgain_lut_data_port = VDIN1_CGAIN_LUT_DATA_PORT;
		hdr_ctrl = VDIN1_HDR2_CTRL;
	}

	for (i = 0; i < HDR2_CGAIN_LUT_SIZE; i++)
		lut[i] = hdr_lut_param->cgain_lut[i];

	/*cgain mode: 0->y domin*/
	/*cgain mode: 1->rgb domin, use r/g/b max*/
	VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, hdr_lut_param->lut_on, 12, 1);
	VSYNC_WR_MPEG_REG_BITS(hdr_ctrl, hdr_lut_param->lut_on, 0, 1);

	if (!hdr_lut_param->lut_on)
		return;

	VSYNC_WR_MPEG_REG(cgain_lut_addr_port, 0x0);
	for (i = 0; i < HDR2_CGAIN_LUT_SIZE / 2; i++)
		VSYNC_WR_MPEG_REG(cgain_lut_data_port,
			(lut[i * 2 + 1] << 16) + lut[i * 2]);
	VSYNC_WR_MPEG_REG(cgain_lut_data_port, lut[64]);
}

struct hdr_proc_lut_param_s hdr_lut_param;

void hdr_func(enum hdr_module_sel module_sel,
	enum hdr_process_sel hdr_process_select)
{
	int bit_depth;
	unsigned int i = 0;
	struct hdr_proc_mtx_param_s hdr_mtx_param;

	memset(&hdr_mtx_param, 0, sizeof(struct hdr_proc_mtx_param_s));
	memset(&hdr_lut_param, 0, sizeof(struct hdr_proc_lut_param_s));

	if (module_sel & (VD1_HDR | VD2_HDR | OSD1_HDR))
		bit_depth = 12;
	else if (module_sel & (VDIN0_HDR | VDIN1_HDR | DI_HDR))
		bit_depth = 10;
	else
		return;

#ifdef HDR2_MODULE
	MenuFun fun[] = {pq_eotf, pq_oetf, gm_eotf, gm_oetf,
		sld_eotf, sld_oetf, hlg_eotf, hlg_oetf, ootf_gain,
		nolinear_cgain, hlg_gain};

	if (hdr_process_select & HDR_BYPASS) {
		/*lut parameters*/
		eotf_float_gen(hdr_lut_param.eotf_lut, fun[2]);
		oetf_float_gen(hdr_lut_param.oetf_lut, fun[1]);
		oetf_float_gen(hdr_lut_param.ogain_lut, fun[8]);
		nolinear_lut_gen(hdr_lut_param.cgain_lut, fun[9]);
		hdr_lut_param.lut_on = LUT_OFF;
		hdr_lut_param.bitdepth = bit_depth;
	} else if (hdr_process_select & HDR_SDR) {
		/*lut parameters*/
		eotf_float_gen(hdr_lut_param.eotf_lut, fun[2]);
		oetf_float_gen(hdr_lut_param.oetf_lut, fun[1]);
		oetf_float_gen(hdr_lut_param.ogain_lut, fun[8]);
		nolinear_lut_gen(hdr_lut_param.cgain_lut, fun[9]);
		hdr_lut_param.lut_on = LUT_ON;
		hdr_lut_param.bitdepth = bit_depth;
	} else if (hdr_process_select & SDR_HDR) {
		/*lut parameters*/
		eotf_float_gen(hdr_lut_param.eotf_lut, fun[2]);
		oetf_float_gen(hdr_lut_param.oetf_lut, fun[1]);
		oetf_float_gen(hdr_lut_param.ogain_lut, fun[8]);
		nolinear_lut_gen(hdr_lut_param.cgain_lut, fun[9]);
		hdr_lut_param.lut_on = LUT_ON;
		hdr_lut_param.bitdepth = bit_depth;
	} else
		return;
#else
	/*lut parameters*/
	if (hdr_process_select & HDR_BYPASS) {
		for (i = 0; i < HDR2_OETF_LUT_SIZE; i++) {
			hdr_lut_param.oetf_lut[i]  = oe_y_lut_sdr[i];
			hdr_lut_param.ogain_lut[i] = oo_y_lut_hdr_sdr[i];
			if (i < HDR2_EOTF_LUT_SIZE)
				hdr_lut_param.eotf_lut[i] = eo_y_lut_hdr[i];
			if (i < HDR2_CGAIN_LUT_SIZE)
				hdr_lut_param.cgain_lut[i] = cgain_lut1[i] - 1;
		}
		hdr_lut_param.lut_on = LUT_OFF;
		hdr_lut_param.bitdepth = bit_depth;
	} else if (hdr_process_select & HDR_SDR) {
		for (i = 0; i < HDR2_OETF_LUT_SIZE; i++) {
			hdr_lut_param.oetf_lut[i]  = oe_y_lut_sdr[i];
			hdr_lut_param.ogain_lut[i] = oo_y_lut_hdr_sdr[i];
			if (i < HDR2_EOTF_LUT_SIZE)
				hdr_lut_param.eotf_lut[i] = eo_y_lut_hdr[i];
			if (i < HDR2_CGAIN_LUT_SIZE)
				hdr_lut_param.cgain_lut[i] = cgain_lut1[i] - 1;
		}
		hdr_lut_param.lut_on = LUT_ON;
		hdr_lut_param.bitdepth = bit_depth;
	} else if (hdr_process_select & SDR_HDR) {
		for (i = 0; i < HDR2_OETF_LUT_SIZE; i++) {
			hdr_lut_param.oetf_lut[i]  = oe_y_lut_hdr[i];
			hdr_lut_param.ogain_lut[i] = oo_y_lut_sdr_hdr[i];
			if (i < HDR2_EOTF_LUT_SIZE)
				hdr_lut_param.eotf_lut[i] = eo_y_lut_sdr[i];
			if (i < HDR2_CGAIN_LUT_SIZE)
				hdr_lut_param.cgain_lut[i] = cgain_lut0[i] - 1;
		}
		hdr_lut_param.lut_on = LUT_ON;
		hdr_lut_param.bitdepth = bit_depth;
	} else if (hdr_process_select & HLG_BYPASS) {
		for (i = 0; i < HDR2_OETF_LUT_SIZE; i++) {
			hdr_lut_param.oetf_lut[i]  = oe_y_lut_sdr[i];
			hdr_lut_param.ogain_lut[i] = oo_y_lut_hdr_sdr[i];
			if (i < HDR2_EOTF_LUT_SIZE)
				hdr_lut_param.eotf_lut[i] = eo_y_lut_hlg[i];
			if (i < HDR2_CGAIN_LUT_SIZE)
				hdr_lut_param.cgain_lut[i] = cgain_lut1[i] - 1;
		}
		hdr_lut_param.lut_on = LUT_OFF;
		hdr_lut_param.bitdepth = bit_depth;
	} else if (hdr_process_select & HLG_SDR) {
		for (i = 0; i < HDR2_OETF_LUT_SIZE; i++) {
			hdr_lut_param.oetf_lut[i]  = oe_y_lut_sdr[i];
			hdr_lut_param.ogain_lut[i] = oo_y_lut_hdr_sdr[i];
			if (i < HDR2_EOTF_LUT_SIZE)
				hdr_lut_param.eotf_lut[i] = eo_y_lut_hlg[i];
			if (i < HDR2_CGAIN_LUT_SIZE)
				hdr_lut_param.cgain_lut[i] = cgain_lut1[i] - 1;
		}
		hdr_lut_param.lut_on = LUT_ON;
		hdr_lut_param.bitdepth = bit_depth;
	} else if (hdr_process_select & HLG_HDR) {
		for (i = 0; i < HDR2_OETF_LUT_SIZE; i++) {
			hdr_lut_param.oetf_lut[i]  = oe_y_lut_hdr[i];
			hdr_lut_param.ogain_lut[i] = oo_y_lut_hlg_hdr[i];
			if (i < HDR2_EOTF_LUT_SIZE)
				hdr_lut_param.eotf_lut[i] = eo_y_lut_hlg[i];
			if (i < HDR2_CGAIN_LUT_SIZE)
				hdr_lut_param.cgain_lut[i] = cgain_lut1[i] - 1;
		}
		hdr_lut_param.lut_on = LUT_ON;
		hdr_lut_param.bitdepth = bit_depth;
	} else
		return;
#endif
	/*mtx parameters*/
	if (hdr_process_select & (HDR_BYPASS | HLG_BYPASS)) {
		hdr_mtx_param.mtx_only = HDR_ONLY;
		for (i = 0; i < 15; i++) {
			hdr_mtx_param.mtx_in[i] = bypass_coeff[i];
			hdr_mtx_param.mtx_cgain[i] = bypass_coeff[i];
			hdr_mtx_param.mtx_ogain[i] = bypass_coeff[i];
			hdr_mtx_param.mtx_out[i] = bypass_coeff[i];
			if (i < 9)
				hdr_mtx_param.mtx_gamut[i] = bypass_coeff[i];
		}
		hdr_mtx_param.mtx_on = MTX_OFF;
		hdr_mtx_param.p_sel = HDR_BYPASS;
	} else if (hdr_process_select & (HDR_SDR | HLG_SDR)) {
		hdr_mtx_param.mtx_only = HDR_ONLY;
		for (i = 0; i < 15; i++) {
			hdr_mtx_param.mtx_in[i] = ycbcr2rgb_ncl2020[i];
			hdr_mtx_param.mtx_cgain[i] = rgb2ycbcr_709[i];
			hdr_mtx_param.mtx_ogain[i] = rgb2ycbcr_ncl2020[i];
			hdr_mtx_param.mtx_out[i] = rgb2ycbcr_709[i];
			if (i < 9)
				hdr_mtx_param.mtx_gamut[i] = ncl_2020_709[i];
		}
		hdr_mtx_param.mtx_on = MTX_ON;
		hdr_mtx_param.p_sel = HDR_SDR;
	} else if (hdr_process_select & SDR_HDR) {
		hdr_mtx_param.mtx_only = HDR_ONLY;
		for (i = 0; i < 15; i++) {
			hdr_mtx_param.mtx_in[i] = ycbcr2rgb_709[i];
			hdr_mtx_param.mtx_cgain[i] = rgb2ycbcr_ncl2020[i];
			hdr_mtx_param.mtx_ogain[i] = rgb2ycbcr_709[i];
			hdr_mtx_param.mtx_out[i] = rgb2ycbcr_ncl2020[i];
			if (i < 9)
				hdr_mtx_param.mtx_gamut[i] = ncl_709_2020[i];
		}
		hdr_mtx_param.mtx_on = MTX_ON;
		hdr_mtx_param.p_sel = SDR_HDR;
	} else if (hdr_process_select & HLG_HDR) {
		hdr_mtx_param.mtx_only = HDR_ONLY;
		for (i = 0; i < 15; i++) {
			hdr_mtx_param.mtx_in[i] = ycbcr2rgb_ncl2020[i];
			hdr_mtx_param.mtx_cgain[i] = bypass_coeff[i];
			hdr_mtx_param.mtx_ogain[i] = bypass_coeff[i];
			hdr_mtx_param.mtx_out[i] = rgb2ycbcr_ncl2020[i];
			if (i < 9)
				hdr_mtx_param.mtx_gamut[i] = bypass_coeff[i];
		}
		hdr_mtx_param.mtx_on = MTX_ON;
		hdr_mtx_param.p_sel = HLG_HDR;
	}

	set_hdr_matrix(module_sel, HDR_IN_MTX, &hdr_mtx_param);

	set_eotf_lut(module_sel, &hdr_lut_param);

	set_hdr_matrix(module_sel, HDR_GAMUT_MTX, &hdr_mtx_param);

	set_ootf_lut(module_sel, &hdr_lut_param);

	set_oetf_lut(module_sel, &hdr_lut_param);

	set_hdr_matrix(module_sel, HDR_OUT_MTX, &hdr_mtx_param);

	set_c_gain(module_sel, &hdr_lut_param);
}

/*G12A matrix setting*/
void mtx_setting(enum vpp_matrix_e mtx_sel,
	enum mtx_csc_e mtx_csc,
	int mtx_on)
{
	unsigned int matrix_coef00_01 = 0;
	unsigned int matrix_coef02_10 = 0;
	unsigned int matrix_coef11_12 = 0;
	unsigned int matrix_coef20_21 = 0;
	unsigned int matrix_coef22 = 0;
	unsigned int matrix_coef13_14 = 0;
	unsigned int matrix_coef23_24 = 0;
	unsigned int matrix_coef15_25 = 0;
	unsigned int matrix_clip = 0;
	unsigned int matrix_offset0_1 = 0;
	unsigned int matrix_offset2 = 0;
	unsigned int matrix_pre_offset0_1 = 0;
	unsigned int matrix_pre_offset2 = 0;
	unsigned int matrix_en_ctrl = 0;

	if (mtx_sel & VD1_MTX) {
		matrix_coef00_01 = VPP_VD1_MATRIX_COEF00_01;
		matrix_coef02_10 = VPP_VD1_MATRIX_COEF02_10;
		matrix_coef11_12 = VPP_VD1_MATRIX_COEF11_12;
		matrix_coef20_21 = VPP_VD1_MATRIX_COEF20_21;
		matrix_coef22 = VPP_VD1_MATRIX_COEF22;
		matrix_coef13_14 = VPP_VD1_MATRIX_COEF13_14;
		matrix_coef23_24 = VPP_VD1_MATRIX_COEF23_24;
		matrix_coef15_25 = VPP_VD1_MATRIX_COEF15_25;
		matrix_clip = VPP_VD1_MATRIX_CLIP;
		matrix_offset0_1 = VPP_VD1_MATRIX_OFFSET0_1;
		matrix_offset2 = VPP_VD1_MATRIX_OFFSET2;
		matrix_pre_offset0_1 = VPP_VD1_MATRIX_PRE_OFFSET0_1;
		matrix_pre_offset2 = VPP_VD1_MATRIX_PRE_OFFSET2;
		matrix_en_ctrl = VPP_VD1_MATRIX_EN_CTRL;

		WRITE_VPP_REG_BITS(VPP_VD1_MATRIX_EN_CTRL, mtx_on, 0, 1);
	} else if (mtx_sel & POST2_MTX) {
		matrix_coef00_01 = VPP_POST2_MATRIX_COEF00_01;
		matrix_coef02_10 = VPP_POST2_MATRIX_COEF02_10;
		matrix_coef11_12 = VPP_POST2_MATRIX_COEF11_12;
		matrix_coef20_21 = VPP_POST2_MATRIX_COEF20_21;
		matrix_coef22 = VPP_POST2_MATRIX_COEF22;
		matrix_coef13_14 = VPP_POST2_MATRIX_COEF13_14;
		matrix_coef23_24 = VPP_POST2_MATRIX_COEF23_24;
		matrix_coef15_25 = VPP_POST2_MATRIX_COEF15_25;
		matrix_clip = VPP_POST2_MATRIX_CLIP;
		matrix_offset0_1 = VPP_POST2_MATRIX_OFFSET0_1;
		matrix_offset2 = VPP_POST2_MATRIX_OFFSET2;
		matrix_pre_offset0_1 = VPP_POST2_MATRIX_PRE_OFFSET0_1;
		matrix_pre_offset2 = VPP_POST2_MATRIX_PRE_OFFSET2;
		matrix_en_ctrl = VPP_POST2_MATRIX_EN_CTRL;

		WRITE_VPP_REG_BITS(VPP_POST2_MATRIX_EN_CTRL, mtx_on, 0, 1);
	} else if (mtx_sel & POST_MTX) {
		matrix_coef00_01 = VPP_POST_MATRIX_COEF00_01;
		matrix_coef02_10 = VPP_POST_MATRIX_COEF02_10;
		matrix_coef11_12 = VPP_POST_MATRIX_COEF11_12;
		matrix_coef20_21 = VPP_POST_MATRIX_COEF20_21;
		matrix_coef22 = VPP_POST_MATRIX_COEF22;
		matrix_coef13_14 = VPP_POST_MATRIX_COEF13_14;
		matrix_coef23_24 = VPP_POST_MATRIX_COEF23_24;
		matrix_coef15_25 = VPP_POST_MATRIX_COEF15_25;
		matrix_clip = VPP_POST_MATRIX_CLIP;
		matrix_offset0_1 = VPP_POST_MATRIX_OFFSET0_1;
		matrix_offset2 = VPP_POST_MATRIX_OFFSET2;
		matrix_pre_offset0_1 = VPP_POST_MATRIX_PRE_OFFSET0_1;
		matrix_pre_offset2 = VPP_POST_MATRIX_PRE_OFFSET2;
		matrix_en_ctrl = VPP_POST_MATRIX_EN_CTRL;

		WRITE_VPP_REG_BITS(VPP_POST_MATRIX_EN_CTRL, mtx_on, 0, 1);
	}

	if (!mtx_on)
		return;

	switch (mtx_csc) {
	case MATRIX_RGB_YUV709:
		WRITE_VPP_REG(matrix_coef00_01, 0x00bb0275);
		WRITE_VPP_REG(matrix_coef02_10, 0x003f1f99);
		WRITE_VPP_REG(matrix_coef11_12, 0x1ea601c2);
		WRITE_VPP_REG(matrix_coef20_21, 0x01c21e67);
		WRITE_VPP_REG(matrix_coef22, 0x00001fd7);
		WRITE_VPP_REG(matrix_offset0_1, 0x00400200);
		WRITE_VPP_REG(matrix_offset2, 0x00000200);
		WRITE_VPP_REG(matrix_pre_offset0_1, 0x0);
		WRITE_VPP_REG(matrix_pre_offset2, 0x0);
		break;
	case MATRIX_YUV709_RGB:
		WRITE_VPP_REG(matrix_coef00_01, 0x04A80000);
		WRITE_VPP_REG(matrix_coef02_10, 0x072C04A8);
		WRITE_VPP_REG(matrix_coef11_12, 0x1F261DDD);
		WRITE_VPP_REG(matrix_coef20_21, 0x04A80876);
		WRITE_VPP_REG(matrix_coef22, 0x0);
		WRITE_VPP_REG(matrix_offset0_1, 0x0);
		WRITE_VPP_REG(matrix_offset2, 0x0);
		WRITE_VPP_REG(matrix_pre_offset0_1, 0x7c00600);
		WRITE_VPP_REG(matrix_pre_offset2, 0x00000600);
		break;
	default:
		break;
	}
}
