/* sonos_mount.c.inc
 *
 * Copyright (c) 2015-2021, Sonos, Inc.
 * SPDX-License-Identifier:	GPL-2.0
 *
 * Logically, and when built, this is part of init/do_mounts.c.  For sustaining,
 * and ease of kernel-kernel porting, it is maintained in a separate source code
 * module.  Initial implementation was for linux 3.10 and imx6 Solo-X SoC.
 *
 * There may be some Solo-X assumptions built-in to this module that will require
 * changes on other SoCs.
 */

#include <linux/inet.h>

#include <linux/crypto.h>
#include <linux/scatterlist.h>

#include <sonos/firmware_allowlist.h>
#include "sonos_digest_data.h"
#include <linux/device-mapper.h>
#include <linux/fcntl.h>
#include "sonos_lock.h"
#include <linux/dm-ioctl.h>
#include <crypto/sonos_signature_common_linux.h>
#include <crypto/sonos_signature_keys.h>
#include <crypto/sonos_signature_verify_linux.h>
#include "mdp.h"
#include "sonos_unlock.h"
#include <linux/sonos_kernel.h>

extern struct manufacturing_data_page sys_mdp;
extern struct manufacturing_data_page3 sys_mdp3;

void sonos_rootfs_failure_notify(void);


rootfs_digest_t		rootfs_digest = {
	{ 0, 0 },
	0,
	0,
	{"Placeholder for computed digest" },
} ;


static int get_verified_serial(uint8_t *serial, size_t serialLen)
{
	int result = 0;
	SonosSignature *sig = NULL;

	if (serialLen != 6 ||
	    sys_mdp.mdp_magic != MDP_MAGIC ||
	    sys_mdp.mdp_version < MDP_VERSION_AUTH_FLAGS ||
	    !(sys_mdp.mdp_pages_present & MDP_PAGE3_PRESENT) ||
	    sys_mdp3.mdp3_magic != MDP_MAGIC3 ||
	    sys_mdp3.mdp3_version < MDP3_VERSION_SECURE_BOOT) {
		printk(KERN_ERR "bad MDP in get_verified_serial: %x %x %x %x %x\n",
		       sys_mdp.mdp_magic,
		       sys_mdp.mdp_version,
		       sys_mdp.mdp_pages_present,
		       sys_mdp3.mdp3_magic,
		       sys_mdp3.mdp3_version);
		return result;
	}

	sig = kmalloc(sizeof(*sig), GFP_KERNEL);
	if (sig && sonosUnlockVerifyCpuSerialSig(sys_mdp.mdp_serial, 6,
					   sys_mdp3.mdp3_cpuid_sig,
					   sizeof sys_mdp3.mdp3_cpuid_sig,
					   sig, sonosHash, sonosRawVerify,
					   sonosKeyLookup, "unit", NULL)) {
		printk(KERN_DEBUG "good cpuid/serial binding statement\n");
		memcpy(serial, sys_mdp.mdp_serial, serialLen);
		result = 1;
	}
	else {
		printk(KERN_ERR "bad cpuid/serial binding statement\n");
	}

	if (sig) {
		kfree(sig);
	}
	return result;
}

int validate_sonos_rootfs_digest(void);
extern int ath_nand_local_read(char *name, loff_t offset, size_t length, size_t *retleng, unsigned char *data);

#define SFW_GETCPUID sonos_get_cpuid
#define SFW_GETSERIAL get_verified_serial
#define SFW_PRINT printk
#define SFW_PLVL_INFO KERN_INFO
#define SFW_PLVL_EMERG KERN_EMERG
#define SFW_BE32_TO_CPU __be32_to_cpu

#include "sonos_fw_allowlist.c.inc"

#undef SFW_GETCPUID
#undef SFW_GETSERIAL
#undef SFW_PRINT
#undef SFW_PLVL_INFO
#undef SFW_PLVL_EMERG
#undef SFW_BE32_TO_CPU
#undef SFW_BE64_TO_CPU


static int __init do_mount_root(char *name, char *fs, int flags, void *data)
{
	struct super_block *s;

	int err;
	int encrypted_root = 0;

	sonosInitKeyTable();

	if (!check_sonos_firmware_allowlist(&SONOS_FIRMWARE_ALLOWLIST.header,
					    sizeof(SONOS_FIRMWARE_ALLOWLIST.x))) {
		BUG();
	}

	encrypted_root = validate_sonos_rootfs_digest();

	if ( encrypted_root ) {
		printk(KERN_INFO "sys_mount - %s, %s, %s, x%x, x%x\n","/dev/mapper/crroot","/root",fs,(int)flags,(int)data);
		err = sys_mount("/dev/mapper/crroot", "/root", fs, flags, data);
	}
	else {
		printk(KERN_INFO "sys_mount - %s, %s, %s, x%x, x%x\n",name,"/root",fs,(int)flags,(int)data);
		err = sys_mount(name, "/root", fs, flags, data);
	}
	if (err)
		return err;

	sys_chdir("/root");
	s = current->fs->pwd.dentry->d_sb;
	ROOT_DEV = s->s_dev;
	printk(KERN_INFO
	       "VFS: Mounted root (%s filesystem)%s on device %u:%u.\n",
	       s->s_type->name,
	       s->s_flags & MS_RDONLY ?  " readonly" : "",
	       MAJOR(ROOT_DEV), MINOR(ROOT_DEV));
	printk(KERN_INFO "Setting noexec and nodev modes to constrained...\n");
	if ( !is_mdp_authorized (MDP_AUTH_FLAG_EXEC_ENABLE) ) {
		printk(KERN_INFO "NOEXEC on all non-rootfs mounts\n");
		proc_noexec_set_constrained();
	}
	if ( !is_mdp_authorized (MDP_AUTH_FLAG_NODEV_CTRL) ) {
		printk(KERN_INFO "NODEV on all non-rootfs mounts\n");
		proc_nodev_set_constrained();
	}
	return 0;
}

#define ENCRYPTED	1
#define PLAINTEXT	0

#define SQUASHFS_LEN_OFFSET	40
#define CRAMFS_LEN_OFFSET	4

#define ROOTFS_CHUNK_SIZE	(8 * 1024 * 1024)


static int	handle_encrypted_rootfs( __s32*, char*, int* );

int validate_sonos_rootfs_digest() {
	unsigned char   *proot;
	__s32   length;
	char	rootfs_file[16];
	rootfs_digest_t	 digest;
	__u32   *presult, *pkern;
	size_t  count = 4;
	size_t  retlen;
	int	 *pData;
	loff_t  offset = (loff_t) 0;
	int	encrypted = 0;
	int	rfd = -1;
	struct  crypto_hash *tfm = NULL;

	strlcpy(rootfs_file, root_device_name, sizeof(rootfs_file));

	encrypted = handle_encrypted_rootfs(&length, rootfs_file, &rfd);

	presult = (__u32 *)&digest;
	*presult = 0;

	if ( !encrypted ) {
		printk(KERN_INFO "Assuming rootfs is not encrypted\n");
		
		length =  4;

		if( (proot = kmalloc(length, GFP_KERNEL)) == NULL) {
			printk(KERN_ERR "Could not get a buffer for the rootfs length check.\n");
			goto validation_fail;
		}

		pData = (int*)proot;

		if ( ath_nand_local_read(rootfs_file, offset, count, &retlen, proot) ) {
			printk(KERN_ERR "Could not read %s\n", rootfs_file);
			goto validation_fail;
		}

		if (*pData == SQUASHFS_MAGIC) {
			printk(KERN_INFO "validating squashfs root\n");
			offset = (loff_t) SQUASHFS_LEN_OFFSET;
		} else {
			printk(KERN_INFO "validating cramfs root\n");
			offset = (loff_t) CRAMFS_LEN_OFFSET;
		}

		if ( ath_nand_local_read(rootfs_file, offset, count, &retlen, proot) ) {
			printk(KERN_ERR "Could not read rootfs length\n");
			goto validation_fail;
		}
		if ( retlen != count ) {
			printk(KERN_ERR "rootfs length read length failure [x%x != x%x]\n",retlen,count);
			goto validation_fail;
		}

		length = *pData;

		if (offset == SQUASHFS_LEN_OFFSET) {
			length = (((length-1)>>12)+1)<<12;
		}

		if ( length != be32_to_cpu(rootfs_digest.rootfs_length)  ) {
			printk(KERN_ERR "Rootfs length (0x%x) invalid, expect 0x%x.\n", 
			       length, be32_to_cpu(rootfs_digest.rootfs_length));
			goto validation_fail;
		}

		if ( proot ) kfree(proot);
		count = length;
		offset = (loff_t)0;
	}


	if ( (proot = kmalloc(ROOTFS_CHUNK_SIZE, GFP_KERNEL)) == NULL ) {
		printk(KERN_ERR "Could not get a buffer for the rootfs digest check.\n");
		goto validation_fail;
	}

	printk(KERN_INFO "calculating the digest on %d byte rootfs\n", (int)length);
	{
		struct  scatterlist sg;
		struct  hash_desc desc;
		size_t	current_length;

		tfm = crypto_alloc_hash(ROOTFS_DIGEST_ALGORITHM, 0, CRYPTO_ALG_ASYNC);
		if(IS_ERR(tfm)) {
			printk(KERN_ERR "could not allocate crypto hash\n");
			goto validation_fail;
		}

		sg_init_one(&sg, proot, ROOTFS_CHUNK_SIZE);

		desc.tfm = tfm;
		desc.flags = 0;

		if (crypto_hash_init(&desc)) {
			printk(KERN_ERR "Error initializing rootfs hash\n");
			goto validation_fail;
		}

		while (length) {
			if (length < ROOTFS_CHUNK_SIZE) {
				current_length = length;
			} else {
				current_length = (size_t) ROOTFS_CHUNK_SIZE;
			}

			if (encrypted) {
				if(sys_read(rfd, proot, current_length) != current_length) {
					printk(KERN_ERR "Could not read rootfs\n");
					goto validation_fail;
				}
			} else {
				if ( ath_nand_local_read(rootfs_file, offset, current_length, &retlen, (unsigned char*)proot) ) {
					printk(KERN_ERR "Could not read rootfs from %s\n", rootfs_file);
					goto validation_fail;
				}
				offset += (loff_t) current_length;
			}

			if (crypto_hash_update(&desc, &sg, current_length)) {
				printk(KERN_ERR "Error calculating rootfs digest\n");
				goto validation_fail;
			}

			length = length - current_length;
		}
		if (crypto_hash_final(&desc, (u8*)&digest.digest_value)) {
			printk(KERN_ERR "Error finalizing rootfs digest\n");
			goto validation_fail;
		}
		crypto_free_hash(tfm);
		tfm = NULL;
	}
	kfree(proot);
	proot = NULL;
	if (rfd >= 0) {
		sys_close(rfd);
		rfd = -1;
	}

	pkern = (__u32 *)&rootfs_digest.digest_value;
	presult = (__u32 *)&digest.digest_value;
	if ( be32_to_cpu(rootfs_digest.digest_value_length) != SONOS_DIGEST_LENGTH ) {
		printk(KERN_ERR "Incorrect rootfs digest length\n");
		goto validation_fail;
	}

	if ( !memcmp(pkern, presult, be32_to_cpu(rootfs_digest.digest_value_length) ) ) {
		printk(KERN_INFO "checking %d byte digest (using %s)\n  calculated digest x%08x, kernel contained x%08x  - good rootfs\n",
				(int)be32_to_cpu(rootfs_digest.digest_value_length), ROOTFS_DIGEST_ALGORITHM, (int)*presult, (int)*pkern);
		return encrypted;
	}
	printk(KERN_ERR "calculated digest x%08x, kernel contained x%08x - MISMATCH ON rootfs DIGEST!!\n",(int)*presult, (int)*pkern);

validation_fail:
	printk(KERN_ERR "rootfs digest validation failed.  Booting halted.\n");
	sonos_rootfs_failure_notify();
	if ( proot ) kfree(proot);
	if (rfd >= 0) sys_close(rfd);
	if (tfm) crypto_free_hash(tfm);
	panic("rootfs digest validation failed.\n");
}

#if defined(SONOS_ARCH_BOOTLEG)
#define ROOTFS_MAX	(205*1024*1024)
#elif defined(SONOS_ARCH_ENCORE)
#define ROOTFS_MAX	(73*1024*1024)
#else
#define ROOTFS_MAX	(95*1024*1024)
#endif

extern int dm_dev_create(struct dm_ioctl *param, size_t param_size);
extern int dm_table_load(struct dm_ioctl *param, size_t param_size);
extern int dm_dev_suspend(struct dm_ioctl *param, size_t param_size);
extern int sonos_get_rootfs_key(int, char *);


static int handle_encrypted_rootfs( __s32 *buffer_length, char *rootfs_device, int *rfd )
{
	unsigned char   *proot = NULL;
	char	rootfs_file[24];
	char	encryption_check[8];
	__s32	length = 8;
	size_t  retlen;
	loff_t  offset = (loff_t)0;
	struct	dm_ioctl *dmi = NULL;
	struct  dm_target_spec	*dms;
	int	control;
	char	*parms;
	unsigned char parms_key[65];
	int	buffer[64];
	int	blocks = 0;
	int	key_type;
	int	dm_result;

	strlcpy(rootfs_file, rootfs_device, sizeof rootfs_file);


	if ( ath_nand_local_read(rootfs_file, offset, length, &retlen, encryption_check) ) {
		printk(KERN_ERR "Could not read rootfs length\n");
		goto dmcrypt_failed;
	}
	if ( retlen != length ) {
		printk(KERN_ERR "rootfs length read length failure [x%x != x%x]\n", retlen, length);
		goto dmcrypt_failed;
	}

	if ( !strncmp(encryption_check,"LUKS",4) ) {
		printk(KERN_INFO "LUKS container - encrypted rootfs\n");
	}
	else {
		printk(KERN_INFO "rootfs_device %s is not encrypted\n", rootfs_device);
		goto plaintext;
	}

	{
		int	tfd;
		char	rootfs_devpoint[24];
		int	device_bytes;

		snprintf(rootfs_devpoint, sizeof(rootfs_devpoint), "/dev/%s",rootfs_device);

		if( (tfd = sys_open(rootfs_devpoint,O_RDONLY,0777)) < 0 )
			printk(KERN_INFO "Could not open %s\n",rootfs_devpoint);
		else {
			device_bytes = sys_lseek(tfd,0,SEEK_END);
			blocks = (device_bytes / 512) - 4096;
			printk(KERN_INFO "end of %s is %d bytes, %d blocks\n",rootfs_devpoint, device_bytes,blocks);
			sys_close(tfd);
		}
	}

	{
#include "sect_upgrade_header.h"

		SECT_UPGRADE_HEADER	sect_header;
		int	kern_block_number;
		char	kern_file[24];

		sscanf(rootfs_file,"mtdblock%d",&kern_block_number);
		kern_block_number--;
		snprintf(kern_file, sizeof(kern_file),"mtdblock%d",kern_block_number);
		printk(KERN_INFO "try to read the kernel section header at %s\n",kern_file);

		if ( ath_nand_local_read(kern_file, offset, sizeof(sect_header), &retlen, (char*)&sect_header) ) {
			printk(KERN_ERR "Could not read kernel section header\n");
			goto dmcrypt_failed;
		}
		key_type = sect_header.rootfsFormat;
		printk(KERN_INFO "rootfs format=%d\n",key_type);
	}

	if ( sonos_get_rootfs_key(key_type, parms_key) != 1 ) {
		goto dmcrypt_failed;
	}



	if ( (control = sys_open("/dev/mapper/control",O_RDWR|O_CREAT,0777)) < 0 )
		printk(KERN_INFO "could not open /dev/mapper/control - %d\n", control);
	else
		printk(KERN_INFO "Opened /dev/mapper/control at descriptor %d\n",control);

	dmi = kmalloc(16384, GFP_KERNEL);

	dmi->version[0] =      4;
	dmi->version[1] =      24;
	dmi->version[2] =      0;
	dmi->data_size =       312;
	dmi->data_start =      312;
	dmi->open_count =      0;
	dmi->padding =         0;
	{
		__u32	dmid;
		for(dmid=0;dmid < 7;dmid++)
			dmi->data[dmid] = 0;
	}

	dmi->target_count =    0;
	dmi->flags =           4;
	dmi->event_nr =        0;
	dmi->dev =             0;
	strlcpy(dmi->name, "crroot", sizeof(dmi->name));
	dmi->uuid[0] = '\0';
	dm_result = dm_dev_create(dmi, 305);
	printk(KERN_INFO "dev_create returned %d\n", dm_result);

	dmi->target_count =    1;
	dmi->flags =           5;
	dmi->event_nr =        0;
	dmi->dev =             0;
	strlcpy(dmi->name, "crroot", sizeof(dmi->name));
	dmi->uuid[0] = '\0';

	dms = (struct dm_target_spec *)((u8 *)dmi + dmi->data_start);
	dms->sector_start =	0;
	dms->length =		blocks;
	dms->status =		0x0;
	dms->next =		0x98;
	strlcpy(dms->target_type, "crypt", sizeof(dms->target_type));
	parms = (char *)(dms + 1);
	sprintf(parms, "aes-cbc-essiv:sha256 %64s 0 /dev/%s 4096", parms_key, rootfs_file);
	dm_result = dm_table_load(dmi, 16384);
	printk(KERN_INFO "table_load returned %d\n", dm_result);

	dmi->target_count =    1;
	dmi->flags =           4;
	dmi->event_nr =        0x400000;
	dmi->dev =             0;
	strlcpy(dmi->name, "crroot", sizeof(dmi->name));
	dmi->uuid[0] = '\0';
	dm_result = dm_dev_suspend(dmi, 305);
	printk(KERN_INFO "dev_suspend returned %d\n", dm_result);

	kfree(dmi);
	dmi = NULL;

	*rfd = sys_open("/dev/mapper/crroot",O_RDONLY,0777);

	if ( *rfd < 0 ) {
		goto dmcrypt_failed;
	}

	sys_read(*rfd, (char*)buffer, 64);
	printk(KERN_INFO "seeking back to the beginning at descriptor %d\n", *rfd);
	sys_lseek(*rfd, 0, SEEK_SET);

	if (buffer[0] == SQUASHFS_MAGIC) {
		int calc_len = buffer[SQUASHFS_LEN_OFFSET/sizeof(int)];
		calc_len = (((calc_len-1)>>12)+1)<<12;
		printk(KERN_ERR "squashfs_bytes 0x%x - rootfs size 0x%x (%d) bytes\n", \
			buffer[SQUASHFS_LEN_OFFSET/sizeof(int)], calc_len, calc_len);
		*buffer_length = calc_len;
	} else {
		*buffer_length = buffer[CRAMFS_LEN_OFFSET/sizeof(int)];
	}

	if ( *buffer_length > ROOTFS_MAX || *buffer_length < 8 ) {
		goto dmcrypt_failed;
	}

	return (int) ENCRYPTED;

dmcrypt_failed:
	printk(KERN_INFO "dmcrypt failed\n");
plaintext:
	if (*rfd >= 0) {
		sys_close(*rfd);
		*rfd = -1;
	}
	if (proot) kfree(proot);
	if (dmi) kfree(dmi);

	return (int)PLAINTEXT;
}

