/*
 * Copyright (c) 2015-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:	GPL-2.0
 *
 * linux/fs/proc/sonos_lock.c
 *
 * /proc entry support to allow control over unlock authorization
 * functionality on secured Sonos products.
 */

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/module.h>
#include <asm/uaccess.h>
#include "mdp.h"
#include <linux/regmap.h>
#include <linux/mfd/syscon.h>
#include "sonos_rollback.h"
#define SONOS_LOCK_C
#include "sonos_lock.h"
#ifdef CONFIG_SONOS_SECBOOT
#include <sonos/firmware_allowlist.h>
#endif

#if defined(SONOS_ARCH_ATTR_SOC_IS_IMX6)
static int saved_state;
static int state_valid = 0;
#endif




static int noexec_mode = NORMAL_MODE;

int sonos_forbid_execution(void)
{
	return ( (noexec_mode == CONSTRAINED_MODE) ? 1 : 0 );
}
EXPORT_SYMBOL(sonos_forbid_execution);
int sonos_allow_execution(void)
{
	return ( (noexec_mode == UNCONSTRAINED_MODE) ? 1 : 0 );
}
EXPORT_SYMBOL(sonos_allow_execution);
void proc_noexec_set_constrained(void)
{
	noexec_mode = CONSTRAINED_MODE;
}
EXPORT_SYMBOL(proc_noexec_set_constrained);

static int noexec_ctrl_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "%d\n", noexec_mode );
	return 0;
}

static int noexec_ctrl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, noexec_ctrl_proc_show, NULL);
}

static int noexec_ctrl_write_data(struct file *file, const char __user * buf, size_t length, loff_t * offset)
{
	char	buffer[64];

        memset(buffer, 0, sizeof(buffer));
        if (length > sizeof(buffer) - 1)
                length = sizeof(buffer) - 1;
        if (copy_from_user(buffer, buf, length))
                return -EFAULT;
	if  ( !strncmp((char*)buffer, "constrained", strlen("constrained")) ) {
		noexec_mode = CONSTRAINED_MODE;
	} else if ( !strncmp((char*)buffer, "normal", strlen("normal")) ) {
		if ( is_mdp_authorized(MDP_AUTH_FLAG_EXEC_ENABLE) || noexec_mode >= NORMAL_MODE ) {
			noexec_mode = NORMAL_MODE;
		} else {
			printk(KERN_INFO "Operation not supported on locked device.\n");
		}
	} else if  ( !strncmp((char*)buffer, "unconstrained", strlen("unconstrained")) ) {
		if ( is_mdp_authorized(MDP_AUTH_FLAG_EXEC_ENABLE) || noexec_mode >= UNCONSTRAINED_MODE ) {
			noexec_mode = UNCONSTRAINED_MODE;
		} else {
			printk(KERN_INFO "Operation not supported on locked device.\n");
		}
	} else {
		printk(KERN_INFO "Unsupported operation.\n");
	}

	printk(KERN_INFO "%d\n", noexec_mode);
	return length;
}

static const struct file_operations noexec_ctrl_proc_fops = {
	.open		= noexec_ctrl_proc_open,
	.read		= seq_read,
	.write		= noexec_ctrl_write_data,
	.llseek		= seq_lseek,
	.release	= single_release,
};




static int insmod_mode = NORMAL_MODE;

int sonos_forbid_insmod(void)
{
	return ( (insmod_mode == CONSTRAINED_MODE) ? 1 : 0 );
}
EXPORT_SYMBOL(sonos_forbid_insmod);
void proc_insmod_set_constrained(void)
{
	insmod_mode = CONSTRAINED_MODE;
}
EXPORT_SYMBOL(proc_insmod_set_constrained);


static int insmod_ctrl_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "%d\n", insmod_mode);
	return 0;
}

static int insmod_ctrl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, insmod_ctrl_proc_show, NULL);
}

static int insmod_ctrl_write_data(struct file *file, const char __user * buf, size_t length, loff_t * offset)
{
	char	buffer[64];

        memset(buffer, 0, sizeof(buffer));
        if (length > sizeof(buffer) - 1)
                length = sizeof(buffer) - 1;
        if (copy_from_user(buffer, buf, length))
                return -EFAULT;
	if  ( !strncmp((char*)buffer, "forbid", strlen("forbid")) ) {
		if ( !is_mdp_authorized(MDP_AUTH_FLAG_INSMOD_CTRL) ) {
			insmod_mode = CONSTRAINED_MODE;
		} else {
			printk(KERN_INFO "insmod_ctrl authorized - do not constrain.\n");
		}
	} else if ( !strncmp((char*)buffer, "allow", strlen("allow")) ) {
		if ( is_mdp_authorized(MDP_AUTH_FLAG_INSMOD_CTRL) || insmod_mode >= NORMAL_MODE ) {
			insmod_mode = NORMAL_MODE;
		} else {
			printk(KERN_INFO "Operation not supported on locked device.\n");
		}
	} else {
		printk(KERN_INFO "Unsupported operation.\n");
	}

	printk(KERN_INFO "%d\n", insmod_mode);
	return length;
}

static const struct file_operations insmod_ctrl_proc_fops = {
	.open		= insmod_ctrl_proc_open,
	.read		= seq_read,
	.write		= insmod_ctrl_write_data,
	.llseek		= seq_lseek,
	.release	= single_release,
};



static int nodev_mode = NORMAL_MODE;

int sonos_force_nodev(void)
{
	return ( (nodev_mode == CONSTRAINED_MODE) ? 1 : 0 );
}
EXPORT_SYMBOL(sonos_force_nodev);
void proc_nodev_set_constrained(void)
{
	nodev_mode = CONSTRAINED_MODE;
}
EXPORT_SYMBOL(proc_nodev_set_constrained);

static int nodev_ctrl_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "%d\n", nodev_mode);
	return 0;
}

static int nodev_ctrl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, nodev_ctrl_proc_show, NULL);
}

static int nodev_ctrl_write_data(struct file *file, const char __user * buf, size_t length, loff_t * offset)
{
	char	buffer[64];

        memset(buffer, 0, sizeof(buffer));
        if (length > sizeof(buffer) - 1)
                length = sizeof(buffer) - 1;
        if (copy_from_user(buffer, buf, length))
                return -EFAULT;
	if  ( !strncmp((char*)buffer, "constrained",strlen("constrained")) ) {
		nodev_mode = CONSTRAINED_MODE;
	} else if ( !strncmp((char*)buffer, "normal",strlen("normal")) ) {
		if ( is_mdp_authorized(MDP_AUTH_FLAG_NODEV_CTRL) || nodev_mode >= NORMAL_MODE ) {
			nodev_mode = NORMAL_MODE;
		} else {
			printk(KERN_INFO "Operation not supported on locked device.\n");
		}
	} else {
		printk(KERN_INFO "Unsupported operation.\n");
	}

	printk(KERN_INFO "%d\n", nodev_mode);
	return length;
}

static const struct file_operations nodev_ctrl_proc_fops = {
	.open		= nodev_ctrl_proc_open,
	.read		= seq_read,
	.write		= nodev_ctrl_write_data,
	.llseek		= seq_lseek,
	.release	= single_release,
};



static int console_ctrl_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "%d\n", is_mdp_authorized(MDP_AUTH_FLAG_CONSOLE_ENABLE) );
	return 0;
}

static int console_ctrl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, console_ctrl_proc_show, NULL);
}

static const struct file_operations console_ctrl_proc_fops = {
	.open		= console_ctrl_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};


static int telnet_ctrl_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "%d\n", is_mdp_authorized(MDP_AUTH_FLAG_TELNET_ENABLE));
	return 0;
}

static int telnet_ctrl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, telnet_ctrl_proc_show, NULL);
}

static const struct file_operations telnet_ctrl_proc_fops = {
	.open		= telnet_ctrl_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};


static int mdp_flags_ctrl_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "0x%08x\n", (int)sys_mdp.mdp_flags);
	return 0;
}

static int mdp_flags_ctrl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, mdp_flags_ctrl_proc_show, NULL);
}

static const struct file_operations mdp_flags_ctrl_proc_fops = {
	.open		= mdp_flags_ctrl_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};


static int kernel_debug_ctrl_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "%d\n", is_mdp_authorized(MDP_AUTH_FLAG_KERNEL_DEBUG_ENABLE));
	return 0;
}

static int kernel_debug_ctrl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, kernel_debug_ctrl_proc_show, NULL);
}

static const struct file_operations kernel_debug_ctrl_proc_fops = {
	.open		= kernel_debug_ctrl_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};


static int current_ubifs_type = 0;
void sonos_set_proc_crypt(int type)
{
	current_ubifs_type = type;
}
EXPORT_SYMBOL(sonos_set_proc_crypt);

static int ubifs_crypt_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "%d\n", current_ubifs_type);
	return 0;
}

static int ubifs_crypt_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ubifs_crypt_proc_show, NULL);
}

static const struct file_operations ubifs_crypt_proc_fops = {
	.open		= ubifs_crypt_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};


#if defined(SONOS_ARCH_ATTR_SOC_IS_IMX6)
static int boot_counter;

static struct regmap *gpr = NULL;

static void update_boot_counter(int write, int value)
{
	BootCounterReg_t *data;
	if ( !gpr ) {
		gpr = syscon_regmap_lookup_by_compatible(SRC_GPR_COMPATIBLE);
	}

	if ( IS_ERR(gpr) ) {
		printk(KERN_ERR "cannot map imx6q-src\n");
	} else {
		if ( !state_valid ) {
			regmap_read(gpr, BOOT_COUNTER_OFFSET, &boot_counter);
			data = (BootCounterReg_t*)&boot_counter;
			saved_state = data->boot_state;
			state_valid = 1;
		}
		if ( write ) {
			regmap_read(gpr, BOOT_COUNTER_OFFSET, &boot_counter);
			data = (BootCounterReg_t*)&boot_counter;

			if ( value == 0 ) {
				data->boot_section = 0;
				data->fallback_flags = 0;
				data->boot_state = 0;
				data->boot_counter = 0;
#if defined CONFIG_SONOS_DIAGS
			} else if ( value == -1 ) {
				data->boot_section = 0;
				data->fallback_flags = BC_FLAG_STOP_BOOT;
				data->boot_state = 0;
				data->boot_counter = 0;
#endif
			} else {
				data->boot_counter = value;
			}
			regmap_write(gpr, BOOT_COUNTER_OFFSET, boot_counter);
		} else {
			regmap_read(gpr, BOOT_COUNTER_OFFSET, &boot_counter);
		}
	}
}

static int boot_counter_proc_show(struct seq_file *m, void *v)
{
	BootCounterReg_t *bc;
	update_boot_counter(0, 0);
	bc = (BootCounterReg_t*)&boot_counter;
	seq_printf(m, "%d.%d.%d.%d\n", bc->boot_counter, bc->boot_state, bc->fallback_flags, bc->boot_section);
	return 0;
}

static int boot_counter_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, boot_counter_proc_show, NULL);
}

static int boot_counter_write_data(struct file *file, const char __user * buf, size_t length, loff_t * offset)
{
	char	buffer[64];
	__s32	new_count;

        memset(buffer, 0, sizeof(buffer));
        if (length > sizeof(buffer) - 1)
                length = sizeof(buffer) - 1;
        if (copy_from_user(buffer, buf, length))
                return -EFAULT;

	sscanf(buffer, "%d", &new_count);

	if ( new_count >= -1 && new_count <= 10 ) {
		update_boot_counter(1, new_count);
	}
	return length;
}

static const struct file_operations boot_counter_proc_fops = {
	.open		= boot_counter_proc_open,
	.read		= seq_read,
	.write		= boot_counter_write_data,
	.llseek		= seq_lseek,
	.release	= single_release,
};
#endif

extern void wdt2_set_service_state(int);
extern int  wdt2_get_service_state(void);

#if defined(SONOS_ARCH_ATTR_SOC_IS_MT8521P)


void wdt2_set_service_state(int new_state)
{

}

int  wdt2_get_service_state(void)
{
	return KERNEL_DEFAULT;
}

#endif

static int wdog_state_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "%d\n", wdt2_get_service_state());
	return 0;
}

static int wdog_state_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, wdog_state_proc_show, NULL);
}

static int wdog_state_write_data(struct file *file, const char __user * buf, size_t length, loff_t * offset)
{
	char	buffer[64];
	__s32	new_state;

        memset(buffer, 0, sizeof(buffer));
        if (length > sizeof(buffer) - 1)
                length = sizeof(buffer) - 1;
        if (copy_from_user(buffer, buf, length))
                return -EFAULT;

	sscanf(buffer, "%d", &new_state);

	if ( new_state < 0 || new_state > 2 ) {
		printk(KERN_INFO "illegal state - must be 0, 1 or 2\n");
	} else {
		wdt2_set_service_state(new_state);
	}
	return length;
}

static const struct file_operations wdog_state_proc_fops = {
	.open		= wdog_state_proc_open,
	.read		= seq_read,
	.write		= wdog_state_write_data,
	.llseek		= seq_lseek,
	.release	= single_release,
};



static int allowlist_flags_ctrl_proc_show(struct seq_file *m, void *v)
{
	uint32_t flags = 0;
#ifdef CONFIG_SONOS_SECBOOT
	if (__be32_to_cpu(SONOS_FIRMWARE_ALLOWLIST.header.numEntries) > 0) {
		flags |= 0x2;
	}
#endif
	seq_printf(m, "0x%x\n", flags);
	return 0;
}

static int allowlist_flags_ctrl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, allowlist_flags_ctrl_proc_show, NULL);
}

static const struct file_operations allowlist_flags_proc_fops = {
	.open		= allowlist_flags_ctrl_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};

#if defined(SONOS_ARCH_ATTR_SOC_IS_IMX6)

static int fallback_state_proc_show(struct seq_file *m, void *v)
{
	if ( !state_valid ) {
		update_boot_counter(0, 0);
	}
	seq_printf(m, "%d\n", saved_state );
	return 0;
}

static int fallback_state_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, fallback_state_proc_show, NULL);
}

static const struct file_operations fallback_state_proc_fops = {
	.open		= fallback_state_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};
#endif


static int __init proc_unlock_auth_init(void)
{
	proc_create("sonos-lock/console_enable", 0, NULL, &console_ctrl_proc_fops);
	proc_create("sonos-lock/telnet_enable", 0, NULL, &telnet_ctrl_proc_fops);
	proc_create("sonos-lock/insmod_ctrl", 0, NULL, &insmod_ctrl_proc_fops);
	proc_create("sonos-lock/mdp_flags", 0, NULL, &mdp_flags_ctrl_proc_fops);
	proc_create("sonos-lock/dev_enable", 0, NULL, &nodev_ctrl_proc_fops);
	proc_create("sonos-lock/exec_enable", 0, NULL, &noexec_ctrl_proc_fops);
	proc_create("sonos-lock/kernel_debug_enable", 0, NULL, &kernel_debug_ctrl_proc_fops);
	proc_create("sonos-lock/ubifs_crypt", 0, NULL, &ubifs_crypt_proc_fops);
#if defined(SONOS_ARCH_ATTR_SOC_IS_IMX6)
	proc_create("sonos-lock/boot_counter", 0, NULL, &boot_counter_proc_fops);
	proc_create("sonos-lock/fallback_state", 0, NULL, &fallback_state_proc_fops);
#endif


	proc_create("sonos-lock/watchdog_service_state", 0, NULL, &wdog_state_proc_fops);
	proc_create("sonos-lock/allowlist_flags", 0, NULL, &allowlist_flags_proc_fops);
	return 0;
}
module_init(proc_unlock_auth_init);
