/*
 * Copyright (c) 2017-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:	GPL-2.0
 *
 * sonos_unlock_token.c.inc: code for checking unlock token and signatures
 *
 * !!NOTE!! this file is meant to be pulled in with #include after the
 * including project has defined some macros:
 *      SU_BE32_TO_CPU(x)
 *      SU_CPU_TO_BE32(x)
 *      // returns true on success, false on failure
 *      bool SU_GET_UNLOCK_COUNTER(uint32_t *pValue)
 *      SU_PRINT(fmt, fmtArgs...)
 *      SU_PLVL_DEBUG
 *      SU_PLVL_ERR
 *
 * Trying to compile this as a .o and link it in multiple places doesn't work
 * well in situations like the u-boot build or the kernel build (hence the
 * weird file inclusion tactic used here).
 */

#include "sonos_unlock_token.h"

#if !defined(SONOS_ARCH_ATTR_SUPPORTS_SECURE_BOOT)
#undef SU_GET_UNLOCK_COUNTER
#define SU_GET_UNLOCK_COUNTER getCounter
#endif

int sonosUnlockIsValidUnlockSig(const uint8_t* serial, size_t serialLen,
                                uint32_t permissions,
                                uint32_t unlockCounter,
                                const uint8_t* sigBuf, size_t sigBufLen,
                                const char* logPrefix,

                                SonosSignature* sig,
                                SonosHashCallback h,
                                SonosRawVerifyCallback v,
                                SonosKeyLookupCallback lookup,
                                const void* lookupArg,
                                SonosKeyReleaseCallback r)
{
    uint8_t msg[SONOS_UNLOCK_TOKEN_PRESIG_LEN];
    uint8_t* p = msg;

    if (serialLen != SONOS_UNLOCK_SERIAL_LEN) {
        if (logPrefix) {
            SU_PRINT(SU_PLVL_ERR
                     "%s: invalid serial length\n", logPrefix);
        }
        return 0;
    }

    if (sonosSignatureParse(sig, sigBuf, sigBufLen) == 0) {
        if (logPrefix) {
            SU_PRINT(SU_PLVL_ERR
                     "%s: signature failed to parse\n", logPrefix);
        }
        return 0;
    }

    memcpy(p, SONOS_UNLOCK_SIG_PREFIX, SONOS_UNLOCK_SIG_PREFIX_LEN);
    p += SONOS_UNLOCK_SIG_PREFIX_LEN;
    memcpy(p, serial, SONOS_UNLOCK_SERIAL_LEN);
    p += SONOS_UNLOCK_SERIAL_LEN;
    permissions = SU_CPU_TO_BE32(permissions);
    memcpy(p, &permissions, sizeof permissions);
    p += sizeof permissions;
    unlockCounter = SU_CPU_TO_BE32(unlockCounter);
    memcpy(p, &unlockCounter, sizeof unlockCounter);
    p += sizeof unlockCounter;

    if (sonosSignatureVerify(sig, h, v, lookup, lookupArg, r,
                             msg, sizeof msg, SONOS_CT_INVALID) != 1) {
        if (logPrefix) {
            SU_PRINT(SU_PLVL_ERR
                     "%s: signature failed to verify\n", logPrefix);
        }
        return 0;
    }

    return 1;
}

#ifndef SONOS_UNLOCK_OMIT_TOKEN_SUPPORT

int sonosUnlockIsValidUnlockToken(const uint8_t* serial, size_t serialLen,
                                  uint32_t* pPermissions,
                                  uint32_t* pUnlockCounter,
                                  const uint8_t* token, size_t tokenLen,
                                  const char* logPrefix,

#if !defined(SONOS_ARCH_ATTR_SUPPORTS_SECURE_BOOT)
				  SonosGetUnlockCounterCallback getCounter,
#endif
                                  SonosSignature* sig,
                                  SonosHashCallback h,
                                  SonosRawVerifyCallback v,
                                  SonosKeyLookupCallback lookup,
                                  const void* lookupArg,
                                  SonosKeyReleaseCallback r)
{
    const uint8_t* p = token;
    uint32_t permissions;
    uint32_t unlockCounterMsg;
    uint32_t unlockCounterActual;

    *pPermissions = *pUnlockCounter = 0;

    if (serialLen != SONOS_UNLOCK_SERIAL_LEN) {
        if (logPrefix) {
            SU_PRINT(SU_PLVL_ERR "%s: invalid serial length\n", logPrefix);
        }
        return 0;
    }

    if (tokenLen <= SONOS_UNLOCK_TOKEN_PRESIG_LEN) {
        if (logPrefix) {
            SU_PRINT(SU_PLVL_ERR "%s: token too short\n", logPrefix);
        }
        return 0;
    }

    if (memcmp(p, SONOS_UNLOCK_SIG_PREFIX, SONOS_UNLOCK_SIG_PREFIX_LEN) != 0) {
        if (logPrefix) {
            SU_PRINT(SU_PLVL_ERR "%s: bad token prefix\n", logPrefix);
        }
        return 0;
    }
    p += SONOS_UNLOCK_SIG_PREFIX_LEN;

    if (memcmp(p, serial, SONOS_UNLOCK_SERIAL_LEN) != 0) {
        if (logPrefix) {
            SU_PRINT(SU_PLVL_ERR "%s: bad token serial\n", logPrefix);
        }
        return 0;
    }
    p += SONOS_UNLOCK_SERIAL_LEN;

    memcpy(&permissions, p, sizeof(permissions));
    permissions = SU_BE32_TO_CPU(permissions);
    p += sizeof(permissions);

    memcpy(&unlockCounterMsg, p, sizeof(unlockCounterMsg));
    unlockCounterMsg = SU_BE32_TO_CPU(unlockCounterMsg);
    p += sizeof(unlockCounterMsg);

    if (!SU_GET_UNLOCK_COUNTER(&unlockCounterActual)) {
        if (logPrefix) {
            SU_PRINT(SU_PLVL_ERR "%s: get unlock counter failed\n", logPrefix);
        }
        return 0;
    }

    if (unlockCounterActual == SONOS_UNLOCK_CTR_MAX_VALUE) {
        if (logPrefix) {
            SU_PRINT(SU_PLVL_ERR "%s: too many unlocks\n", logPrefix);
        }
        return 0;
    }

    if (sonosUnlockGetNextFuseVal(unlockCounterActual) != unlockCounterMsg) {
        if (logPrefix) {
            SU_PRINT(SU_PLVL_ERR "%s: bad token fuse value\n", logPrefix);
        }
        return 0;
    }

    if (!sonosUnlockIsValidUnlockSig(serial, serialLen,
                                     permissions,
                                     unlockCounterMsg,
                                     token + SONOS_UNLOCK_TOKEN_PRESIG_LEN,
                                     tokenLen - SONOS_UNLOCK_TOKEN_PRESIG_LEN,
                                     logPrefix,
                                     sig, h, v, lookup, lookupArg, r)) {
        return 0;
    }

    *pPermissions = permissions;
    *pUnlockCounter = unlockCounterMsg;
    return 1;
}

#ifndef SONOS_UNLOCK_OMIT_GET_NEXT_FUSE_VAL

uint32_t sonosUnlockGetNextFuseVal(uint32_t fuseval)
{
    uint32_t val;
    int bit;

    for (bit = 0, val = 1;
         bit < SONOS_UNLOCK_CTR_BIT_LEN;
         bit++, val <<= 1) {
        if ((val & fuseval) == 0) {
            return val | fuseval;
        }
    }

    return 0;
}

#endif
#endif
