/*
 * Copyright (c) 2015-2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:	GPL-2.0
 *
 * sonos_unlock.c.inc: code for checking unlock signatures and cpuid/serial
 *                     binding signaturees as used with secure boot
 *
 * !!NOTE!! this file is meant to be pulled in with #include after the
 * including project has defined some macros:
 *      // next two return true on success, false on failure
 *      bool SU_GET_CPUID(uint8_t *buf, size_t len)
 *      bool SU_GET_UNLOCK_COUNTER(uint32_t *pValue)
 *      SU_PRINT(fmt, fmtArgs...)
 *      SU_PLVL_DEBUG
 *      SU_PLVL_ERR
 *
 * Note that SU_GET_CPUID returns true if it can get a known good value for
 * cpuid; false otherwise.
 *
 * Trying to compile this as a .o and link it in multiple places doesn't work
 * well in situations like the u-boot build or the kernel build (hence the
 * weird file inclusion tactic used here).
 */

#include "sonos_unlock.h"

int sonosUnlockVerifyCpuSerialSig(const uint8_t* serial, size_t serialLen,
                                  const uint8_t* sigBuf, size_t sigBufLen,

                                  SonosSignature* sig,
                                  SonosHashCallback h,
                                  SonosRawVerifyCallback v,
                                  SonosKeyLookupCallback lookup,
                                  const void* lookupArg,
                                  SonosKeyReleaseCallback r)
{
    static const uint8_t cpuSerialSigMagic[] =
    {
        0xec, 0x31, 0x14, 0xef
    };
    uint8_t cpuid[SONOS_UNLOCK_CPUID_LEN];
    uint8_t msg[sizeof(cpuSerialSigMagic) + SONOS_UNLOCK_SERIAL_LEN + sizeof(cpuid)];
    uint8_t* p = msg;
    size_t sigLen = sonosSignatureParse(sig, sigBuf, sigBufLen);

    if (serialLen != SONOS_UNLOCK_SERIAL_LEN) {
        SU_PRINT(SU_PLVL_ERR
                 "cpuid/serial signature invalid serialLen: %d\n",
                 (int)serialLen);
        return 0;
    }

    if (sigLen == 0) {
        SU_PRINT(SU_PLVL_ERR
                 "cpuid/serial signature failed to parse\n");
        return 0;
    }

    if (!SU_GET_CPUID(cpuid, sizeof cpuid)) {
        SU_PRINT(SU_PLVL_ERR "cpuid/serial signature get_cpuid failed\n");
        return 0;
    }

    memcpy(p, cpuSerialSigMagic, sizeof cpuSerialSigMagic);
    p += sizeof cpuSerialSigMagic;
    memcpy(p, serial, SONOS_UNLOCK_SERIAL_LEN);
    p += SONOS_UNLOCK_SERIAL_LEN;
    memcpy(p, cpuid, sizeof cpuid);
    p += sizeof cpuid;

    if (!sonosSignatureVerify(sig, h, v, lookup, lookupArg, r, msg, sizeof msg,
                              SONOS_CT_INVALID)) {
        SU_PRINT(SU_PLVL_ERR
                 "cpuid/serial signature failed to verify\n");
        return 0;
    }

    SU_PRINT(SU_PLVL_DEBUG
             "cpuid/serial signature verified successfully\n");
    return 1;
}

#ifndef SU_UNLOCK_VERIFY_SIG_ONLY
int sonosUnlockIsDeviceUnlocked(const struct manufacturing_data_page* mdp,
                                const struct manufacturing_data_page3* mdp3,

                                SonosSignature* sig,
                                SonosHashCallback h,
                                SonosRawVerifyCallback v,
                                SonosKeyLookupCallback lookup,
                                const void* lookupArgCpuid,
                                const void* lookupArgUnlock,
                                SonosKeyReleaseCallback r)
{
    uint32_t fuseval;

    if (mdp->mdp_version < MDP_VERSION_AUTH_FLAGS ||
        mdp->mdp3_version < MDP3_VERSION_SECURE_BOOT) {
        SU_PRINT(SU_PLVL_INFO "IsDeviceUnlocked: versions too low (%d %d)\n",
                 (int)mdp->mdp_version, (int)mdp->mdp3_version);
        return 0;
    }

    if (!SU_GET_UNLOCK_COUNTER(&fuseval)) {
        SU_PRINT(SU_PLVL_ERR
                 "IsDeviceUnlocked: unlock counter fuse read failed\n");
        return 0;
    }

    if (!sonosUnlockIsValidUnlockSig(mdp->mdp_serial, SONOS_UNLOCK_SERIAL_LEN,
                                     mdp->mdp_authorized_flags,
                                     fuseval,
                                     mdp3->mdp3_auth_sig,
                                     sizeof(mdp3->mdp3_auth_sig),
                                     "IsDeviceUnlocked",
                                     sig, h, v, lookup, lookupArgUnlock, r)) {
        return 0;
    }

    if (!sonosUnlockVerifyCpuSerialSig(mdp->mdp_serial, SONOS_UNLOCK_SERIAL_LEN,
                                       mdp3->mdp3_cpuid_sig,
                                       sizeof(mdp3->mdp3_cpuid_sig),
                                       sig, h, v, lookup, lookupArgCpuid, r)) {
        SU_PRINT(SU_PLVL_ERR
                 "IsDeviceUnlocked: bad cpu/serial signature\n");
        return 0;
    }

    SU_PRINT(SU_PLVL_DEBUG
             "IsDeviceUnlocked: signature verified successfully\n");
    return 1;
}

int sonosUnlockIsAuthFeatureEnabled(uint32_t flag,
                                    const struct manufacturing_data_page* mdp,
                                    const struct manufacturing_data_page3* mdp3,
                                    SonosSignature* sig,
                                    SonosHashCallback h,
                                    SonosRawVerifyCallback v,
                                    SonosKeyLookupCallback lookup,
                                    const void* lookupArgCpuid,
                                    const void* lookupArgUnlock,
                                    SonosKeyReleaseCallback r)
{
#ifdef SONOS_ARCH_ATTR_RELAX_SECBOOT_STRICTNESS
    uint32_t permissions = MDP_AUTH_FLAG_ALL_FEATURES & (~MDP_AUTH_FLAG_MFG_KEY_ENABLE);
    uint32_t features = permissions;
#else
    uint32_t permissions = 0;
    uint32_t features = 0;
#endif
    int enabled;

#if (defined DIAG_BUILD || defined CONFIG_SONOS_DIAGS) && !defined SONOS_STRICT_DIAG_BUILD
    if (flag != MDP_AUTH_FLAG_MFG_KEY_ENABLE) {
        permissions = MDP_AUTH_FLAG_ALL_FEATURES & (~MDP_AUTH_FLAG_MFG_KEY_ENABLE);
        features = permissions;
        goto decision;
    }
#endif

    if (mdp == NULL ||
        mdp->mdp_magic != MDP_MAGIC ||
        mdp->mdp_version < MDP_VERSION_AUTH_FLAGS) {
        goto decision;
    }

    if (!(mdp->mdp_pages_present & MDP_PAGE3_PRESENT) ||
        mdp3 == NULL ||
        mdp3->mdp3_magic != MDP_MAGIC3 ||
        mdp3->mdp3_version < MDP3_VERSION_DEV_CERT) {
        goto decision;
    }

    permissions = 0;
    features = mdp->mdp_sw_features;
    if (((mdp->mdp_authorized_flags & features) & flag) &&
        sonosUnlockIsDeviceUnlocked(mdp, mdp3, sig, h, v, lookup,
                                    lookupArgCpuid, lookupArgUnlock, r)) {
        permissions = mdp->mdp_authorized_flags;
    }

decision:
    enabled = ((permissions & features) & flag) ? 1 : 0;
    SU_PRINT(SU_PLVL_DEBUG
             "IsAuthFeatureEnabled (0x%lx): %s\n",
             (unsigned long)flag, enabled ? "true" : "false");
    return enabled;
}
#endif

