/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 *
 * sonos_signature_verify.h: Sonos digital signature format
 *                           verification and parsing
 */

#ifndef SONOS_SIGNATURE_VERIFY_H
#define SONOS_SIGNATURE_VERIFY_H

#include "sonos_signature.h"
#include "sonos_signature_common.h"
#include "sonos_attr.h"

#ifdef __cplusplus
extern "C" {
#endif


size_t sonosSignatureParse(SonosSignature *sig,
                           const uint8_t *buf, size_t bufLen);

size_t sonosSignatureParseFromEnd(SonosSignature *sig,
                                  const uint8_t *buf, size_t bufLen,
                                  size_t numHitsToSkip);

typedef int (*SonosRawVerifyCallback)(SonosSigningKey_t key,
                                      SonosSignatureAlg_t signAlg,
                                      SonosDigestAlg_t digestAlg,
                                      const uint8_t *digest, size_t digestLen,
                                      const uint8_t *signature, size_t sigLen);

typedef SonosSigningKey_t (*SonosKeyLookupCallback)(const void *cbArg,
        SonosKeyIdentifierScheme_t keyIdscheme,
        const uint8_t *keyId,
        size_t keyIdLen);

typedef void (*SonosKeyReleaseCallback)(SonosSigningKey_t key);

int sonosSignatureVerifyInternal(const SonosSignature *sig,
                         SonosHashCallback hash,
                         SonosRawVerifyCallback rawVerify,
                         SonosKeyLookupCallback keyLookup,
                         const void *keyLookupArg,
                         SonosKeyReleaseCallback keyRelease,
                         const uint8_t *msg, size_t msgLen,
                         SonosContentType_t contentType,
                         int relaxContentTypeSigs);

int sonosSignatureVerify(const SonosSignature *sig,
                         SonosHashCallback hash,
                         SonosRawVerifyCallback rawVerify,
                         SonosKeyLookupCallback keyLookup,
                         const void *keyLookupArg,
                         SonosKeyReleaseCallback keyRelease,
                         const uint8_t *msg, size_t msgLen,
                         SonosContentType_t contentType);

int sonosSignatureVerifyFromDigestInternal(const SonosSignature *sig,
                                   SonosHashCallback hash,
                                   SonosRawVerifyCallback rawVerify,
                                   SonosKeyLookupCallback keyLookup,
                                   const void *keyLookupArg,
                                   SonosKeyReleaseCallback keyRelease,
                                   SonosDigestAlg_t digestAlg,
                                   const uint8_t *digest, size_t digestLen,
                                   SonosContentType_t contentType,
                                   int relaxContentTypeSigs);

int sonosSignatureVerifyFromDigest(const SonosSignature *sig,
                                   SonosHashCallback hash,
                                   SonosRawVerifyCallback rawVerify,
                                   SonosKeyLookupCallback keyLookup,
                                   const void *keyLookupArg,
                                   SonosKeyReleaseCallback keyRelease,
                                   SonosDigestAlg_t digestAlg,
                                   const uint8_t *digest, size_t digestLen,
                                   SonosContentType_t contentType);

int sonosSignatureGetContentType(SonosAttributeContentType *ct,
                                 const SonosSignature *sig);

int sonosSignatureGetMessageDigest(SonosAttributeMessageDigest *md,
                                   const SonosSignature *sig);

int sonosSignatureGetSrkRevoke(SonosAttributeSrkRevoke *sr,
                               const SonosSignature *sig);

int sonosSignatureGetAllowlist(SonosAttributeAllowlist *al,
                               const SonosSignature *sig);

int sonosSignatureGetArVersion(SonosAttributeArVersion *arv,
                               const SonosSignature *sig);

#ifdef __cplusplus
}
#endif

#endif
