/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 *
 * sonos_signature_sign.c.inc: Sonos digital signature format
 *                             creation and serialization
 *
 * !!NOTE!! this file is meant to be pulled in with #include after the
 * including project has defined some macros:
 *      SS_MAX_DIGEST_LENGTH
 *      SS_MEMSET(d, c, n)
 *      SS_SIZET_MAX
 *      NULL
 */

#include "sonos_signature_common.h"
#include "sonos_signature_sign.h"
#include "sonos_nasn_serialize.h"
#include "sonos_attr.h"

static int
serializeSonosSignatureSignerInfo(const SonosSignatureSignerInfo *si,
                                  uint8_t **buf_p,
                                  const uint8_t *end)
{
    uint8_t *buf = *buf_p;

    PUT_INT_BOUNDED(si->keyIdLen, SS_MAX_KEYID_LEN);
    PUT_ENUM(si->keyIdScheme, sonosSignatureIsValidKeyIdentifierScheme);
    PUT_BUF(si->keyId, si->keyIdLen);
    PUT_ENUM(si->digestAlg, sonosSignatureIsValidDigestAlg);
    PUT_ENUM(si->signatureAlg, sonosSignatureIsValidSignatureAlg);
    PUT_INT_BOUNDED(si->sigLen, SS_MAX_SIG_LEN);
    PUT_BUF(si->signature, si->sigLen);

    *buf_p = buf;
    return 1;
}

size_t
sonosSignatureSerialize(const SonosSignature *sig,
                        uint8_t *buf, size_t bufLen)
{
    const uint8_t *start = buf;
    const uint8_t *end;
    uint8_t i8;
    uint32_t i32;
    size_t serializedLen = 0;
    int dryRun = buf == NULL;

    if (dryRun) {
        bufLen = SS_SIZET_MAX;
        end = NULL;
    }
    else {
        end = buf + bufLen;
    }

    PUT_INT(sig->magic);
    if (sig->magic != SS_MAGIC) {
        return 0;
    }

    PUT_INT_BOUNDED(sig->totalLen, bufLen);

    PUT_INT(sig->versionMajor);
    if (sig->versionMajor != SS_VERSION_MAJOR) {
        return 0;
    }
    PUT_INT(sig->versionMinor);

    PUT_INT_BOUNDED(sig->numUnsignedAttrs, SS_MAX_NUM_UNSIGNED_ATTRS);
    for (i32 = 0; i32 < sig->numUnsignedAttrs; i32++) {
        if (!sonosAttributeSerialize(&sig->unsignedAttrs[i32], &buf, end)) {
            return 0;
        }
    }

    PUT_INT_BOUNDED(sig->numSignedAttrs, SS_MAX_NUM_SIGNED_ATTRS);
    for (i32 = 0; i32 < sig->numSignedAttrs; i32++) {
        if (!sonosAttributeSerialize(&sig->signedAttrs[i32], &buf, end)) {
            return 0;
        }
    }

    PUT_INT_BOUNDED(sig->numSigners, SS_MAX_NUM_SIGNERS);
    if (sig->numSigners == 0) {
        return 0;
    }
    for (i8 = 0; i8 < sig->numSigners; i8++) {
        if (!serializeSonosSignatureSignerInfo(&sig->signerInfos[i8],
                                               &buf, end)) {
            return 0;
        }
    }

    serializedLen = buf - start;

    if (!dryRun && serializedLen != sig->totalLen) {
        return 0;
    }

    return serializedLen;
}

void sonosSignatureParamsInit(SonosSignatureParams *sigParams)
{
    memset(sigParams, 0, sizeof(*sigParams));
#if defined(STATIC_ASSERT)
    STATIC_ASSERT(SONOS_CT_INVALID == 0);
    STATIC_ASSERT(SONOS_SR_NONE == 0);
#endif
}

int
sonosSignatureCreate(SonosSignature *sig,
                     SonosHashCallback hash,
                     SonosRawSignCallback rawSign,
                     SonosDigestAlg_t digestAlg,
                     SonosSignatureAlg_t sigAlg,
                     SonosSigningKey_t key,
                     SonosKeyIdentifierScheme_t keyIdScheme,
                     const uint8_t *keyId, size_t keyIdLen,
                     const uint8_t *msg, size_t msgLen,
                     const SonosSignatureParams *sigParams)
{
    unsigned char digest[SS_MAX_DIGEST_LENGTH];
    size_t digestLen = sizeof digest;

    if (!hash(digestAlg, msg, msgLen, digest, &digestLen)) {
        return 0;
    }

    return sonosSignatureCreateFromDigest(sig, hash, rawSign, digestAlg, sigAlg,
                                          key, keyIdScheme, keyId, keyIdLen,
                                          digest, digestLen, sigParams);
}

int
sonosSignatureCreateFromDigest(SonosSignature *sig,
                               SonosHashCallback hash,
                               SonosRawSignCallback rawSign,
                               SonosDigestAlg_t digestAlg,
                               SonosSignatureAlg_t sigAlg,
                               SonosSigningKey_t key,
                               SonosKeyIdentifierScheme_t keyIdScheme,
                               const uint8_t *keyId, size_t keyIdLen,
                               const uint8_t *digest, size_t digestLen,
                               const SonosSignatureParams *sigParams)
{
    SonosSignatureSignerInfo *si;
    uint8_t signedAttrsDigest[SS_MAX_DIGEST_LENGTH];
    const uint8_t *digestToSign = digest;
    int rc = 0;
    uint8_t *serializedAttrs = NULL;
    size_t serializedAttrsLen = 4;
    size_t sigLen;

    SS_MEMSET(sig, 0, sizeof(*sig));

    sig->magic = SS_MAGIC;
    sig->versionMajor = SS_VERSION_MAJOR;
    sig->versionMinor = SS_VERSION_MINOR;
    sig->numSigners = 1;
    si = &sig->signerInfos[0];

    if (!sonosSignatureIsValidKeyIdentifierScheme(keyIdScheme) ||
        keyIdLen > SS_MAX_KEYID_LEN) {
        goto error;
    }
    si->keyIdScheme = keyIdScheme;
    si->keyIdLen = keyIdLen;
    SONOS_NASN_MEMCPY(si->keyId, keyId, si->keyIdLen);
    si->digestAlg = digestAlg;
    si->signatureAlg = sigAlg;
    si->sigLen = SS_MAX_SIG_LEN;

    if (!sonosSignatureIsValidDigestAlg(digestAlg) ||
        SONOS_DIGEST_ALG_LEN(digestAlg) != digestLen) {
        goto error;
    }

    if (sigParams->contentType != SONOS_CT_INVALID ||
        sigParams->srkRevoke != SONOS_SR_NONE ||
        (sigParams->allowlist && sigParams->allowlistLen) ||
        sigParams->pArVersion) {
        SonosAttribute *sa;
        size_t signedAttrsDigestLen = sizeof signedAttrsDigest;

        uint8_t *buf;
        const uint8_t *end;
        uint32_t i32;

        sa = &sig->signedAttrs[(sig->numSignedAttrs)++];
        sa->attributeId = SONOS_ATTR_ID_MESSAGE_DIGEST;
        sa->attributeValueLen = sizeof(SonosDigestAlg_t) + digestLen;
        sa->x.md.alg = digestAlg;
        memcpy(sa->x.md.digest, digest, digestLen);
        serializedAttrsLen += 2*sizeof(uint32_t) + sa->attributeValueLen;

        if (sigParams->contentType != SONOS_CT_INVALID) {
            sa = &sig->signedAttrs[(sig->numSignedAttrs)++];
            sa->attributeId = SONOS_ATTR_ID_CONTENT_TYPE;
            sa->attributeValueLen = sizeof(sa->x.ct);
            sa->x.ct.contentType = sigParams->contentType;
            serializedAttrsLen += 2*sizeof(uint32_t) + sa->attributeValueLen;
        }

        if (sigParams->srkRevoke != SONOS_SR_NONE) {
            sa = &sig->signedAttrs[(sig->numSignedAttrs)++];
            sa->attributeId = SONOS_ATTR_ID_SRK_REVOKE | SONOS_ATTR_CRIT_BIT;
            sa->attributeValueLen = sizeof(sa->x.sr);
            sa->x.sr.srkRevokeFuse = sigParams->srkRevoke;
            serializedAttrsLen += 2*sizeof(uint32_t) + sa->attributeValueLen;
        }

        if (sigParams->allowlist != NULL && sigParams->allowlistLen != 0) {

            const uint8_t *allowlist = sigParams->allowlist;

            sa = &sig->signedAttrs[(sig->numSignedAttrs)++];
            sa->attributeId = SONOS_ATTR_ID_ALLOWLIST | SONOS_ATTR_CRIT_BIT;
            sa->attributeValueLen = sigParams->allowlistLen;
            if (!sonosAttributeParseAllowlist(sa, &allowlist,
                                              allowlist +
                                              sigParams->allowlistLen)) {
                goto error;
            }
            serializedAttrsLen += 2*sizeof(uint32_t) + sa->attributeValueLen;
        }

        if (sigParams->pArVersion) {
            sa = &sig->signedAttrs[(sig->numSignedAttrs)++];
            sa->attributeId = SONOS_ATTR_ID_AR_VERSION;
            sa->attributeValueLen = sizeof(sa->x.arv);
            sa->x.arv.arVersion = *(sigParams->pArVersion);
            serializedAttrsLen += 2*sizeof(uint32_t) + sa->attributeValueLen;
        }


        serializedAttrs = malloc(serializedAttrsLen);
        if (serializedAttrs == NULL) {
            goto error;
        }

        buf = serializedAttrs;
        end = serializedAttrs + serializedAttrsLen;

        i32 = sig->numSignedAttrs;
        i32 = SONOS_NASN_CPU_TO_BE32(i32);
        SONOS_NASN_MEMCPY(buf, &i32, 4);
        buf += 4;
        for (i32 = 0; i32 < sig->numSignedAttrs; i32++) {
            if (!sonosAttributeSerialize(&sig->signedAttrs[i32], &buf, end)) {
                goto error;
            }
        }

        if (!sonosSignatureHashSignedAttributes(sig, serializedAttrs, buf - serializedAttrs,
                                                hash, digestAlg,
                                                signedAttrsDigest,
                                                &signedAttrsDigestLen) ||
            signedAttrsDigestLen != digestLen) {
            goto error;
        }
        digestToSign = signedAttrsDigest;
    }

    sigLen = si->sigLen;
    if (!rawSign(key, digestAlg, sigAlg, digestToSign, digestLen,
                 si->signature, &sigLen)) {
        goto error;
    }
    si->sigLen = sigLen;

    sig->totalLen = sonosSignatureSerialize(sig, NULL, 0);
    if (sig->totalLen == 0) {
        goto error;
    }

    {
        const uint32_t alignment = 4;
        const uint32_t minPaddedLen =
            sig->totalLen +
            sizeof(sig->unsignedAttrs[0].attributeId) +
            sizeof(sig->unsignedAttrs[0].attributeValueLen);
        uint32_t padValueLen;

        if (sigParams->padToLen) {
            if (sigParams->padToLen == sig->totalLen) {
                goto success;
            }

            if (sigParams->padToLen < minPaddedLen) {
                goto error;
            }

            padValueLen = sigParams->padToLen - minPaddedLen;
        }
        else {
            if (sig->totalLen % alignment == 0) {
                goto success;
            }

            padValueLen = alignment - (minPaddedLen % alignment);
        }

        if (padValueLen > sizeof(sig->unsignedAttrs[0].x.attributeValue)) {
            goto error;
        }
        sig->numUnsignedAttrs = 1;
        sig->unsignedAttrs[0].attributeId = SONOS_ATTR_ID_PADDING;
        sig->unsignedAttrs[0].attributeValueLen = padValueLen;

        sig->totalLen = 0;
        sig->totalLen = sonosSignatureSerialize(sig, NULL, 0);

        if (sigParams->padToLen) {
            if (sig->totalLen != sigParams->padToLen) {
                goto error;
            }
        }
        else {
            if (sig->totalLen == 0 || (sig->totalLen % alignment != 0)) {
                goto error;
            }
        }
    }

success:
    rc = 1;

error:
    free(serializedAttrs);
    return rc;
}
