/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 *
 * sonos_signature_common.c.inc: helper functions for Sonos digital signature format
 */

#include "sonos_signature_common.h"

int
sonosSignatureIsValidDigestAlg(SonosDigestAlg_t a)
{
    return a == SONOS_DIGEST_ALG_SHA256;
}

int
sonosSignatureIsValidSignatureAlg(SonosSignatureAlg_t a)
{
    return a == SONOS_SIGNATURE_ALG_RSAPKCS1;
}

int
sonosSignatureIsValidKeyIdentifierScheme(SonosKeyIdentifierScheme_t s)
{
    return s == SS_KI_SCHEME_X509_INLINE ||
           s == SS_KI_SCHEME_X509_SKI ||
           s == SS_KI_SCHEME_SHA1 ||
           s == SS_KI_SCHEME_NAME ||
           s == SS_KI_SCHEME_RSA_INLINE;
}


int sonosSignatureIsValidAllowlistType(uint8_t t)
{
    return t == SONOS_FWA_TYPE_CPUID ||
           t == SONOS_FWA_TYPE_SERIAL ||
           t == SONOS_FWA_TYPE_CPUID16;
}


#ifdef SONOS_ARCH_ATTR_SUPPORTS_SECURE_BOOT
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wframe-larger-than="
#endif
int
sonosSignatureHashSignedAttributes(const SonosSignature *sig,
                                   const uint8_t *serializedAttrs,
                                   size_t serializedAttrsLen,
                                   SonosHashCallback hash,
                                   SonosDigestAlg_t alg,
                                   uint8_t *digest,
                                   size_t *pDigestLen)
{
    const uint8_t *buf = NULL;
    size_t bufLen = 0;

    if (sig->serializedSignedAttrs != NULL &&
        sig->serializedSignedAttrsLen != 0) {
        buf = sig->serializedSignedAttrs;
        bufLen = sig->serializedSignedAttrsLen;
    }
    else if (serializedAttrs != NULL && serializedAttrsLen != 0) {
        buf = serializedAttrs;
        bufLen = serializedAttrsLen;
    }
    else {
        return 0;
    }

    return hash(alg, buf, bufLen, digest, pDigestLen);
}

#ifdef SONOS_ARCH_ATTR_SUPPORTS_SECURE_BOOT
#pragma GCC diagnostic pop
#endif

#ifdef SONOS_ARCH_ATTR_IS_BUILD_TOOLS

#include <inttypes.h>
#include "sonos_attr.h"

void
sonosSignaturePrintAttributes(FILE *f, const SonosSignature *sig)
{
    uint32_t i;

    fprintf(f, "%"PRIu32" unsigned attributes:\n", sig->numUnsignedAttrs);
    for (i = 0; i < sig->numUnsignedAttrs; i++) {
        sonosAttributePrint(f, &sig->unsignedAttrs[i]);
    }
    fprintf(f, "%"PRIu32" signed attributes:\n", sig->numSignedAttrs);
    for (i = 0; i < sig->numSignedAttrs; i++) {
        sonosAttributePrint(f, &sig->signedAttrs[i]);
    }
}

#endif

