/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:	GPL-2.0
 *
 * sonos_fw_allowlist.h: structures representing a firmware allowlist
 *                       that binds a u-boot or kernel image to a specific
 *                       list of devices
 */

#ifndef SONOS_FW_ALLOWLIST_H
#define SONOS_FW_ALLOWLIST_H

#include "sonos_stdint.h"

#ifndef __GNUC__
#define __attribute__(x)
#endif

#define SONOS_FWA_MAGIC_INIT \
    0x56, 0xea, 0x9f, 0x16, 0x21, 0x54, 0x31, 0xa5, \
    0xba, 0x0f, 0x50, 0xab, 0xf1, 0x64, 0xfa, 0x17, \
    0x6d, 0xab, 0xb0, 0x10, 0xc4, 0xb5, 0x71, 0x9e, \
    0x33, 0x31, 0x59, 0xc6, 0x82, 0xba, 0x4c,

#define SONOS_FWA_MAGIC_INIT_INVERTED \
    0xa9, 0x15, 0x60, 0xe9, 0xde, 0xab, 0xce, 0x5a, \
    0x45, 0xf0, 0xaf, 0x54, 0x0e, 0x9b, 0x05, 0xe8, \
    0x92, 0x54, 0x4f, 0xef, 0x3b, 0x4a, 0x8e, 0x61, \
    0xcc, 0xce, 0xa6, 0x39, 0x7d, 0x45, 0xb3,

#define SONOS_FWA_MAGIC_INIT_UNCOMPRESSED_COPY \
    0x47, 0x49, 0x46, 0x3B, 0x70, 0xB4, 0x33, 0xA6, \
    0x67, 0xF1, 0x7B, 0xD7, 0x4B, 0xA3, 0x1A, 0x4D, \
    0x5B, 0xA1, 0xE8, 0xAA, 0x99, 0x83, 0xA3, 0x8D, \
    0x58, 0xEE, 0xA1, 0xB6, 0xE6, 0x00, 0xCF,

#define SONOS_FWA_MAGIC_INIT_UNCOMPRESSED_COPY_INVERTED \
    0xb8, 0xb6, 0xb9, 0xc4, 0x8f, 0x4b, 0xcc, 0x59, \
    0x98, 0x0e, 0x84, 0x28, 0xb4, 0x5c, 0xe5, 0xb2, \
    0xa4, 0x5e, 0x17, 0x55, 0x66, 0x7c, 0x5c, 0x72, \
    0xa7, 0x11, 0x5e, 0x49, 0x19, 0xff, 0x30,

#ifdef SONOS_FWA_IN_UNCOMPRESSED_KERNEL
#undef SONOS_FWA_MAGIC_INIT
#define SONOS_FWA_MAGIC_INIT SONOS_FWA_MAGIC_INIT_UNCOMPRESSED_COPY
#endif

#define SONOS_FWA_MAGIC_LEN    31

#define SONOS_FWA_MAX_ENTRIES  10000

#define SONOS_FWA_TYPE_CPUID   1
#define SONOS_FWA_TYPE_SERIAL  2
#define SONOS_FWA_TYPE_CPUID16 3

#define SONOS_FWA_LEN_CPUID    8
#define SONOS_FWA_LEN_CPUID16  16
#define SONOS_FWA_LEN_SERIAL   6

#define SONOS_FWA_FMT_CPUID     "%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x"
#define SONOS_FWA_FMT_CPUID16   "%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x"
#define SONOS_FWA_FMT_SERIAL    "%02x:%02x:%02x:%02x:%02x:%02x"

#define SONOS_FWA_FMTARG_CPUID(p)   p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7]
#define SONOS_FWA_FMTARG_CPUID16(p) p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9], p[10], p[11], p[12], p[13], p[14], p[15]
#define SONOS_FWA_FMTARG_SERIAL(p)  p[0], p[1], p[2], p[3], p[4], p[5]

typedef struct __attribute__((packed)) {
    uint8_t magic[SONOS_FWA_MAGIC_LEN];
    uint8_t allowlistType;
    uint32_t numEntries;
} SonosFirmwareAllowlistHeader;

#ifndef NDEBUG
typedef struct {
    uint8_t magic[SONOS_FWA_MAGIC_LEN];
    uint8_t allowlistType;
    uint32_t numEntries;
} SonosFirmwareAllowlistHeaderUnpacked;
#endif

typedef struct { uint8_t value[SONOS_FWA_LEN_CPUID]; } SonosCpuid_t;
typedef struct { uint8_t value[SONOS_FWA_LEN_CPUID16]; } SonosCpuid16_t;
typedef struct { uint8_t value[SONOS_FWA_LEN_SERIAL]; } SonosSerial_t;

#if defined(SONOS_ARCH_ATTR_SOC_IS_A113) || \
    defined(SONOS_ARCH_ATTR_SOC_IS_MT8518S)
#define SONOS_FWA_OWN_TYPE_CPUID    SONOS_FWA_TYPE_CPUID16
#define SONOS_FWA_OWN_FMT_CPUID     SONOS_FWA_FMT_CPUID16
#define SONOS_FWA_OWN_FMTARG_CPUID  SONOS_FWA_FMTARG_CPUID16
typedef SonosCpuid16_t SonosOwnCpuid_t;
#else
#define SONOS_FWA_OWN_TYPE_CPUID    SONOS_FWA_TYPE_CPUID
#define SONOS_FWA_OWN_FMT_CPUID     SONOS_FWA_FMT_CPUID
#define SONOS_FWA_OWN_FMTARG_CPUID  SONOS_FWA_FMTARG_CPUID
typedef SonosCpuid_t SonosOwnCpuid_t;
#endif

#if defined(SONOS_ARCH_ATTR_IS_BUILD_TOOLS)

static inline int find_and_check_sonos_allowlist(const uint8_t* buf,
                                                 size_t bufLen,
                                                 int isKernel,
                                                 int isPermissive)
{
    (void)buf;
    (void)bufLen;
    (void)isKernel;
    (void)isPermissive;
    return 1;
}

#else


int find_and_check_sonos_allowlist(const uint8_t* buf, size_t bufLen,
                                   int isKernel, int isPermissive);


int check_sonos_firmware_allowlist_ex(const SonosFirmwareAllowlistHeader* hdr,
                                      const uint8_t *entries,
                                      size_t maxEntriesSize);

#endif

#endif
