/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:	GPL-2.0
 *
 * sonos_fw_allowlist.c.inc: implementation of check_sonos_firmware_allowlist
 *
 * !!NOTE!! this file is meant to be pulled in with #include after the
 * including project has defined some macros:
 *      int SFW_GETCPUID(uint8_t *buf, size_t len)
 *      int SFW_GETSERIAL(uint8_t *buf, size_t len)
 *      SFW_PRINT(fmt, fmtArgs...)
 *      SFW_PLVL_INFO
 *      SFW_PLVL_EMERG
 *      SFW_BE32_TO_CPU(x)
 *
 * Note that SWF_GETSERIAL should return true if it can get a known
 * good (cryptographically secure) value for serial; false otherwise.
 * If it fails then any non-empty allowlist causes us to not boot.
 *
 * Callers needing to validate allowlists in other images should also define:
 *      SFW_INCLUDE_FIND_AND_CHECK_SONOS_ALLOWLIST
 *
 * Trying to compile this as a .o and link it in multiple places doesn't work
 * well in situations like the u-boot build or the kernel build (hence the
 * weird file inclusion tactic used here).
 */

static void
initialPrintHelper(uint8_t type, const uint8_t* id)
{
    const char* typeName;

    if (type == SONOS_FWA_TYPE_SERIAL) {
        typeName = "serial";
        SFW_PRINT(SFW_PLVL_INFO
                  "checking %s allowlist (my %s is "
                  SONOS_FWA_FMT_SERIAL ")...\n",
                  typeName, typeName, SONOS_FWA_FMTARG_SERIAL(id));
    }
    else {
        typeName = "cpuid";
        SFW_PRINT(SFW_PLVL_INFO
                  "checking %s allowlist (my %s is "
                  SONOS_FWA_OWN_FMT_CPUID ")...\n",
                  typeName, typeName, SONOS_FWA_OWN_FMTARG_CPUID(id));
    }
    (void)typeName;
}

static void
foreachPrintHelper(uint8_t type, const uint8_t* id)
{
    if (type == SONOS_FWA_TYPE_SERIAL) {
        SFW_PRINT(SFW_PLVL_INFO
                  "checking serial "SONOS_FWA_FMT_SERIAL"\n",
                  SONOS_FWA_FMTARG_SERIAL(id));
    }
    else {
        SFW_PRINT(SFW_PLVL_INFO
                  "checking cpuid "SONOS_FWA_OWN_FMT_CPUID"\n",
                  SONOS_FWA_OWN_FMTARG_CPUID(id));
    }
}

int
check_sonos_firmware_allowlist_ex(const SonosFirmwareAllowlistHeader* hdr,
                                  const uint8_t *entries,
                                  size_t maxEntriesSize)
{
    uint32_t numEntries = SFW_BE32_TO_CPU(hdr->numEntries);
    uint8_t type = hdr->allowlistType;
    int foundAllowlistEntry = 0;
    const uint8_t* myId = NULL;
    const uint8_t* listId = entries;
    size_t listIdLen;
    const char* typeName = NULL;
    uint32_t i;
    SonosOwnCpuid_t myCpuid;
    SonosSerial_t mySerial;
    int result = 0;

    if (type != SONOS_FWA_TYPE_CPUID && type != SONOS_FWA_TYPE_SERIAL && type != SONOS_FWA_TYPE_CPUID16) {
        SFW_PRINT(SFW_PLVL_EMERG
                  "bad Sonos firmware allowlist type: %d\n", (int)type);
        return result;
    }
    if (numEntries > SONOS_FWA_MAX_ENTRIES) {
        SFW_PRINT(SFW_PLVL_EMERG
                  "Firmware allowlist too large (%lu)\n", (unsigned long)numEntries);
        return result;
    }

    if (type == SONOS_FWA_TYPE_SERIAL) {
        typeName = "serial";
        if (SFW_GETSERIAL(mySerial.value, sizeof(mySerial.value))) {
            myId = mySerial.value;
        }
        listIdLen = sizeof(mySerial);
    }
    else {
        typeName = "cpuid";
        if (SFW_GETCPUID(myCpuid.value, sizeof(myCpuid.value))) {
            myId = myCpuid.value;
        }
        listIdLen = sizeof(myCpuid);
    }

    if (numEntries * listIdLen > maxEntriesSize) {
        SFW_PRINT(SFW_PLVL_EMERG "Firmware allowlist invalid size\n");
        return result;
    }

    if (myId) {
        initialPrintHelper(type, myId);

        if (numEntries != 0) {
            if (type != SONOS_FWA_TYPE_SERIAL &&
                type != SONOS_FWA_OWN_TYPE_CPUID) {
                SFW_PRINT(SFW_PLVL_EMERG
                          "bad Sonos firmware allowlist type: %d\n", (int)type);
                return result;
            }

            for (i = 0; i < numEntries; i++, listId += listIdLen) {
                if (i < 10) {
                    foreachPrintHelper(type, listId);
                }
                else if (i == 10) {
                    SFW_PRINT(SFW_PLVL_INFO
                              "... checking too many to list ...\n");
                }
                if (memcmp(myId, listId, listIdLen) == 0) {
                    foundAllowlistEntry = 1;
                    break;
                }
            }
            if (!foundAllowlistEntry) {
                SFW_PRINT(SFW_PLVL_EMERG
                          "failed %s allowlist check\n", typeName);
                return 0;
            }
        }
    }
    else {
        SFW_PRINT(SFW_PLVL_EMERG
                  "checking %s allowlist against unknown %s...\n",
                  typeName, typeName);

        if (numEntries != 0) {
            SFW_PRINT(SFW_PLVL_EMERG
                      "non-empty (%d entries) Sonos %s allowlist; dying...\n",
                      (int)numEntries, typeName);
            return 0;
        }
    }

    SFW_PRINT(SFW_PLVL_INFO "allowlist check completed\n");
    (void)typeName;
    return 1;
}


static inline int
check_sonos_firmware_allowlist(const SonosFirmwareAllowlistHeader* hdr, size_t entriesSize)
{
    return check_sonos_firmware_allowlist_ex(hdr, (const uint8_t *)&hdr[1], entriesSize);
}

#ifdef SFW_INCLUDE_FIND_AND_CHECK_SONOS_ALLOWLIST

int find_and_check_sonos_allowlist(const uint8_t* buf, size_t bufLen,
                                   int isKernel, int isPermissive)
{
    static const uint8_t magicUBoot[] = { SONOS_FWA_MAGIC_INIT_INVERTED };
    static const uint8_t magicKernel[] = { SONOS_FWA_MAGIC_INIT_UNCOMPRESSED_COPY_INVERTED };
    uint8_t magic[SONOS_FWA_MAGIC_LEN];
    const uint8_t* pMagic;
    const uint8_t* p;
    const uint8_t* pEnd;
    const uint8_t* pEntries = NULL;
    size_t i;
    SonosFirmwareAllowlistHeader allowlistHdr;

    memset(&allowlistHdr, 0, sizeof(allowlistHdr));

    if (bufLen < sizeof(SonosFirmwareAllowlistHeader)) {
        return 0;
    }

    pMagic = isKernel ? magicKernel : magicUBoot;
    for (i = 0; i < sizeof(magic); i++) {
        magic[i] = pMagic[i] ^ 0xff;
    }

    pEnd = &buf[bufLen - sizeof(SonosFirmwareAllowlistHeader)];

    for (p = buf; p <= pEnd; p++) {
        if (memcmp(p, magic, sizeof(magic)) == 0) {
            if (pEntries) {
                SFW_PRINT(SFW_PLVL_EMERG "Duplicate allowlist\n");
                return 0;
            }
            memcpy(&allowlistHdr, p, sizeof(allowlistHdr));
            pEntries = p + sizeof(SonosFirmwareAllowlistHeader);
        }
    }

    if (!pEntries) {
        if (isPermissive) {
            return 1;
        }
        return 0;
    }

    return check_sonos_firmware_allowlist_ex(&allowlistHdr, pEntries,
                                             &buf[bufLen] - pEntries);
}

#endif

