/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 *
 * sonos_attr_serialize.c.inc: Sonos attribute serialization
 */

#include "sonos_attr.h"
#include "sonos_nasn_serialize.h"
#include "sonos_signature_common.h"

static int sonosAttributeSerializeValue(const SonosAttribute* a,
                                        uint8_t** buf_p,
                                        const uint8_t* end)
{
    uint8_t* buf = *buf_p;

    switch SONOS_ATTR_ID(a->attributeId) {
    default:
    case SONOS_ATTR_ID_PADDING:
        PUT_BUF(a->x.attributeValue, a->attributeValueLen);
        break;

    case SONOS_ATTR_ID_MESSAGE_DIGEST:
    {
        PUT_BUF(a->x.attributeValue, a->attributeValueLen);
        break;
    }

    case SONOS_ATTR_ID_CONTENT_TYPE:
    {
        SonosAttributeContentType ct;

        ct = a->x.ct;
        ct.contentType = SONOS_NASN_CPU_TO_BE32(ct.contentType);

        PUT_BUF(&ct, sizeof ct);
        break;
    }

    case SONOS_ATTR_ID_SRK_REVOKE:
    {
        PUT_BUF(a->x.attributeValue, a->attributeValueLen);
        break;
    }

    case SONOS_ATTR_ID_ALLOWLIST:
    {
        size_t alEntriesLen;
        const SonosFirmwareAllowlistHeader *hdr = &a->x.al.header;

        switch (hdr->allowlistType) {
            case SONOS_FWA_TYPE_CPUID:
                alEntriesLen = hdr->numEntries * SONOS_FWA_LEN_CPUID;
                break;
            case SONOS_FWA_TYPE_SERIAL:
                alEntriesLen = hdr->numEntries * SONOS_FWA_LEN_SERIAL;
                break;
            case SONOS_FWA_TYPE_CPUID16:
                alEntriesLen = hdr->numEntries * SONOS_FWA_LEN_CPUID16;
                break;
            default:
                return 0;
        }

        PUT_BUF(hdr->magic, sizeof(hdr->magic));
        PUT_ENUM(hdr->allowlistType, sonosSignatureIsValidAllowlistType);
        PUT_INT(hdr->numEntries);
        PUT_BUF(a->x.al.entriesPtr, alEntriesLen);
        break;
    }

    case SONOS_ATTR_ID_AR_VERSION:
    {
        SonosAttributeArVersion arv;

        arv = a->x.arv;
        arv.arVersion = SONOS_NASN_CPU_TO_BE32(arv.arVersion);

        PUT_BUF(&arv, sizeof arv);
        break;
    }

    }

    *buf_p = buf;
    return 1;
}

int sonosAttributeSerialize(const SonosAttribute* a,
                            uint8_t** buf_p,
                            const uint8_t* end)
{
    uint8_t* buf = *buf_p;

    PUT_INT(a->attributeId);
    if (SONOS_ATTR_IS_CRITICAL(a->attributeId) &&
        !sonosIsKnownAttribute(a->attributeId)) {
        return 0;
    }

    PUT_INT(a->attributeValueLen);

    if (!sonosAttributeSerializeValue(a, &buf, end)) {
        return 0;
    }

    *buf_p = buf;
    return 1;
}
