/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 *
 * sonos_attr_parse.c.inc: Sonos attribute parsing
 */

#include "sonos_attr.h"
#include "sonos_nasn_parse.h"
#include "sonos_signature_common.h"

static int sonosAttributeParseValue(SonosAttribute* a)
{
    switch SONOS_ATTR_ID(a->attributeId) {
    case SONOS_ATTR_ID_PADDING:
        break;

    case SONOS_ATTR_ID_MESSAGE_DIGEST:
    {
        if (a->attributeValueLen < 1 ||
            !sonosSignatureIsValidDigestAlg(a->x.md.alg) ||
            a->attributeValueLen != sizeof(a->x.md.alg) +
                                    SONOS_DIGEST_ALG_LEN(a->x.md.alg)) {
            return 0;
        }

        break;
    }

    case SONOS_ATTR_ID_CONTENT_TYPE:
    {
        if (a->attributeValueLen != sizeof(SonosAttributeContentType)) {
            return 0;
        }

        a->x.ct.contentType = SONOS_NASN_BE32_TO_CPU(a->x.ct.contentType);

        break;
    }

    case SONOS_ATTR_ID_SRK_REVOKE:
    {
        if (a->attributeValueLen != sizeof(SonosAttributeSrkRevoke)) {
            return 0;
        }

        break;
    }

    case SONOS_ATTR_ID_AR_VERSION:
    {
        if (a->attributeValueLen != sizeof(SonosAttributeArVersion)) {
            return 0;
        }

        a->x.arv.arVersion = SONOS_NASN_BE32_TO_CPU(a->x.arv.arVersion);

        break;
    }

    default:
        return 0;
    }

    return 1;
}

int
sonosAttributeParseAllowlist(SonosAttribute *sa,
                             const uint8_t **buf_p,
                             const uint8_t *end)
{
    SonosAttributeAllowlist al;
    size_t alEntriesLen;
    const uint8_t *buf = *buf_p;
    static const uint8_t magicInverted[] = { SONOS_FWA_MAGIC_INIT_INVERTED };
    uint8_t magic[SONOS_FWA_MAGIC_LEN];
    size_t i;

    GET_BUF(al.header.magic, sizeof(al.header.magic));
    for (i = 0; i < sizeof(magic); i++) {
        magic[i] = magicInverted[i] ^ 0xff;
    }
    if (memcmp(al.header.magic, magic, sizeof(magic)) != 0) {
        return 0;
    }

    GET_ENUM(al.header.allowlistType, sonosSignatureIsValidAllowlistType);
    GET_INT(al.header.numEntries);
    switch (al.header.allowlistType) {
        case SONOS_FWA_TYPE_CPUID:
            alEntriesLen = al.header.numEntries * SONOS_FWA_LEN_CPUID;
            break;
        case SONOS_FWA_TYPE_SERIAL:
            alEntriesLen = al.header.numEntries * SONOS_FWA_LEN_SERIAL;
            break;
        case SONOS_FWA_TYPE_CPUID16:
            alEntriesLen = al.header.numEntries * SONOS_FWA_LEN_CPUID16;
            break;
        default:
            return 0;
    }
    GET_BUF_PTR(al.entriesPtr, alEntriesLen);

    sa->x.al = al;

    *buf_p = buf;
    return 1;
}

int sonosAttributeParse(SonosAttribute* a,
                        const uint8_t** buf_p,
                        const uint8_t* end)
{
    const uint8_t* buf = *buf_p;

    GET_INT(a->attributeId);
    if (SONOS_ATTR_ID(a->attributeId) == SONOS_ATTR_ID_ALLOWLIST) {
        GET_INT(a->attributeValueLen);
        *buf_p = buf;
        return sonosAttributeParseAllowlist(a, buf_p, end);
    }

    GET_INT_BOUNDED(a->attributeValueLen, SONOS_ATTR_MAX_VALUE_LEN);
    GET_BUF(a->x.attributeValue, a->attributeValueLen);

    if (sonosIsKnownAttribute(a->attributeId)) {
        if (!sonosAttributeParseValue(a)) {
            return 0;
        }
    }
    else {
        if (SONOS_ATTR_IS_CRITICAL(a->attributeId)) {
            return 0;
        }
    }

    *buf_p = buf;
    return 1;
}

