/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 *
 * sonos_attr.c.inc: Sonos attribute common code
 */

#include "sonos_attr.h"

int
sonosIsKnownAttribute(uint32_t id)
{
    id &= ~SONOS_ATTR_CRIT_BIT;

    return 0 < id && id <= SONOS_ATTR_ID_MAX;
}

int
sonosHasDuplicateAttribute(uint32_t numUnsignedAttrs,
                           const SonosAttribute unsignedAttrs[],
                           uint32_t numSignedAttrs,
                           const SonosAttribute signedAttrs[])
{
    uint32_t i, j;
    uint32_t iAttr, jAttr;

    for (i = 0; i < numUnsignedAttrs; i++) {
        iAttr = SONOS_ATTR_ID(unsignedAttrs[i].attributeId);

        for (j = i + 1; j < numUnsignedAttrs; j++) {
            jAttr = SONOS_ATTR_ID(unsignedAttrs[j].attributeId);
            if (iAttr == jAttr) {
                return 1;
            }
        }
        for (j = 0; j < numSignedAttrs; j++) {
            jAttr = SONOS_ATTR_ID(signedAttrs[j].attributeId);
            if (iAttr == jAttr) {
                return 1;
            }
        }
    }

    for (i = 0; i < numSignedAttrs; i++) {
        iAttr = SONOS_ATTR_ID(signedAttrs[i].attributeId);

        for (j = i + 1; j < numSignedAttrs; j++) {
            jAttr = SONOS_ATTR_ID(signedAttrs[j].attributeId);
            if (iAttr == jAttr) {
                return 1;
            }
        }
    }

    return 0;
}

const char *
sonosContentTypeToString(SonosContentType_t ct)
{
    const char* ctStrings[] =
    {
        "invalid",
        "UPD",
        "Kernel",
        "FPGA",
        "M4",
        "LK",
        "TEE",
        "U-Boot",
        "BL Combined",
        "BL30",
        "BL31",
        "BL32",
        "BL33",
        "FIP",
        "FIRMWARE",
        "ROOT_CERTS",
    };

#if defined(STATIC_ASSERT) && defined(ARRAY_SIZE)
    STATIC_ASSERT(ARRAY_SIZE(ctStrings) == SONOS_CT_MAX + 1);
#endif

    return (ct <= SONOS_CT_MAX) ? ctStrings[ct] : "unknown";
}

#ifdef SONOS_ARCH_ATTR_IS_BUILD_TOOLS

#include <inttypes.h>
#include <string.h>

static void
sonosAttributeDumpHex(FILE* f, const char* indent,
                      const void* vbuf, size_t bufLen)
{
    size_t i, mod, finalChar;
    const uint8_t *buf = vbuf;

    for (i = 0; i < bufLen; i++) {
        mod = i % 8;
        finalChar = i == bufLen - 1;

        fprintf(f, "%s0x%02x%s%s",
                mod == 0 ? indent : "",
                buf[i],
                finalChar ? "" : ",",
                mod == 7 || finalChar ? "\n" : " ");
    }
}

static const char* ctStringsLower[] =
{
    "invalid",
    "upd",
    "kernel",
    "fpga",
    "m4",
    "lk",
    "tee",
    "uboot",
    "bl_combined",
    "bl30",
    "bl31",
    "bl32",
    "bl33",
    "fip",
    "firmware",
    "root_certs"
};

#if defined(STATIC_ASSERT) && defined(ARRAY_SIZE)
STATIC_ASSERT(ARRAY_SIZE(ctStringsLower) == SONOS_CT_MAX + 1);
#endif

SonosContentType_t
sonosContentTypeFromString(const char* s)
{
    size_t i;

    for (i = 1; i < (1+SONOS_CT_MAX); i++) {
        if (strcmp(s, ctStringsLower[i]) == 0) {
            return (SonosContentType_t)i;
        }
    }
    return SONOS_CT_INVALID;
}

void
sonosAttributePrint(FILE* f, const SonosAttribute* a)
{
    uint32_t id = SONOS_ATTR_ID(a->attributeId);
    uint32_t len = a->attributeValueLen;
    const char* attrName = NULL;
    char attrNameBuf[32];
    const char* attrPrettyValue = NULL;
    char attrPrettyValueBuf[64];
    const char* alTypeStrings[] =
    {
         "cpuid", "serial", "cpuid16"
    };
    const size_t alEntrySize[] =
    {
         SONOS_FWA_LEN_CPUID,
         SONOS_FWA_LEN_SERIAL,
         SONOS_FWA_LEN_CPUID16
    };

    switch (id) {
    case SONOS_ATTR_ID_PADDING:
        attrName = "padding";
        break;
    case SONOS_ATTR_ID_MESSAGE_DIGEST:
        attrName = "message digest";
        break;
    case SONOS_ATTR_ID_CONTENT_TYPE:
    {
        SonosContentType_t contentType;
        attrName = "content-type";
        attrPrettyValue = attrPrettyValueBuf;
        contentType = a->x.ct.contentType;
        if (contentType <= SONOS_CT_MAX) {
            snprintf(attrPrettyValueBuf, sizeof attrPrettyValueBuf, "%s",
                     ctStringsLower[contentType]);
        }
        else {
            snprintf(attrPrettyValueBuf, sizeof attrPrettyValueBuf,
                     "%s (%"PRIu32")",
                     "unknown", contentType);
        }
        break;
    }
    case SONOS_ATTR_ID_SRK_REVOKE:
    {
        SonosSrkRevoke_t srkValue;
        attrName = "SRK revoke";
        attrPrettyValue = attrPrettyValueBuf;
        srkValue = a->x.sr.srkRevokeFuse;
        snprintf(attrPrettyValueBuf, sizeof attrPrettyValueBuf,
                 "0x%x", (unsigned)srkValue);
        break;
    }
    case SONOS_ATTR_ID_ALLOWLIST:
        attrName = "allowlist";
        break;
    case SONOS_ATTR_ID_AR_VERSION:
    {
        attrName = "AR version";
        attrPrettyValue = attrPrettyValueBuf;
        snprintf(attrPrettyValueBuf, sizeof attrPrettyValueBuf,
                 "0x%" PRIu32, a->x.arv.arVersion);
        break;
    }
    default:
        attrName = attrNameBuf;
        snprintf(attrNameBuf, sizeof attrNameBuf, "unknown (%"PRIu32")", id);
        break;
    }

    fprintf(f, "%s %s(%"PRIu32" bytes):\n",
            attrName,
            SONOS_ATTR_IS_CRITICAL(a->attributeId) ? "(critical) " : "",
            len);
    if (id != SONOS_ATTR_ID_ALLOWLIST) {
        if (attrPrettyValue) {
            fprintf(f, "  pretty value: %s\n", attrPrettyValue);
        }
        fprintf(f, "  raw value:\n");
        sonosAttributeDumpHex(f, "    ", a->x.attributeValue, len);
    }
    else {
        SonosAttributeAllowlist al;
        size_t i, entrySize = 0;
        const uint8_t *buf;

        al = a->x.al;
        switch (al.header.allowlistType) {
            case SONOS_FWA_TYPE_CPUID:
            case SONOS_FWA_TYPE_SERIAL:
            case SONOS_FWA_TYPE_CPUID16:
                fprintf(f, "  pretty value: %s %"PRIu32"\n",
                    alTypeStrings[al.header.allowlistType-1],
                    al.header.numEntries);
                entrySize = alEntrySize[al.header.allowlistType-1];
                break;
            default:
                fprintf(f, "unknown type %d\n", al.header.allowlistType);
                return;
        }
        buf = al.entriesPtr;
        for (i = 0; i < al.header.numEntries; ++i) {
            sonosAttributeDumpHex(f, "    ", buf, entrySize);
            buf += entrySize;
        }
    }
}

#endif
