/*
 * Copyright (c) 2003-2021, Sonos, Inc.  All rights reserved.
 *
 * SPDX-License-Identifier: GPL-2.0
 */

#ifndef UPGRADE_MDP_H
#define UPGRADE_MDP_H

#include "sonos_stdint.h"

#ifdef SONOS_ARCH_ATTR_IS_HARDWARE
#if defined(SONOS_ARCH_FILLMORE)
#define MDP_DEVICE "/dev/mtd1"
#elif defined(SONOS_ARCH_CONNECTX) || \
      defined(SONOS_ARCH_FENWAY) || defined(SONOS_ARCH_LIMELIGHT)
#define MDP_DEVICE "/dev/mtd/0"
#elif defined(SONOS_ARCH_ATTR_SUPPORTS_SECURE_BOOT)
#define MDP_DEVICE "/dev/mdp"
#endif
#else
#define MDP_DEVICE "/dev/mtd/0"
#endif

#define MDP_MAGIC 0xce10e47d
#define MDP_MAGIC2 0xca989b4a
#define MDP_MAGIC2_ENC 0xfa87b921
#define MDP_MAGIC3 0xcba979f0
#define MDP_UNKNOWN 0
#define MDP_RESERVED 0xffffffff
#define MDP_VENDOR_RINCONNETWORKS 0x1
#define MDP_MAX_VENDOR 0x1


#define MDP_MODEL_ZP                        1

#define MDP_SUBMODEL_ZP_ES2                 1
#define MDP_SUBMODEL_ZP_TS1                 2
#define MDP_SUBMODEL_ZP_TS2                 3
#define MDP_SUBMODEL_ZP_REDROCKS            16

#define MDP_MODEL_HH                        2

#define MDP_MODEL_WDCR                      3

#define MDP_SUBMODEL_WDCR_6_1               2
#define MDP_SUBMODEL_WDCR                   3

#define MDP_MODEL_MDCR                      4

#define MDP_SUBMODEL_MDCR_10_10             4
#define MDP_SUBMODEL_MDCR                   5

#define MDP_MODEL_LINK                      5

#define MDP_MODEL_WOODSTOCK                 6

#define MDP_MODEL_CASBAH                    7

#define MDP_MODEL_FENWAY                    8

#define MDP_SUBMODEL_FENWAY                 1
#define MDP_SUBMODEL_FENWAY_ANVIL           2
#define MDP_SUBMODEL_FENWAY_AMOEBA          3
#define MDP_SUBMODEL_FENWAY_ANVILLP         6
#define MDP_SUBMODEL_FENWAY_AMOEBALP        7

#define MDP_REVISION_FENWAY_128MB           3

#define MDP_REVISION_ANVIL_P1               1
#define MDP_REVISION_ANVIL_P2               2
#define MDP_REVISION_ANVIL_P3               3
#define MDP_REVISION_ANVIL_DVT              4
#define MDP_REVISION_ANVIL_128MB            5

#define MDP_REVISION_ANVILLP                6

#define MDP_REVISION_AMOEBA_128MB           7

#define MDP_MODEL_LIMELIGHT                 9

#define MDP_SUBMODEL_LIMELIGHT              1

#define MDP_REVISION_LIMELIGHT_EVT          6
#define MDP_REVISION_LIMELIGHT_MMA_SWAP     11

#define MDP_MODEL_ICR                       10

#define MDP_SUBMODEL_ICR_11                 7
#define MDP_SUBMODEL_ICR                    8

#define MDP_MODEL_ACR                       11

#define MDP_SUBMODEL_ACR_7                  7
#define MDP_SUBMODEL_ACR                    8

#define MDP_MODEL_FILLMORE                  12

#define MDP_SUBMODEL_FILLMORE               1

#define MDP_REVISION_FILLMORE_BB2           1
#define MDP_REVISION_FILLMORE_P1            2

#define MDP_MODEL_ENCORE                    13

#define MDP_SUBMODEL_ENCORE                 1

#define MDP_REVISION_ENCORE_PROTO3          3
#define MDP_REVISION_ENCORE_PROTO4          4
#define MDP_REVISION_ENCORE_EVT1            4
#define MDP_REVISION_ENCORE_EVT2            5
#define MDP_REVISION_ENCORE_DVT             6
#define MDP_REVISION_ENCORE_PVT             7
#define MDP_REVISION_ENCORE_BEETHOVEN       8
#define MDP_REVISION_ENCORE_RF_REGULATOR    9

#define MDP_MODEL_SOLBASE                   14

#define MDP_SUBMODEL_SOLBASE                1

#define MDP_REVISION_SOLBASE_B1             1
#define MDP_REVISION_SOLBASE_P0             1
#define MDP_REVISION_SOLBASE_M0             1
#define MDP_REVISION_SOLBASE_M2             2
#define MDP_REVISION_SOLBASE_PROTO3         3
#define MDP_REVISION_SOLBASE_PRE_PROTO4     4
#define MDP_REVISION_SOLBASE_PROTO4         5
#define MDP_REVISION_SOLBASE_P4_RETROFIT    6
#define MDP_REVISION_SOLBASE_VAL1           7
#define MDP_REVISION_SOLBASE_PROTO5         8
#define MDP_REVISION_SOLBASE_EVT            9
#define MDP_REVISION_SOLBASE_DVT            10
#define MDP_REVISION_SOLBASE_PVT            11
#define MDP_REVISION_SOLBASE_WILDLANDS      12
#define MDP_REVISION_SOLBASE_IMX6SX_REVC    13

#define MDP_MODEL_TEST_CONTROLLER           15

#define MDP_MODEL_WEMBLEY                   16

#define MDP_SUBMODEL_WEMBLEY_WEMBLEY        3
#define MDP_SUBMODEL_WEMBLEY_ZPS5           4
#define MDP_SUBMODEL_WEMBLEY_EDEN           5

#define MDP_MODEL_CONNECTX                  17

#define MDP_SUBMODEL_CONNECTX_WEMBLEY       MDP_SUBMODEL_WEMBLEY_WEMBLEY
#define MDP_SUBMODEL_CONNECTX_ZPS5          MDP_SUBMODEL_WEMBLEY_ZPS5
#define MDP_SUBMODEL_CONNECTX_EDEN          MDP_SUBMODEL_WEMBLEY_EDEN

#define MDP_MODEL_RESERVED1                 18
#define MDP_MODEL_RESERVED2                 19

#define MDP_MODEL_ROYALE                    20

#define MDP_SUBMODEL_ROYALE                 1
#define MDP_SUBMODEL_ROYALE_MARQUEE         2
#define MDP_SUBMODEL_ROYALE_LARGO           3

#define MDP_REVISION_ROYALE_PROTO1          1
#define MDP_REVISION_ROYALE_PROTO2          2
#define MDP_REVISION_ROYALE_PROTO3          3
#define MDP_REVISION_ROYALE_PROTO3_CAPS     4
#define MDP_REVISION_ROYALE_EVTDVT          5
#define MDP_REVISION_ROYALE_PVT             6
#define MDP_REVISION_ROYALE_BL_TRANSDUCERS  7

#define MDP_REVISION_MARQUEE_P2B            0
#define MDP_REVISION_MARQUEE_EVT            1
#define MDP_REVISION_MARQUEE_DVT2           2
#define MDP_REVISION_MARQUEE_PVT            3
#define MDP_REVISION_MARQUEE_AUTH_3_0       4

#define MDP_REVISION_LARGO_P2               0
#define MDP_REVISION_LARGO_EVT              1
#define MDP_REVISION_LARGO_DVT2             2
#define MDP_REVISION_LARGO_PVT              3
#define MDP_REVISION_LARGO_AUTH_3_0         4

#define MDP_MODEL_BOOTLEG                   21

#define MDP_SUBMODEL_BOOTLEG                1

#define MDP_REVISION_BOOTLEG_PROTO1         1
#define MDP_REVISION_BOOTLEG_PROTO1A        2
#define MDP_REVISION_BOOTLEG_PROTO2A        3
#define MDP_REVISION_BOOTLEG_PROTO2B        4
#define MDP_REVISION_BOOTLEG_EVT            5
#define MDP_REVISION_BOOTLEG_DVT1           6
#define MDP_REVISION_BOOTLEG_DVT2           7
#define MDP_REVISION_BOOTLEG_PVT            8

#define MDP_MODEL_PARAMOUNT                 22

#define MDP_SUBMODEL_PARAMOUNT              1

#define MDP_REVISION_PARAMOUNT_PROTO1       1
#define MDP_REVISION_PARAMOUNT_PROTO2B      2
#define MDP_REVISION_PARAMOUNT_EVT          3
#define MDP_REVISION_PARAMOUNT_PVT          4
#define MDP_REVISION_PARAMOUNT_MP_SOLOX     5

#define MDP_MODEL_ELREY                     23

#define MDP_SUBMODEL_ELREY                  1

#define MDP_REVISION_ELREY_PROTO1           1
#define MDP_REVISION_ELREY_PROTO1B          2
#define MDP_REVISION_ELREY_PROTO2A          3
#define MDP_REVISION_ELREY_PRE_EMI          4
#define MDP_REVISION_ELREY_EMI              5
#define MDP_REVISION_ELREY_PRE_PROTO2B      6
#define MDP_REVISION_ELREY_PROTO2B          7
#define MDP_REVISION_ELREY_EVT              8
#define MDP_REVISION_ELREY_DVT              9
#define MDP_REVISION_ELREY_PVT              10
#define MDP_REVISION_ELREY_AUTH_3_0         11
#define MDP_REVISION_ELREY_AUTH_REMOVE      12
#define MDP_REVISION_ELREY_PMIC             13
#define MDP_REVISION_ELREY                  MDP_REVISION_ELREY_PMIC

#define MDP_MODEL_HIDEOUT                   24

#define MDP_SUBMODEL_HIDEOUT                1

#define MDP_REVISION_HIDEOUT_BS1            2
#define MDP_REVISION_HIDEOUT_PROTO1A        3
#define MDP_REVISION_HIDEOUT_PROTO1B        4
#define MDP_REVISION_HIDEOUT_PROTO1C        5
#define MDP_REVISION_HIDEOUT_PROTO1D        6
#define MDP_REVISION_HIDEOUT_PROTO2A        7
#define MDP_REVISION_HIDEOUT_PROTO2B        8
#define MDP_REVISION_HIDEOUT_PROTO2B_MLCC   9
#define MDP_REVISION_HIDEOUT_EVT            10
#define MDP_REVISION_HIDEOUT_DVT            11
#define MDP_REVISION_HIDEOUT_PVT            12
#define MDP_REVISION_HIDEOUT_AUTH_3_0       13
#define MDP_REVISION_HIDEOUT_PMIC           14
#define MDP_REVISION_HIDEOUT_ADP8863        15
#define MDP_REVISION_HIDEOUT                MDP_REVISION_HIDEOUT_ADP8863

#define MDP_MODEL_DHUEZ                     25

#define MDP_SUBMODEL_DHUEZ                  1

#define MDP_REVISION_DHUEZ_P1A              1
#define MDP_REVISION_DHUEZ_P1B              2
#define MDP_REVISION_DHUEZ_P2A              3
#define MDP_REVISION_DHUEZ_P2B              4
#define MDP_REVISION_DHUEZ_EVT              5
#define MDP_REVISION_DHUEZ_DVT              6
#define MDP_REVISION_DHUEZ_PVT              7
#define MDP_REVISION_DHUEZ_WIFI_BIN         8
#define MDP_REVISION_DHUEZ_CSR8311_SUPPORT  9
#define MDP_REVISION_DHUEZ_BMA420_SUPPORT   10
#define MDP_REVISION_DHUEZ                  MDP_REVISION_DHUEZ_BMA420_SUPPORT

#define MDP_MODEL_TUPELO                    26

#define MDP_SUBMODEL_TUPELO                 1
#define MDP_SUBMODEL_TUPELO_GOLDENEYE       2

#define MDP_REVISION_TUPELO_EB0             1
#define MDP_REVISION_TUPELO_EB2             2
#define MDP_REVISION_TUPELO_QB1             3
#define MDP_REVISION_TUPELO_QB2             4
#define MDP_REVISION_TUPELO_QB3             5
#define MDP_REVISION_TUPELO_CR1             6
#define MDP_REVISION_TUPELO_AUTH_3_0        7
#define MDP_REVISION_TUPELO_AUTH_REMOVE     8
#define MDP_REVISION_TUPELO_KAPITAL_ONE_EB0 9
#define MDP_REVISION_TUPELO_LED_NTC	        10
#define MDP_REVISION_TUPELO                 MDP_REVISION_TUPELO_LED_NTC

#define MDP_REVISION_GOLDENEYE_P0           1
#define MDP_REVISION_GOLDENEYE              MDP_REVISION_GOLDENEYE_P0

#define MDP_MODEL_APOLLO                    27

#define MDP_SUBMODEL_APOLLO                 1
#define MDP_SUBMODEL_APOLLO_APOLLOX         2

#define MDP_REVISION_APOLLO_P1A             1
#define MDP_REVISION_APOLLO_P1B             2
#define MDP_REVISION_APOLLO_P1C             3
#define MDP_REVISION_APOLLO_P2A             4
#define MDP_REVISION_APOLLO_P2B             5
#define MDP_REVISION_APOLLO_EVT             6
#define MDP_REVISION_APOLLO_DVT             7
#define MDP_REVISION_APOLLO_PVT             8
#define MDP_REVISION_APOLLO_AUTH_REMOVE     9
#define MDP_REVISION_APOLLO_PSOC            10
#define MDP_REVISION_APOLLO_RICHTEK         11
#define MDP_REVISION_APOLLO_LTR311          12
#define MDP_REVISION_APOLLO_TAS_LTR         13
#define MDP_REVISION_APOLLO                 MDP_REVISION_APOLLO_TAS_LTR

#define MDP_REVISION_APOLLOX_P0             1
#define MDP_REVISION_APOLLOX_PVT            2
#define MDP_REVISION_APOLLOX_AUTH_REMOVE    3
#define MDP_REVISION_APOLLOX_PSOC           4
#define MDP_REVISION_APOLLOX_RICHTEK        5
#define MDP_REVISION_APOLLOX_LTR311         6
#define MDP_REVISION_APOLLOX_TAS_LTR        7
#define MDP_REVISION_APOLLOX                MDP_REVISION_APOLLOX_TAS_LTR

#define MDP_MODEL_CHAPLIN                   28

#define MDP_SUBMODEL_CHAPLIN                1
#define MDP_SUBMODEL_CHAPLIN_KAPITAL        2

#define MDP_REVISION_CHAPLIN_BS1            1
#define MDP_REVISION_CHAPLIN_PROTO2A        2
#define MDP_REVISION_CHAPLIN_PROTO2B        3
#define MDP_REVISION_CHAPLIN_EVT            4
#define MDP_REVISION_CHAPLIN_EVT2           5
#define MDP_REVISION_CHAPLIN_PVT            6
#define MDP_REVISION_CHAPLIN                MDP_REVISION_CHAPLIN_PVT

#define MDP_REVISION_KAPITAL_P0             1
#define MDP_REVISION_KAPITAL_EB1            2
#define MDP_REVISION_KAPITAL_EB2            3
#define MDP_REVISION_KAPITAL_QB             4
#define MDP_REVISION_KAPITAL_CR             5
#define MDP_REVISION_KAPITAL_PSOC           6
#define MDP_REVISION_KAPITAL                MDP_REVISION_KAPITAL_PSOC

#define MDP_MODEL_NEPTUNE                   29

#define MDP_SUBMODEL_NEPTUNE                1

#define MDP_REVISION_NEPTUNE_BS1            1
#define MDP_REVISION_NEPTUNE_PROTO1A        2
#define MDP_REVISION_NEPTUNE_PROTO2A        3
#define MDP_REVISION_NEPTUNE_PROTO2B        4
#define MDP_REVISION_NEPTUNE_EVT            5
#define MDP_REVISION_NEPTUNE_DVT            6
#define MDP_REVISION_NEPTUNE_PVT            7
#define MDP_REVISION_NEPTUNE_PSOC4247       8
#define MDP_REVISION_NEPTUNE_CS43130        9
#define MDP_REVISION_NEPTUNE_CS43198        10
#define MDP_REVISION_NEPTUNE                MDP_REVISION_NEPTUNE_CS43198

#define MDP_MODEL_DOMINO                    30

#define MDP_SUBMODEL_DOMINO                 1

#define MDP_REVISION_DOMINO_PROTO1          1
#define MDP_REVISION_DOMINO_PROTO2A         2
#define MDP_REVISION_DOMINO_PROTO2B         3
#define MDP_REVISION_DOMINO_DVT             4
#define MDP_REVISION_DOMINO_PVT             5
#define MDP_REVISION_DOMINO_PMIC            6
#define MDP_REVISION_DOMINO_PCM1690         7
#define MDP_REVISION_DOMINO_BMA420          8
#define MDP_REVISION_DOMINO                 MDP_REVISION_DOMINO_BMA420

#define MDP_MODEL_TRIDENT                   31

#define MDP_SUBMODEL_TRIDENT                1

#define MDP_REVISION_TRIDENT_RY8            1
#define MDP_REVISION_TRIDENT_P1             2
#define MDP_REVISION_TRIDENT_P2A            3
#define MDP_REVISION_TRIDENT_P2B            4
#define MDP_REVISION_TRIDENT                MDP_REVISION_TRIDENT_P2B

#define MDP_MODEL_VERTIGO                   32

#define MDP_SUBMODEL_VERTIGO                1

#define MDP_REVISION_VERTIGO_PROTO0         1
#define MDP_REVISION_VERTIGO_PROTO2A        2
#define MDP_REVISION_VERTIGO_DVT            3
#define MDP_REVISION_VERTIGO_PVT            4
#define MDP_REVISION_VERTIGO_PSOC4247       5
#define MDP_REVISION_VERTIGO_PCM5101        6
#define MDP_REVISION_VERTIGO_SH_G2_FLASH    7
#define MDP_REVISION_VERTIGO                MDP_REVISION_VERTIGO_SH_G2_FLASH

#define MDP_MODEL_MONACO                    33

#define MDP_SUBMODEL_MONACO                 1
#define MDP_SUBMODEL_MONACO_MONACOSL        2

#define MDP_REVISION_MONACO_PROTO0          1
#define MDP_REVISION_MONACO_PROTO1          2
#define MDP_REVISION_MONACO_PROTO2A         3
#define MDP_REVISION_MONACO_EVT             4
#define MDP_REVISION_MONACO_DVT             5
#define MDP_REVISION_MONACO_DVTR2           6
#define MDP_REVISION_MONACO_PREPVT          7
#define MDP_REVISION_MONACO_PVT             8
#define MDP_REVISION_MONACO_AMPS            9
#define MDP_REVISION_MONACO_PICFREQ         10
#define MDP_REVISION_MONACO                 MDP_REVISION_MONACO_PICFREQ

#define MDP_REVISION_MONACOSL_EVT           1
#define MDP_REVISION_MONACOSL_DVT           2
#define MDP_REVISION_MONACOSL_DVTR2         3
#define MDP_REVISION_MONACOSL_PREPVT        4
#define MDP_REVISION_MONACOSL_PVT           5
#define MDP_REVISION_MONACOSL_AMPS          6
#define MDP_REVISION_MONACOSL_PICFREQ       7
#define MDP_REVISION_MONACOSL               MDP_REVISION_MONACOSL_PICFREQ

#define MDP_MODEL_TITAN                     34

#define MDP_SUBMODEL_TITAN                  0
#define MDP_SUBMODEL_TITAN_WOWP             1
#define MDP_SUBMODEL_TITAN_WOWT             2
#define MDP_SUBMODEL_TITAN_WOWB             3
#define MDP_SUBMODEL_TITAN_WOWG             4

#define MDP_REVISION_TITAN_PROTO0           1
#define MDP_REVISION_TITAN_PROTO1A          2
#define MDP_REVISION_TITAN_PROTO1A_MT7613   3
#define MDP_REVISION_TITAN_PROTO1B          4
#define MDP_REVISION_TITAN_PROTO2A          5
#define MDP_REVISION_TITAN_EVT              6
#define MDP_REVISION_TITAN_DVT              7
#define MDP_REVISION_TITAN_PVT              8
#define MDP_REVISION_WOWP_PVT               MDP_REVISION_TITAN_PVT
#define MDP_REVISION_WOWP                   MDP_REVISION_WOWP_PVT
#define MDP_REVISION_WOWT_PVT               MDP_REVISION_TITAN_PVT
#define MDP_REVISION_WOWT                   MDP_REVISION_WOWT_PVT
#define MDP_REVISION_WOWB_PVT               MDP_REVISION_TITAN_PVT
#define MDP_REVISION_WOWB                   MDP_REVISION_WOWB_PVT
#define MDP_REVISION_WOWG_PVT               MDP_REVISION_TITAN_PVT
#define MDP_REVISION_WOWG                   MDP_REVISION_WOWG_PVT

#define MDP_MODEL_BRAVO                     35

#define MDP_SUBMODEL_BRAVO                  1
#define MDP_SUBMODEL_BRAVO_BRAVOSL          2

#define MDP_REVISION_BRAVO_PROTO0           1
#define MDP_REVISION_BRAVO_PROTO1           2
#define MDP_REVISION_BRAVO_P2A              3
#define MDP_REVISION_BRAVO_EVT              4
#define MDP_REVISION_BRAVO_DVT              5
#define MDP_REVISION_BRAVO_PVT              6
#define MDP_REVISION_BRAVO                  MDP_REVISION_BRAVO_PVT
#define MDP_REVISION_BRAVOSL                MDP_REVISION_BRAVO_PROTO1

#define MDP_MODEL_FURY                      36

#define MDP_SUBMODEL_FURY                   1

#define MDP_REVISION_FURY_PROTO0            1
#define MDP_REVISION_FURY_PROTO1            2
#define MDP_REVISION_FURY_P2A               3
#define MDP_REVISION_FURY_P2B               4
#define MDP_REVISION_FURY_EVT               5
#define MDP_REVISION_FURY_DVT               6
#define MDP_REVISION_FURY_DVT2              7
#define MDP_REVISION_FURY                   MDP_REVISION_FURY_DVT2

#define MDP_MODEL_GRAVITY                   37

#define MDP_SUBMODEL_GRAVITY                1

#define MDP_REVISION_GRAVITY_PROTO0         1
#define MDP_REVISION_GRAVITY_PROTO1A        2
#define MDP_REVISION_GRAVITY_PROTO2A        3
#define MDP_REVISION_GRAVITY_P2B            4
#define MDP_REVISION_GRAVITY_EVT            5
#define MDP_REVISION_GRAVITY                MDP_REVISION_GRAVITY_EVT

#define MDP_MODEL_OPTIMO                    38

#define MDP_SUBMODEL_OPTIMO_OPTIMO1         1
#define MDP_SUBMODEL_OPTIMO_OPTIMO1SL       2
#define MDP_SUBMODEL_OPTIMO_OPTIMO2         3

#define MDP_REVISION_OPTIMO_PROTO0          1
#define MDP_REVISION_OPTIMO1                MDP_REVISION_OPTIMO_PROTO0
#define MDP_REVISION_OPTIMO1SL              MDP_REVISION_OPTIMO_PROTO0
#define MDP_REVISION_OPTIMO2                MDP_REVISION_OPTIMO_PROTO0

#define MDP_MODEL_QICHARGER                 39

#define MDP_SUBMODEL_QICHARGER_MONACO	    1

#define MDP_REVISION_QICHARGER_MONACO1	    1

#define MDP_REGION_UNKNOWN 0x0
#define MDP_REGION_USCA 0x1
#define MDP_REGION_EU 0x2
#define MDP_REGION_CHINA 0x3
#define MDP_REGION_JAPAN 0x4
#define MDP_REGION_ISRAEL 0x5
#define MDP_REGION_RUSSIA 0x6
#define MDP_REGION_SOUTH_KOREA 0x7
#define MDP_MAX_REGION 0x7

#define MDP_DEFAULT_COPYRIGHT "Copyright 2021 Sonos, Inc."

#define MDP_EMPTY_SERIAL {0,0,0,0,0,0,0,0}

#define MDP_SERIAL_LENGTH 6
#define MDP_PIN_LENGTH 8
#define MDP_SERIES_ID_LENGTH 4

struct manufacturing_data_page {
    uint32_t mdp_magic;
    uint32_t mdp_vendor;
    uint32_t mdp_model;
    uint32_t mdp_submodel;
    uint32_t mdp_revision;
    uint8_t mdp_serial[8];
    uint32_t mdp_region;
    uint32_t mdp_reserved;
    char mdp_copyright_statement[64];
    uint32_t mdp_flags;
    uint32_t mdp_hwfeatures;
    uint8_t mdp_ch11spurimmunitylevel;
    uint8_t mdp_reserved2[3];
    uint32_t mdp_version;
    uint32_t mdp2_version;
    uint32_t mdp3_version;
    uint32_t mdp_pages_present;
    uint32_t mdp_authorized_flags;
    uint32_t mdp_unused;
    uint32_t mdp_fusevalue;
    uint32_t mdp_sw_features;
    char mdp_pin[MDP_PIN_LENGTH];
    char mdp_series_id[MDP_SERIES_ID_LENGTH];
    uint32_t mdp_wildcard_variant;
    uint32_t mdp_location;

    uint8_t mdp_reserved3[92];
    union {
        uint8_t uboot_version[120];
        uint8_t u_reserved[256];
        struct {
            int32_t mdp_zp_dcofs[4];
        } zp;
    } u;
};
#define MDP1_BYTES 512

#define MDP_VERSION_AUTH_FLAGS          0x1
#define MDP_VERSION_SW_FEATURES         0x2
#define MDP_VERSION_PIN                 0x3
#define MDP_VERSION_SERIES_ID           0x4
#define MDP_VERSION_WILDCARD_VARIANT    0x5
#define MDP_VERSION_LOCATION            0x6
#define MDP_CURRENT_VERSION             MDP_VERSION_LOCATION

#define MDP2_VERSION_LEGACY_RSA         0x1
#define MDP2_VERSION_FSN                0x2
#define MDP2_VERSION_UNIT_SIG           0x3
#define MDP2_VERSION_VARIANT            0x4
#define MDP2_CURRENT_VERSION            MDP2_VERSION_VARIANT

#define MDP2_ENC_VERSION_DEV_CERT       0x4
#define MDP2_ENC_VERSION_ENC_VARIANT    0x5
#define MDP2_ENC_CURRENT_VERSION        MDP2_ENC_VERSION_ENC_VARIANT

#define MDP3_VERSION_SECURE_BOOT        0x1
#define MDP3_VERSION_DEV_CERT           0x2
#define MDP3_CURRENT_VERSION            MDP3_VERSION_DEV_CERT

#define MDP2_LEGACY_KEYLEN              836
#define MDP2_LEGACY_FSN_KEYLEN          964

#define MDP_FLAG_KERNEL_PRINTK_ENABLE   0x01
#define MDP_FLAG_CONSOLE_ENABLE         0x02
#define MDP_FLAG_HAS_HWFEATURES         0x04
#define MDP_FLAG_RUN_DIAGS              0x08
#define MDP_FLAG_HAS_VERSION            0x10
#define MDP_FLAG_REVOKE_MFG_KEY         0x20
#define MDP_FLAG_POSTMFG_FIX            0x20
#define MDP_FLAG_RUN_BURNFUSE           0x40


#define MDP_AUTH_FLAG_KERNEL_PRINTK_ENABLE      0x00000001
#define MDP_AUTH_FLAG_CONSOLE_ENABLE            0x00000002
#define MDP_AUTH_FLAG_MFG_KEY_ENABLE            0x00000004
#define MDP_AUTH_FLAG_TELNET_ENABLE             0x00000008
#define MDP_AUTH_FLAG_EXEC_ENABLE               0x00000010
#define MDP_AUTH_FLAG_UBOOT_UNLOCK_ENABLE       0x00000020
#define MDP_AUTH_FLAG_KERNEL_DEBUG_ENABLE       0x00000040
#define MDP_AUTH_FLAG_INSMOD_CTRL               0x00000080
#define MDP_AUTH_FLAG_BUTTON_DEBUG              0x00000100
#define MDP_AUTH_FLAG_NODEV_CTRL                0x00000200
#define MDP_AUTH_FLAG_MIC_DBG_ENABLE            0x00000400
#define MDP_AUTH_FLAG_BYPASS_ENABLE             0x00000800
#define MDP_AUTH_FLAG_HWV_DEFAULTS              0x00001000
#define MDP_AUTH_FLAG_PROTECTED_HW              0x00002000
#define MDP_AUTH_FLAG_DBG_UI_ENABLE             0x00004000
#define MDP_AUTH_FLAG_VOICE_DBG_ENABLE          0x00008000

#define MDP_AUTH_FLAG_ALL_FEATURES              0x0000ffff

#define MDP_PAGE1_PRESENT   0x01
#define MDP_PAGE2_PRESENT   0x02
#define MDP_PAGE3_PRESENT   0x04



#define MDP_KERNEL_PRINTK_ENABLE    MDP_FLAG_KERNEL_PRINTK_ENABLE

#define MDP_HWFEATURE_ATH_AR2414_WIFI   0x01
#define MDP_HWFEATURE_STMICRO_NAND      0x02
#define MDP_HWFEATURE_REDROCKS          0x04
#define MDP_HWFEATURE_ATH_AR5416_WIFI   0x08
#define MDP_HWFEATURE_ATH_AR9000_WIFI   0x10
#define MDP_HWFEATURE_LED_SWAP          0x20
#define MDP_HWFEATURE_ALT_ANTENNAS      0x40
#define MDP_HWFEATURE_CYPRESS_G1_NAND   0x80

#if defined SONOS_ARCH_MONACO
#define MDP_HWFEATURE_BATTERY_PACK_ID_MASK  0x70000
#define MDP_HWFEATURE_BATTERY_PACK_ID_LG        0x0
#define MDP_HWFEATURE_BATTERY_PACK_ID_PANASONIC 0x1
#define MDP_HWFEATURE_BATTERY_PACK_ID_LISHEN    0x2
#endif

struct mdp_key_hdr {
    uint32_t m_magic;
    uint32_t m_len;
    uint32_t m_reserved;
};

#define MDP_KEY_HDR_MAGIC_CAAM_RSAREF_PRIV      0x19283746
#define MDP_KEY_HDR_MAGIC_CAAM_PKCS1_RSA_PRIV   0xF4CCC68D
#define MDP_KEY_HDR_MAGIC_CAAM_AES128_BLACK_KEY 0xF0A2ADE7
#define MDP_KEY_HDR_MAGIC_CAAM_AES128_RED_KEY   0x618C4DE8


struct manufacturing_data_page2 {
    uint32_t mdp2_magic;
    uint32_t mdp2_keylen;
    union {
        uint8_t mdp2_key[4088];
        struct {
            uint8_t old_rsa_private[708];
            uint8_t old_rsa_sig[128];
            uint8_t old_fsn_sig[128];
            uint8_t old_unit_sig[128];
            uint32_t  old_variant;

            uint8_t old_reserved[4088 - (708 + (128 * 3) + 4)];
        } ;
        struct {
            uint8_t prod_rsa_private[1024];
            uint8_t prod_unit_sig[128];
            uint32_t  prod_cert_flags;

            uint8_t dev_rsa_private[1024];
            uint8_t dev_unit_sig[128];
            uint32_t  dev_cert_flags;

            uint8_t prod_rsa_sig[128];
            uint8_t dev_rsa_sig[128];

            uint32_t  variant;


            uint8_t dev_reserved[4088 - ((1024 * 2) + (128 * 4) + (4 * 3))];
        } ;
    } mdp2_sigdata;
};
#define MDP2_BYTES 4096

#define MDP_VARIANT_NONE         0
#define MDP_VARIANT_WHITE        1
#define MDP_VARIANT_BLACK        2
#define MDP_VARIANT_WHITE_GLOSS  3
#define MDP_VARIANT_WHITE_MATTE  4
#define MDP_VARIANT_BLACK_GLOSS  5
#define MDP_VARIANT_BLACK_MATTE  6
#define MDP_VARIANT_BLUE         7
#define MDP_VARIANT_LIMITED      8
#define MDP_VARIANT_SHADOW       9
#define MDP_VARIANT_LUNAR_WHITE  10
#define MDP_VARIANT_OLIVE        12
#define MDP_VARIANT_WAVE         13
#define MDP_VARIANT_SUNSET       14
#define MDP_VARIANT_MAX          MDP_VARIANT_SUNSET


#define MDP_VARIANT_WILDCARD     11

struct manufacturing_data_page3 {
    uint32_t mdp3_magic;
    uint32_t mdp3_version;
    uint8_t mdp3_reserved[376];

    uint8_t mdp3_auth_sig[512];
    uint8_t mdp3_cpuid_sig[512];

    uint8_t mdp3_fskey1[256];
    uint8_t mdp3_fskey2[256];
    uint8_t mdp3_model_private_key[2048];

    uint8_t mdp3_prod_unit_rsa_key[2048];
    uint8_t mdp3_prod_unit_rsa_cert[2048];
    uint8_t mdp3_dev_unit_rsa_key[2048];
    uint8_t mdp3_dev_unit_rsa_cert[2048];

    uint8_t mdp3_reserved2[4096 + 128];
};
#define MDP3_BYTES 16384

struct smdp {
    struct manufacturing_data_page      mdp;
    struct manufacturing_data_page2     mdp2;
    struct manufacturing_data_page3     mdp3;
};

#if defined(SONOS_ARCH_ATTR_IS_BUILD_TOOLS)
#define MDP_HAS_SB_REAL_CERT(m) (1)
#else
#define MDP_HAS_SB_REAL_CERT(m) ((m).mdp2.mdp2_magic == MDP_MAGIC2_ENC && \
                                 (m).mdp3.mdp3_version >= MDP3_VERSION_SECURE_BOOT)
#endif

#define MDP_SUBMODEL_IS_FENWAY(submodel) \
    ((submodel) == MDP_SUBMODEL_FENWAY)
#define MDP_SUBMODEL_IS_ANVIL(submodel) \
    ((submodel) == MDP_SUBMODEL_FENWAY_ANVIL || \
     (submodel) == MDP_SUBMODEL_FENWAY_ANVILLP)
#define MDP_SUBMODEL_IS_AMOEBA(submodel) \
    ((submodel) == MDP_SUBMODEL_FENWAY_AMOEBA || \
     (submodel) == MDP_SUBMODEL_FENWAY_AMOEBALP)

#define MDP_IS_SUBMODEL_REDROCKS(m, sm) \
    ((m) == MDP_MODEL_ZP && (sm) == MDP_SUBMODEL_ZP_REDROCKS)

#define MDP_MODEL_IS_WEMBLEYISH(m) \
    ((m) == MDP_MODEL_WEMBLEY || (m) == MDP_MODEL_CONNECTX)

#define MDP_IS_SUBMODEL_WEMBLEY(m, sm) \
    (MDP_MODEL_IS_WEMBLEYISH(m) && (sm) == MDP_SUBMODEL_WEMBLEY_WEMBLEY)

#define MDP_IS_SUBMODEL_ZPS5(m, sm) \
    (MDP_MODEL_IS_WEMBLEYISH(m) && (sm) == MDP_SUBMODEL_WEMBLEY_ZPS5)

#define MDP_IS_SUBMODEL_EDEN(m, sm) \
    (MDP_MODEL_IS_WEMBLEYISH(m) && (sm) == MDP_SUBMODEL_WEMBLEY_EDEN)

#define MDP_IS_SUBMODEL_FENWAY(m, sm) \
    ((m) == MDP_MODEL_FENWAY && MDP_SUBMODEL_IS_FENWAY(sm))

#define MDP_IS_SUBMODEL_ANVIL(m, sm) \
    ((m) == MDP_MODEL_FENWAY && MDP_SUBMODEL_IS_ANVIL(sm))

#define MDP_IS_SUBMODEL_AMOEBA(m, sm) \
    ((m) == MDP_MODEL_FENWAY && MDP_SUBMODEL_IS_AMOEBA(sm))

#define MDP_IS_SUBMODEL_ROYALE(m, sm) \
    ((m) == MDP_MODEL_ROYALE && (sm) == MDP_SUBMODEL_ROYALE)

#define MDP_IS_SUBMODEL_MARQUEE(m, sm) \
    ((m) == MDP_MODEL_ROYALE && (sm) == MDP_SUBMODEL_ROYALE_MARQUEE)

#define MDP_IS_SUBMODEL_LARGO(m, sm) \
    ((m) == MDP_MODEL_ROYALE && (sm) == MDP_SUBMODEL_ROYALE_LARGO)

#define MDP_IS_SUBMODEL_APOLLO(m, sm) \
    ((m) == MDP_MODEL_APOLLO && (sm) == MDP_SUBMODEL_APOLLO)

#define MDP_IS_SUBMODEL_APOLLOX(m, sm) \
    ((m) == MDP_MODEL_APOLLO && (sm) == MDP_SUBMODEL_APOLLO_APOLLOX)

#define MDP_IS_SUBMODEL_MONACO(m, sm) \
    ((m) == MDP_MODEL_MONACO && (sm) == MDP_SUBMODEL_MONACO)

#define MDP_IS_SUBMODEL_MONACOSL(m, sm) \
    ((m) == MDP_MODEL_MONACO && (sm) == MDP_SUBMODEL_MONACO_MONACOSL)

#define MDP_IS_SUBMODEL_TITAN(m, sm) \
    ((m) == MDP_MODEL_TITAN && (sm) == MDP_SUBMODEL_TITAN)

#define MDP_IS_SUBMODEL_WOWP(m, sm) \
    ((m) == MDP_MODEL_TITAN && (sm) == MDP_SUBMODEL_TITAN_WOWP)

#define MDP_IS_SUBMODEL_WOWT(m, sm) \
    ((m) == MDP_MODEL_TITAN && (sm) == MDP_SUBMODEL_TITAN_WOWT)

#define MDP_IS_SUBMODEL_WOWB(m, sm) \
    ((m) == MDP_MODEL_TITAN && (sm) == MDP_SUBMODEL_TITAN_WOWB)

#define MDP_IS_SUBMODEL_WOWG(m, sm) \
    ((m) == MDP_MODEL_TITAN && (sm) == MDP_SUBMODEL_TITAN_WOWG)

#define MDP_IS_SUBMODEL_BRAVO(m, sm) \
    ((m) == MDP_MODEL_BRAVO && (sm) == MDP_SUBMODEL_BRAVO)

#define MDP_IS_SUBMODEL_BRAVOSL(m, sm) \
    ((m) == MDP_MODEL_BRAVO && (sm) == MDP_SUBMODEL_BRAVO_BRAVOSL)

#define MDP_IS_SUBMODEL_CHAPLIN(m, sm) \
    ((m) == MDP_MODEL_CHAPLIN && (sm) == MDP_SUBMODEL_CHAPLIN)

#define MDP_IS_SUBMODEL_KAPITAL(m, sm) \
    ((m) == MDP_MODEL_CHAPLIN && (sm) == MDP_SUBMODEL_CHAPLIN_KAPITAL)

#define MDP_IS_SUBMODEL_OPTIMO1(m, sm) \
    ((m) == MDP_MODEL_OPTIMO && (sm) == MDP_SUBMODEL_OPTIMO_OPTIMO1)

#define MDP_IS_SUBMODEL_OPTIMO1SL(m, sm) \
    ((m) == MDP_MODEL_OPTIMO && (sm) == MDP_SUBMODEL_OPTIMO_OPTIMO1SL)

#define MDP_IS_SUBMODEL_OPTIMO2(m, sm) \
    ((m) == MDP_MODEL_OPTIMO && (sm) == MDP_SUBMODEL_OPTIMO_OPTIMO2)

#define MDP_IS_SUBMODEL_TUPELO(m, sm) \
    ((m) == MDP_MODEL_TUPELO && (sm) == MDP_SUBMODEL_TUPELO)

#define MDP_IS_SUBMODEL_GOLDENEYE(m, sm) \
    ((m) == MDP_MODEL_TUPELO && (sm) == MDP_SUBMODEL_TUPELO_GOLDENEYE)


#ifdef __KERNEL__

extern struct manufacturing_data_page sys_mdp;
extern struct smdp secure_mdp;

#define MDP_SIMPLE2_PRED(model_str, submodel) \
    (sys_mdp.mdp_model == MDP_MODEL_##model_str && \
     sys_mdp.mdp_submodel == (submodel))
#define MDP_SIMPLE_PRED(model_str) \
    MDP_SIMPLE2_PRED(model_str, MDP_SUBMODEL_##model_str)
#define MDP_SIMPLE2_MINREV_PRED(model_str, submodel, min_rev) \
    (MDP_SIMPLE2_PRED(model_str, submodel) && sys_mdp.mdp_revision >= (min_rev))
#define MDP_SUBMODEL_PRED(submodel_str) \
    MDP_IS_SUBMODEL_##submodel_str(sys_mdp.mdp_model, sys_mdp.mdp_submodel)

#define WEMBLEY         MDP_SUBMODEL_PRED(WEMBLEY)
#define EDEN            MDP_SUBMODEL_PRED(EDEN)
#define ZPS5            MDP_SUBMODEL_PRED(ZPS5)

#define IS_FENWAY       MDP_SUBMODEL_PRED(FENWAY)
#define IS_FENWAY_128MB (IS_FENWAY && \
                         sys_mdp.mdp_revision >= MDP_REVISION_FENWAY_128MB)
#define IS_ANVIL        MDP_SUBMODEL_PRED(ANVIL)
#define IS_ANVIL_NOVCXO (IS_ANVIL && \
                         sys_mdp.mdp_revision >= MDP_REVISION_ANVIL_P3)
#define IS_ANVIL_128MB  MDP_SIMPLE2_MINREV_PRED(FENWAY, \
                                                MDP_SUBMODEL_FENWAY_ANVIL, \
                                                MDP_REVISION_ANVIL_128MB)
#define IS_ANVIL_LP     MDP_SIMPLE2_PRED(FENWAY, MDP_SUBMODEL_FENWAY_ANVILLP)
#define IS_AMOEBA       MDP_SUBMODEL_PRED(AMOEBA)
#define IS_AMOEBA_128MB MDP_SIMPLE2_MINREV_PRED(FENWAY, \
                                                MDP_SUBMODEL_FENWAY_AMOEBA, \
                                                MDP_REVISION_AMOEBA_128MB)
#define IS_AMOEBA_LP    MDP_SIMPLE2_PRED(FENWAY, MDP_SUBMODEL_FENWAY_AMOEBALP)

#define IS_LIMELIGHT    MDP_SIMPLE_PRED(LIMELIGHT)
#define IS_FILLMORE     MDP_SIMPLE_PRED(FILLMORE)
#define IS_ENCORE       MDP_SIMPLE_PRED(ENCORE)
#define IS_SOLBASE      MDP_SIMPLE_PRED(SOLBASE)
#define IS_BOOTLEG      MDP_SIMPLE_PRED(BOOTLEG)
#define IS_ROYALE       MDP_SUBMODEL_PRED(ROYALE)
#define IS_MARQUEE      MDP_SUBMODEL_PRED(MARQUEE)
#define IS_LARGO        MDP_SUBMODEL_PRED(LARGO)
#define IS_PARAMOUNT    MDP_SIMPLE_PRED(PARAMOUNT)
#define IS_ELREY        MDP_SIMPLE_PRED(ELREY)
#define IS_HIDEOUT      MDP_SIMPLE_PRED(HIDEOUT)
#define IS_DHUEZ        MDP_SIMPLE_PRED(DHUEZ)
#define IS_TUPELO       MDP_SUBMODEL_PRED(TUPELO)
#define IS_GOLDENEYE    MDP_SUBMODEL_PRED(GOLDENEYE)
#define IS_APOLLO       MDP_SUBMODEL_PRED(APOLLO)
#define IS_APOLLOX      MDP_SUBMODEL_PRED(APOLLOX)
#define IS_CHAPLIN      MDP_SUBMODEL_PRED(CHAPLIN)
#define IS_KAPITAL      MDP_SUBMODEL_PRED(KAPITAL)
#define IS_NEPTUNE      MDP_SIMPLE_PRED(NEPTUNE)
#define IS_DOMINO       MDP_SIMPLE_PRED(DOMINO)
#define IS_TRIDENT      MDP_SIMPLE_PRED(TRIDENT)
#define IS_VERTIGO      MDP_SIMPLE_PRED(VERTIGO)
#define IS_MONACO       MDP_SUBMODEL_PRED(MONACO)
#define IS_MONACOSL     MDP_SUBMODEL_PRED(MONACOSL)
#define IS_BRAVO        MDP_SUBMODEL_PRED(BRAVO)
#define IS_BRAVOSL      MDP_SUBMODEL_PRED(BRAVOSL)
#define IS_FURY         MDP_SIMPLE_PRED(FURY)
#define IS_GRAVITY      MDP_SIMPLE_PRED(GRAVITY)
#define IS_OPTIMO1      MDP_SUBMODEL_PRED(OPTIMO1)
#define IS_OPTIMO1SL    MDP_SUBMODEL_PRED(OPTIMO1SL)
#define IS_OPTIMO2      MDP_SUBMODEL_PRED(OPTIMO2)

#endif
#endif
