/*
 * Copyright (c) 2015-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:	GPL-2.0
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "sonos_common.h"
/* this program only works with the legacy version of virtual_block */
#define SONOS_VIRTUAL_BLOCK_WANT_LEGACY_INTERFACE 1
#include "virtual_block_ioctl.h"

#define DATA_BUFFER_LEN 4096

static int ubi_commit(const sonos_enfs *config)
{
	int fd;
	int ret;

	fd = open(config->block_dev, O_RDWR);
	if ( fd < 0 ) {
		syslog(LOG_ERR, "open %s fail\n", config->block_dev);
		return 1;
	}

	ret = ioctl(fd, VIRTUAL_BLOCK_COMMIT);
	if ( ret < 0 ) {
		syslog(LOG_ERR, "VIRTUAL_BLOCK_COMMIT error %d\n", ret);
	}
	close(fd);
	return ret;
}

static int ubi_check(const sonos_enfs *config, unsigned long len)
{
	int fd;
	int ret = 0;
	unsigned long read_len = 0;
	int timeout_count = 0;

	if ( len == 0 ) {
		return 0;
	}

	fd = open(config->block_dev, O_RDWR);
	if ( fd < 0 ) {
		syslog(LOG_ERR, "open %s fail\n", config->block_dev);
		return 1;
	}
	while (1) {
		ret = ioctl(fd, VIRTUAL_BLOCK_CHECK, &read_len);
		if ( ret < 0 ) {
			syslog(LOG_ERR, "VIRTUAL_BLOCK_CHECK error %d\n", ret);
			break;
		}

		if ( read_len == len ) {
			ret = 0;
			break;
		}
		sleep(1);
		timeout_count ++;
		if ( timeout_count > 60 ) {
			ret = 1;
			syslog(LOG_ERR, "Can't complete rootfs write\n");
			break;
		}
	}
	close(fd);
	return ret;
}

static int ubi_stop(const sonos_enfs *config)
{
	int fd;
	int ret;

	fd = open(config->block_dev, O_RDWR);
	if ( fd < 0 ) {
		syslog(LOG_ERR, "open %s fail\n", config->block_dev);
		return 1;
	}
	ret = ioctl(fd, VIRTUAL_BLOCK_COMPLETE);
	if ( ret < 0 ) {
		syslog(LOG_ERR, "UBI_IOCVOLUP error %d\n", ret);
	}
	close(fd);
	return ret;
}

int ubi_update(const sonos_enfs *config)
{
	int fd, out_fd;
	int ret = 0, count = 0;
	int bytes_read = 0;
	char buffer[DATA_BUFFER_LEN];

	memset(buffer, 0, DATA_BUFFER_LEN);
	snprintf(buffer, DATA_BUFFER_LEN, "/dev/mapper/%s", config->virtual_dev);

	out_fd = open(buffer, O_RDWR | O_SYNC);
	if ( out_fd < 0 ) {
		syslog(LOG_ERR, "open %s fail\n", buffer);
		return 1;
	}
	syslog(LOG_INFO, "Using stdin\n");
	while ( 1 ) {
		bytes_read = read(0, buffer, DATA_BUFFER_LEN);
		if ( bytes_read < 0 ) {
			syslog(LOG_ERR, "%d failure reading from stdin\n", bytes_read);
			count = 0;
			break;
		} else if ( bytes_read ) {
			ret = write(out_fd, buffer, bytes_read);
			if (ret == bytes_read) {
				count += bytes_read;
			} else {
				syslog(LOG_ERR, "write of %d bytes failed:  %d\n", bytes_read, ret);
				break;
			}
		} else {
			break; /* end of read */
		}
		if (count == 0) {
			ret = 0;
			break;
		}
	}
	if ( count ) {
		ret = ubi_check(config, count);
	}
	ret |= ubi_commit(config);
	ret |= ubi_stop(config);
	close(out_fd);
	return ret;
}
