/*
 * Copyright (c) 2018-2019 Sonos Inc.
 * All rights reserved.
 */

#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/regmap.h>
#include <linux/sysfs.h>
#include <linux/debugfs.h>
#include <linux/proc_fs.h>
#include <linux/string.h>
#include <linux/of_gpio.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>
#include <linux/bitrev.h>
#include <linux/mutex.h>

#include <sound/soc.h>
#include <sound/soc-dai.h>
#include <sound/pcm.h>
#include <sound/control.h>

#include "hdmictl.h"
#include "sonos_asound.h"

#include "sii_platform_api.h"
#include "sii_platform_linux.h"
#include "sii9437_api.h"

#define DRV_NAME		"sii9437"
#define PROCFS_DIR		"driver/"DRV_NAME
#define PROCFS_LOG		"log"
#define PROCFS_TEST_I2S_MAP	"test_i2s_map"
#define PROCFS_TEST_PLL		"test_pll"

//~ #define DEBUG_SII9437
#ifdef DEBUG_SII9437
#define PRINTK_DBG(FORMAT, ARG...)	printk("%s [%s:%d] "FORMAT, DRV_NAME, __FUNCTION__, __LINE__, ## ARG)
#define DEV_DBG(DEV, FORMAT, ARG...)	dev_info(DEV, "[%s:%d] "FORMAT, __FUNCTION__, __LINE__, ## ARG)
#else /* !DEBUG_SII9437 */
#define PRINTK_DBG(FORMAT, ARG...)
#define DEV_DBG(DEV, FORMAT, ARG...)	(void)DEV
#endif /* DEBUG_SII9437 */

#define ASSERT_LOCKED(DRV_OBJ)	\
	if (!mutex_is_locked(&(DRV_OBJ->fw_lock))) \
		dev_err(sii9437->dev, "ASSERT: %s called without lock!\n", __FUNCTION__)

enum sii9437_ctrl_idx {
	CTRL_IDX_EARC_ENABLE = 0,
	CTRL_IDX_LINK_STATE,
	CTRL_IDX_CHANNEL_COUNT,
	CTRL_IDX_SAMPLE_RATE,
	CTRL_IDX_CSB_VALUE,
	CTRL_IDX_CSB_MASK,
	CTRL_IDX_CAP_DS,
	CTRL_IDX_LATENCY,
	CTRL_IDX_LATENCY_REQ,
	CTRL_IDX_MAX,
};

struct sii9437 {
	struct device		*dev;

	struct i2c_client	*i2c_main;
	struct regmap		*reg_main;

	struct i2c_client	*i2c_rx;
	struct regmap		*reg_rx;

	struct i2c_client	*i2c_phy;
	struct regmap		*reg_phy;

	int			reset_gpio;
	int			mute_gpio;
	int			irq_gpio;

	int			irq_num;

	sii_inst_t		fw_inst;
	struct mutex		fw_lock;
	bool			earc_timeout;

	struct delayed_work	hpd_watchdog_work;

	struct snd_soc_codec	*snd_codec;
	struct snd_card		*snd_card;
	struct snd_kcontrol	*snd_kctrls[CTRL_IDX_MAX];

	struct proc_dir_entry	*procfs_dir;
	struct proc_dir_entry   *procfs_test_pll;
	struct proc_dir_entry	*procfs_log;
	struct proc_dir_entry	*procfs_test_i2s_map;
};

#define MAX_EARC_SAMPLE_RATE	(768000)
#define MAX_ALSA_SAMPLE_RATE	(192000)

#define CSB_SIZE FIELD_SIZEOF(struct snd_aes_iec958, status)

static enum hdmi_earc_link sii9437_get_link_state_locked(struct sii9437 *sii9437);

/* Global context pointer because the sii9437 event callback provides no context. */
static struct sii9437 *g_event_context = NULL;

static const struct regmap_config regmap_config_main = {
	.name = DRV_NAME"_main",
	.reg_bits = 8,
	.val_bits = 8,
	.max_register = 0xff,
	/* TODO - define RO, WO, and RW registers */
};

static const struct regmap_config regmap_config_rx = {
	.name = DRV_NAME"_rx",
	.reg_bits = 8,
	.val_bits = 8,
	.max_register = 0xff,
	/* TODO - define RO, WO, and RW registers */
};

static const struct regmap_config regmap_config_phy = {
	.name = DRV_NAME"_phy",
	.reg_bits = 8,
	.val_bits = 8,
	.max_register = 0xff,
	/* TODO - define RO, WO, and RW registers */
};

static inline enum sii_arc_mode _sii9437_arc_mode_get(struct sii9437 *sii9437)
{
	enum sii_arc_mode mode = SII_ARC_MODE__NONE;
	uint32_t sii_ret;
	ASSERT_LOCKED(sii9437);

	sii_ret = sii9437_arc_mode_get(sii9437->fw_inst, &mode);
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_arc_mode_get() failed err=%d\n", sii_ret);
	}
	return mode;
}

static inline void _sii9437_arc_mode_set(struct sii9437 *sii9437, enum sii_arc_mode mode)
{
	uint32_t sii_ret;
	ASSERT_LOCKED(sii9437);

	sii_ret = sii9437_arc_mode_set(sii9437->fw_inst, &mode);
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_arc_mode_set() failed err=%d\n", sii_ret);
	}
}

static inline void _sii9437_preferred_extraction_mode_set(struct sii9437 *sii9437, enum sii9437_pref_extraction_mode mode)
{
	uint32_t sii_ret;
	ASSERT_LOCKED(sii9437);

	sii_ret = sii9437_preferred_extraction_mode_set(sii9437->fw_inst, &mode);
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_preferred_extraction_mode_set() failed err=%d\n", sii_ret);
	}
}

static inline enum sii9437_extraction_mode _sii9437_extraction_mode_query(struct sii9437 *sii9437)
{
	enum sii9437_extraction_mode mode = SII9437_EXTRACTION_MODE__NONE;
	uint32_t sii_ret;
	ASSERT_LOCKED(sii9437);

	sii_ret = sii9437_extraction_mode_query(sii9437->fw_inst, &mode);
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_extraction_mode_query() failed err=%d\n", sii_ret);
	}
	return mode;
}

static inline bool_t _sii9437_hpd_query(struct sii9437 *sii9437)
{
	bool_t hpd = 0;
	uint32_t sii_ret;
	ASSERT_LOCKED(sii9437);

	sii_ret = sii9437_hpd_query(sii9437->fw_inst, &hpd);
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_hpd_query() failed err=%d\n", sii_ret);
	}
	return hpd;
}

static inline bool_t _sii9437_earc_link_query(struct sii9437 *sii9437)
{
	bool_t link = 0;
	uint32_t sii_ret;
	ASSERT_LOCKED(sii9437);

	sii_ret = sii9437_earc_link_query(sii9437->fw_inst, &link);
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_earc_link_query() failed err=%d\n", sii_ret);
	}
	return link;
}

static inline uint8_t _sii9437_erx_latency_get(struct sii9437 *sii9437)
{
	struct sii_erx_latency latency;
	uint32_t sii_ret;
	ASSERT_LOCKED(sii9437);

	sii_ret = sii9437_erx_latency_get(sii9437->fw_inst, &latency);
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_erx_latency_get() failed err=%d\n", sii_ret);
	}
	return latency.data;
}

static inline uint8_t _sii9437_erx_latency_req_query(struct sii9437 *sii9437)
{
	struct sii_erx_latency_req latency;
	uint32_t sii_ret;
	ASSERT_LOCKED(sii9437);

	sii_ret = sii9437_erx_latency_req_query(sii9437->fw_inst, &latency);
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_erx_latency_req_query() failed err=%d\n", sii_ret);
	}
	return latency.data;
}

static inline void _sii9437_channel_status_query(struct sii9437 *sii9437, uint8_t *buffer, int len)
{
	uint32_t sii_ret;
	struct sii_channel_status channel_status;
	int pos;
	int csb_byte = 0;
	ASSERT_LOCKED(sii9437);

	sii_ret = sii9437_channel_status_query(sii9437->fw_inst, &channel_status);
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_channel_status_query() failed err=%d\n", sii_ret);
		return;
	}

	/*
	 * The FW returns 14 bytes of CSB data which contains the first and last 7 bytes
	 * contiguously.  This loop copies those seperate blocks into their correct positions
	 * into the standard 24 byte CSB.
	 *
	 * In other words the following FW CSB data:
	 *	11:22:33:44:55:66:77:88:99:AA:BB:CC:DD:EE
	 * Becomes the following in the CSB array:
	 *	11:22:33:44:55:66:77:00
	 *	00:00:00:00:00:00:00:00
	 *	00:88:99:AA:BB:CC:DD:EE
	 */
	for (pos = 0; pos < len; pos++) {
		if ((pos < 7) || (pos > 16)) {
			/* FW stores each CSB byte in MSB but userspace expects LSB */
			buffer[pos] = bitrev8(channel_status.data[csb_byte]);
			csb_byte++;
		}
		else {
			/* Set zero's for portions of the CSB not provided by FW (middle 10 bytes) */
			buffer[pos] = 0;
		}
	}
}

static inline uint32_t _sii9437_sample_rate_query(struct sii9437 *sii9437)
{
	uint32_t sample_rate = 0;
	uint32_t sii_ret;
	ASSERT_LOCKED(sii9437);

	sii_ret = sii9437_sample_rate_query(sii9437->fw_inst, &sample_rate);
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_sample_rate_query() failed err=%d\n", sii_ret);
		return 0;
	}
	return sample_rate;
}

static inline uint32_t _convert_earc_sample_rate_to_alsa(uint32_t earc_rate)
{
	/* If the sample rate is invalid */
	if (earc_rate > MAX_EARC_SAMPLE_RATE) {
		return 0;
	}

	/*
	 * ALSA does not support audio rates above 192kHz and most I2S
	 * peripherals do not either.  Since audio rates above 192kHz are
	 * transmitted using 4x i2s lines at 1/4th the frame rate
	 * we will always tell ALSA the line rate and not the actual
	 * frame rate declared in the CSB.
	 */
	return (earc_rate > MAX_ALSA_SAMPLE_RATE) ? (earc_rate / 4) : earc_rate;
}

#define case_enum_ret_str(e) \
		case e: \
			return #e

static const char *_extract_mode_to_str(enum sii9437_extraction_mode mode)
{
	switch (mode) {
		case_enum_ret_str(SII9437_EXTRACTION_MODE__NONE);
		case_enum_ret_str(SII9437_EXTRACTION_MODE__SPDIF2);
		case_enum_ret_str(SII9437_EXTRACTION_MODE__SPDIF8);
		case_enum_ret_str(SII9437_EXTRACTION_MODE__I2S2);
		case_enum_ret_str(SII9437_EXTRACTION_MODE__I2S8);
		default:
			return "?";
	}
}

static const char *_pref_extract_mode_to_str(enum sii9437_pref_extraction_mode mode)
{
	switch (mode) {
		case_enum_ret_str(SII9437_PREF_EXTRACTION_MODE__SPDIF);
		case_enum_ret_str(SII9437_PREF_EXTRACTION_MODE__I2S);
		default:
			return "?";
	}
}

static const char *_arc_mode_to_str(enum sii_arc_mode mode)
{
	switch (mode) {
		case_enum_ret_str(SII_ARC_MODE__NONE);
		case_enum_ret_str(SII_ARC_MODE__ARC);
		case_enum_ret_str(SII_ARC_MODE__EARC);
		default:
			return "?";
	}
}

static __maybe_unused const char *_earc_link_to_str(enum hdmi_earc_link earc_link)
{
	switch (earc_link) {
		case_enum_ret_str(HDMI_EARC_LINK_DISABLED);
		case_enum_ret_str(HDMI_EARC_LINK_OFFLINE);
		case_enum_ret_str(HDMI_EARC_LINK_TIMEOUT);
		case_enum_ret_str(HDMI_EARC_LINK_ONLINE);
		default:
			return "?";
	}
}

static int procfs_log_show(struct seq_file *m, void *v)
{
	(void)v; /* Unused */
	sii_seq_show_platform_log(m);
	return 0;
}

static int procfs_log_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_log_show, PDE_DATA(inode));
}

static const struct file_operations procfs_log_ops = {
	.owner = THIS_MODULE,
	.open = procfs_log_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

/*
 * Testpoint: /proc/driver/sii9437/test_i2s_map
 *
 * Allows direct control of the eARC channel pairs to I2S line mapping.
 *
 * User can enter in a 16-bit hexadecimal value where each digit corresponds
 * to each I2S line in the form of 0xDCBA where:
 *	D = I2S line 3 mapping
 *	C = I2S line 2 mapping
 *	B = I2S line 1 mapping
 *	A = I2S line 0 mapping
 *
 * Valid values:
 *	0 = Disable I2S line
 *	1 = eARC channels 0,1 (FIFO 0)
 *	2 = eARC channels 2,3 (FIFO 1)
 *	3 = eARC channels 4,5 (FIFO 2)
 *	4 = eARC channels 6,7 (FIFO 3)
 *
 * If -1 is passed in all lines will reset to default values.  The interface
 * behavior is undefined when eARC is not established because the I2S lines
 * are often disabled in other modes of operation.
 */

#define REG_RX_I2S_CTRL2		(0x00)
#define REG_RX_I2S_MAP			(0x01)
#define REG_MAIN_SW_RESET		(0x05)
#define REG_MAIN_SW_RESET_I2S		(0x80)

static int procfs_test_i2s_map_show(struct seq_file *m, void *v)
{
	struct sii9437 *sii9437 = m->private;
	unsigned int i2s_ctrl = 0;
	unsigned int i2s_map = 0;
	int err = 0;
	u64 value = 0;
	int idx;

	mutex_lock(&(sii9437->fw_lock));

	/* Read ctrl and map registers */
	err = regmap_read(sii9437->reg_rx, REG_RX_I2S_CTRL2, &i2s_ctrl);
	if (err) goto failed;
	err = regmap_read(sii9437->reg_rx, REG_RX_I2S_MAP, &i2s_map);
	if (err) goto failed;

	/* The i2s enable bits are in the top nibble */
	i2s_ctrl >>= 4;

	/* Loop through all 4 i2s lines */
	for (idx = 0; idx < 4; idx++) {
		u64 subval = 0;

		/* Determine if the current line is enabled */
		if (i2s_ctrl & 0x1) {
			/* Determine which fifo maps to the current line */
			subval = (i2s_map & 0x03) + 1;
		}

		/* OR in the current line value into the returned value */
		value |= subval << (idx * 4);

		/* Shift the current line out so the next loop examines the next line */
		i2s_ctrl >>= 1;
		i2s_map >>= 2;
	}

	seq_printf(m, "%llx\n", value);

failed:
	mutex_unlock(&(sii9437->fw_lock));
	return err;
}

static int procfs_test_i2s_map_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_test_i2s_map_show, PDE_DATA(inode));
}

static ssize_t procfs_test_i2s_map_write(struct file *file, const char __user *buffer, size_t count, loff_t *ppos)
{
	struct sii9437 *sii9437 = PDE_DATA(file_inode(file));
	ssize_t result = count;
	char arg[8];

	if (count >= sizeof(arg)) {
		result = -EIO;
	}
	else if (copy_from_user(&arg, buffer, count)) {
		result = -EFAULT;
	}
	else {
		unsigned int i2s_ctrl = 0;
		unsigned int i2s_map = 0;
		int err = 0;
		int idx;
		u64 value;

		/* Force null termination and convert to integer (base flexible) */
		arg[count] = '\0';
		value = simple_strtoll(arg, NULL, 0);

		/* -1 results in resetting the registers to default */
		if (value == -1llu) {
			dev_info(sii9437->dev, "%s: reseting i2s behavior to default\n", __FUNCTION__);
			value = 0x4321;
		}

		/* Validate data range */
		if (value & ~(0xFFFFull)) {
			dev_err(sii9437->dev, "%s: data out of range\n", __FUNCTION__);
			return -EINVAL;
		}

		/* Loop through all 4 i2s lines */
		for (idx = 0; idx < 4; idx++) {
			/* Get current line value */
			u64 subval = (value >> (4 * idx)) & 0xf;

			/* Determine if the current line should be enabled (0 = disabled) */
			if (subval) {
				if (subval > 4) {
					dev_err(sii9437->dev, "%s: bad digit\n", __FUNCTION__);
					return -EINVAL;
				}

				/* Enable the line */
				i2s_ctrl |= 0x1 << idx;
				/* Set the FIFO (zero based) */
				i2s_map |= (subval - 1) << (idx * 2);
			}
		}

		/* The i2s enable bits are in the top nibble */
		i2s_ctrl <<= 4;

		mutex_lock(&(sii9437->fw_lock));

		/* Log change for debugging purposes */
		dev_warn(sii9437->dev, "%s: value=%04llx earc=%s\n", __FUNCTION__, value, _sii9437_earc_link_query(sii9437) ? "online" : "OFFLINE!");

		/* Write ctrl and map registers */
		err = regmap_write(sii9437->reg_rx, REG_RX_I2S_CTRL2, i2s_ctrl);
		if (err) goto failed;
		err = regmap_write(sii9437->reg_rx, REG_RX_I2S_MAP, i2s_map);
		if (err) goto failed;

		/* Reset I2S output to eliminate any stale shift registers */
		err = regmap_update_bits(sii9437->reg_main, REG_MAIN_SW_RESET, REG_MAIN_SW_RESET_I2S, 0);
		if (err) goto failed;
		err = regmap_update_bits(sii9437->reg_main, REG_MAIN_SW_RESET, REG_MAIN_SW_RESET_I2S, REG_MAIN_SW_RESET_I2S);
		if (err) goto failed;

failed:
		mutex_unlock(&(sii9437->fw_lock));
		if (err) {
			result = -err;
		}
	}
	return result;
}

static const struct file_operations procfs_test_i2s_map_ops = {
	.owner = THIS_MODULE,
	.open = procfs_test_i2s_map_open,
	.read = seq_read,
	.write = procfs_test_i2s_map_write,
	.llseek = seq_lseek,
	.release = single_release,
};

static int procfs_test_pll_show(struct seq_file *m, void *v)
{
	struct sii9437 *sii9437 = m->private;
	uint8_t val = 0;

	sii9437_get_viola_pll(sii9437->fw_inst, &val);

	seq_printf(m, "PLL value = 0x%2.2X\n", val);
	return 0;
}

static int procfs_test_pll_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_test_pll_show, PDE_DATA(inode));
}

static ssize_t procfs_test_pll_write(struct file *file, const char __user *buffer, size_t count, loff_t *ppos)
{
	struct sii9437 *sii9437 = PDE_DATA(file_inode(file));
	ssize_t result = count;
	char arg[8];
	if (count >= sizeof(arg)) {
		result = -EIO;
	}
	else if (copy_from_user(&arg, buffer, count)) {
		result = -EFAULT;
	}
	else {
		unsigned val;
		arg[count] = '\0'; /* Force null termination */
		if (sscanf(arg, "%X", &val) > 0) {
			sii9437_set_viola_pll(sii9437->fw_inst, val);
		}
		else {
			result = -EIO;
		}
	}

	return result;
}

static const struct file_operations procfs_test_pll_ops = {
	.owner = THIS_MODULE,
	.open = procfs_test_pll_open,
	.read = seq_read,
	.write = procfs_test_pll_write,
	.llseek = seq_lseek,
	.release = single_release,
};

static ssize_t fw_ver_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	char fw_ver[32];
	(void)dev;
	(void)attr;

	sii9437_software_version_query(fw_ver, sizeof(fw_ver));
	return scnprintf(buf, PAGE_SIZE, "%s\n", fw_ver);
}

static ssize_t hw_id_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	uint32_t hw_id;
	int sii_ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	sii_ret = sii9437_chip_id_query(sii9437->fw_inst, &hw_id);
	mutex_unlock(&(sii9437->fw_lock));

	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		return -EIO;
	}
	return scnprintf(buf, PAGE_SIZE, "%04x\n", hw_id);
}

static ssize_t hw_rev_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	uint8_t hw_rev;
	int sii_ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	sii_ret = sii9437_chip_version_query(sii9437->fw_inst, &hw_rev);
	mutex_unlock(&(sii9437->fw_lock));

	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		return -EIO;
	}
	return scnprintf(buf, PAGE_SIZE, "%02x\n", hw_rev);
}

static ssize_t gpio_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	uint16_t gpio_status;
	int sii_ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	sii_ret = sii9437_gpio_query(sii9437->fw_inst, &gpio_status);
	mutex_unlock(&(sii9437->fw_lock));

	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		return -EIO;
	}
	return scnprintf(buf, PAGE_SIZE, "%02x\n", gpio_status);
}

static ssize_t standby_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	bool_t standby;
	int sii_ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	sii_ret = sii9437_standby_get(sii9437->fw_inst, &standby);
	mutex_unlock(&(sii9437->fw_lock));

	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		return -EIO;
	}
	return scnprintf(buf, PAGE_SIZE, "%d\n", standby);
}

static ssize_t hpd_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	int ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	ret = scnprintf(buf, PAGE_SIZE, "%d\n", _sii9437_hpd_query(sii9437));
	mutex_unlock(&(sii9437->fw_lock));
	return ret;
}

static ssize_t arc_mode_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	int ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	ret = scnprintf(buf, PAGE_SIZE, "%s\n", _arc_mode_to_str(_sii9437_arc_mode_get(sii9437)));
	mutex_unlock(&(sii9437->fw_lock));
	return ret;
}

static ssize_t earc_link_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	int ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	ret = scnprintf(buf, PAGE_SIZE, "%d\n", _sii9437_earc_link_query(sii9437));
	mutex_unlock(&(sii9437->fw_lock));
	return ret;
}

static ssize_t earc_timeout_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	(void)attr;
	return scnprintf(buf, PAGE_SIZE, "%d\n", sii9437->earc_timeout);
}

static ssize_t earc_extract_mode_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	int ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	ret = scnprintf(buf, PAGE_SIZE, "%s\n", _extract_mode_to_str(_sii9437_extraction_mode_query(sii9437)));
	mutex_unlock(&(sii9437->fw_lock));
	return ret;
}

static ssize_t earc_pref_extract_mode_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	enum sii9437_pref_extraction_mode pref_mode;
	int sii_ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	sii_ret = sii9437_preferred_extraction_mode_get(sii9437->fw_inst, &pref_mode);
	mutex_unlock(&(sii9437->fw_lock));

	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		return -EIO;
	}
	return scnprintf(buf, PAGE_SIZE, "%s\n", _pref_extract_mode_to_str(pref_mode));
}

static ssize_t earc_caps_ds_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	uint8_t caps_ds[SII_EARC_CAPS_DS_MAX_LENGTH];
	int row;
	const int row_size = 16;
	ssize_t out_len = 0;
	int sii_ret;
	(void)attr;

	memset(caps_ds, 0, sizeof(caps_ds));

	mutex_lock(&(sii9437->fw_lock));
	sii_ret = sii9437_earc_caps_ds_get(sii9437->fw_inst, 0, caps_ds, sizeof(caps_ds));
	mutex_unlock(&(sii9437->fw_lock));

	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		return -EIO;
	}

	for (row = 0; row < (sizeof(caps_ds) / row_size); row++) {
		out_len += scnprintf(buf + out_len, PAGE_SIZE - out_len,
				     "%*ph\n",
				     row_size, &caps_ds[row * row_size]);
	}
	return out_len;
}

static ssize_t earc_latency_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	int ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	ret = scnprintf(buf, PAGE_SIZE, "%d\n", _sii9437_erx_latency_get(sii9437));
	mutex_unlock(&(sii9437->fw_lock));
	return ret;
}

static ssize_t earc_latency_req_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	int ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	ret = scnprintf(buf, PAGE_SIZE, "%d\n", _sii9437_erx_latency_req_query(sii9437));
	mutex_unlock(&(sii9437->fw_lock));
	return ret;
}

static ssize_t earc_mute_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	int ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	ret = scnprintf(buf, PAGE_SIZE, "%d\n", gpio_get_value(sii9437->mute_gpio));
	mutex_unlock(&(sii9437->fw_lock));
	return ret;
}

static ssize_t earc_channel_status_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	uint8_t csb[CSB_SIZE];
	int row;
	const int row_size = 8;
	ssize_t out_len = 0;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	_sii9437_channel_status_query(sii9437, csb, sizeof(csb));
	mutex_unlock(&(sii9437->fw_lock));

	for (row = 0; row < (sizeof(csb) / row_size); row++) {
		out_len += scnprintf(buf + out_len, PAGE_SIZE - out_len,
				     "%*ph\n",
				     row_size, &(csb[row * row_size]));
	}

	return out_len;
}

static ssize_t earc_sample_rate_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sii9437 *sii9437 = dev_get_drvdata(dev);
	int ret;
	(void)attr;

	mutex_lock(&(sii9437->fw_lock));
	ret = scnprintf(buf, PAGE_SIZE, "%d\n", _sii9437_sample_rate_query(sii9437));
	mutex_unlock(&(sii9437->fw_lock));
	return ret;
}

static DEVICE_ATTR_RO(fw_ver);
static DEVICE_ATTR_RO(hw_id);
static DEVICE_ATTR_RO(hw_rev);
static DEVICE_ATTR_RO(gpio);
static DEVICE_ATTR_RO(standby);
static DEVICE_ATTR_RO(hpd);
static DEVICE_ATTR_RO(arc_mode);
static DEVICE_ATTR_RO(earc_link);
static DEVICE_ATTR_RO(earc_timeout);
static DEVICE_ATTR_RO(earc_extract_mode);
static DEVICE_ATTR_RO(earc_pref_extract_mode);
static DEVICE_ATTR_RO(earc_caps_ds);
static DEVICE_ATTR_RO(earc_latency);
static DEVICE_ATTR_RO(earc_latency_req);
static DEVICE_ATTR_RO(earc_mute);
static DEVICE_ATTR_RO(earc_channel_status);
static DEVICE_ATTR_RO(earc_sample_rate);

static struct attribute *sii9437_attr_group_list[] = {
	&dev_attr_fw_ver.attr,
	&dev_attr_hw_id.attr,
	&dev_attr_hw_rev.attr,
	&dev_attr_gpio.attr,
	&dev_attr_standby.attr,
	&dev_attr_hpd.attr,
	&dev_attr_arc_mode.attr,
	&dev_attr_earc_link.attr,
	&dev_attr_earc_timeout.attr,
	&dev_attr_earc_extract_mode.attr,
	&dev_attr_earc_pref_extract_mode.attr,
	&dev_attr_earc_caps_ds.attr,
	&dev_attr_earc_latency.attr,
	&dev_attr_earc_latency_req.attr,
	&dev_attr_earc_mute.attr,
	&dev_attr_earc_sample_rate.attr,
	&dev_attr_earc_channel_status.attr,
	NULL,
};

static struct attribute_group sii9437_attr_group = {
	.attrs = sii9437_attr_group_list,
};

static int snd_ctrl_earc_enable_info(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_BOOLEAN;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 1;
	return 0;
}

static int snd_ctrl_earc_enable_get(struct snd_kcontrol *kcontrol,
				   struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	mutex_lock(&(sii9437->fw_lock));
	uvalue->value.integer.value[0] = (_sii9437_arc_mode_get(sii9437) == SII_ARC_MODE__EARC);
	mutex_unlock(&(sii9437->fw_lock));
	return 0;
}

static int snd_ctrl_earc_enable_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	bool_t enable = !!(uvalue->value.integer.value[0]);

	DEV_DBG(sii9437->dev, "enable = %d\n", enable);

	mutex_lock(&(sii9437->fw_lock));

	sii9437->earc_timeout = FALSE;
	_sii9437_arc_mode_set(sii9437, enable ? SII_ARC_MODE__EARC : SII_ARC_MODE__ARC);
	snd_ctl_notify(sii9437->snd_card, SNDRV_CTL_EVENT_MASK_VALUE, &(sii9437->snd_kctrls[CTRL_IDX_LINK_STATE]->id));
	DEV_DBG(sii9437->dev, "earc state = %s\n", _earc_link_to_str(sii9437_get_link_state_locked(sii9437)));

	mutex_unlock(&(sii9437->fw_lock));

	return 0;
}

static int snd_ctrl_link_state_info(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_info *uinfo)
{
	const char *link_str;
	uinfo->type = SNDRV_CTL_ELEM_TYPE_ENUMERATED;
	uinfo->count = 1;
	uinfo->value.enumerated.items = HDMI_EARC_LINK_MAX;

	switch (uinfo->value.enumerated.item) {
		case HDMI_EARC_LINK_DISABLED:
			link_str = "disabled";
			break;

		case HDMI_EARC_LINK_OFFLINE:
			link_str = "offline";
			break;

		case HDMI_EARC_LINK_ONLINE:
			link_str = "online";
			break;

		case HDMI_EARC_LINK_TIMEOUT:
			link_str = "timeout";
			break;

		case HDMI_EARC_LINK_MAX:
		default:
			return -EINVAL;
	}

	strlcpy(uinfo->value.enumerated.name,
		link_str,
		sizeof(uinfo->value.enumerated.name));

	return 0;
}

static int snd_ctrl_link_state_get(struct snd_kcontrol *kcontrol,
				   struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	mutex_lock(&(sii9437->fw_lock));
	uvalue->value.enumerated.item[0] = sii9437_get_link_state_locked(sii9437);
	mutex_unlock(&(sii9437->fw_lock));
	return 0;
}

static int snd_ctrl_channel_count_info(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 8;
	return 0;
}

static int snd_ctrl_channel_count_get(struct snd_kcontrol *kcontrol,
				     struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	int ret = 0;

	mutex_lock(&(sii9437->fw_lock));

	switch (_sii9437_extraction_mode_query(sii9437)) {
		case SII9437_EXTRACTION_MODE__NONE:
			uvalue->value.integer.value[0] = 0;
			break;

		case SII9437_EXTRACTION_MODE__SPDIF2:
		case SII9437_EXTRACTION_MODE__I2S2:
			uvalue->value.integer.value[0] = 2;
			break;

		case SII9437_EXTRACTION_MODE__SPDIF8:
		case SII9437_EXTRACTION_MODE__I2S8:
			uvalue->value.integer.value[0] = 8;
			break;

		default:
			ret = -EIO;
	}

	mutex_unlock(&(sii9437->fw_lock));

	return ret;
}

static int snd_ctrl_sample_rate_info(struct snd_kcontrol *kcontrol,
				     struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = MAX_ALSA_SAMPLE_RATE;
	return 0;
}

static int snd_ctrl_sample_rate_get(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	mutex_lock(&(sii9437->fw_lock));
	uvalue->value.integer.value[0] = _convert_earc_sample_rate_to_alsa(_sii9437_sample_rate_query(sii9437));
	mutex_unlock(&(sii9437->fw_lock));
	return 0;
}

static int snd_ctrl_csb_info(struct snd_kcontrol *kcontrol,
			     struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_IEC958;
	uinfo->count = 1;
	return 0;
}

static int snd_ctrl_csb_get_default(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	mutex_lock(&(sii9437->fw_lock));
	_sii9437_channel_status_query(sii9437, uvalue->value.iec958.status, sizeof(uvalue->value.iec958.status));
	mutex_unlock(&(sii9437->fw_lock));
	return 0;
}

static int snd_ctrl_csb_get_mask(struct snd_kcontrol *kcontrol,
				 struct snd_ctl_elem_value *uvalue)
{
	/*
	 * The mask indicates we return the first and last 7 bytes of
	 * the 24 byte CSB.  See _sii9437_channel_status_query() for
	 * details.
	 */
	memset(&(uvalue->value.iec958.status[0]), 0xFF, 7);
	memset(&(uvalue->value.iec958.status[17]), 0xFF, 7);
	return 0;
}

static int snd_ctrl_cap_ds_info(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_BYTES;
	uinfo->count = SII_EARC_CAPS_DS_MAX_LENGTH;
	return 0;
}

static int snd_ctrl_cap_ds_get(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	uint32_t sii_ret;

	mutex_lock(&(sii9437->fw_lock));
	sii_ret = sii9437_earc_caps_ds_get(sii9437->fw_inst, 0, uvalue->value.bytes.data, SII_EARC_CAPS_DS_MAX_LENGTH);
	mutex_unlock(&(sii9437->fw_lock));

	return (sii_ret != SII_RETURN_VALUE__SUCCESS) ? -EIO : 0;
}

static int snd_ctrl_cap_ds_put(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	uint32_t sii_ret;

	DEV_DBG(sii9437->dev, "\n");

	mutex_lock(&(sii9437->fw_lock));
	sii_ret = sii9437_earc_caps_ds_set(sii9437->fw_inst, 0, uvalue->value.bytes.data, SII_EARC_CAPS_DS_MAX_LENGTH);
	mutex_unlock(&(sii9437->fw_lock));

	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_earc_caps_ds_set() failed (%d)\n", sii_ret);
		return -EIO;
	}
	return 0;
}

static int snd_ctrl_latency_info(struct snd_kcontrol *kcontrol,
				 struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = U8_MAX;
	return 0;
}

static int snd_ctrl_latency_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	mutex_lock(&(sii9437->fw_lock));
	uvalue->value.integer.value[0] = _sii9437_erx_latency_get(sii9437);
	mutex_unlock(&(sii9437->fw_lock));
	return 0;
}

static int snd_ctrl_latency_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	struct sii_erx_latency latency = {
		.data = uvalue->value.integer.value[0]
	};
	uint32_t sii_ret;

	DEV_DBG(sii9437->dev, "latency = %d\n", latency.data);

	mutex_lock(&(sii9437->fw_lock));
	sii_ret = sii9437_erx_latency_set(sii9437->fw_inst, &latency);
	mutex_unlock(&(sii9437->fw_lock));

	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_erx_latency_set() failed (%d)\n", sii_ret);
		return -EIO;
	}
	return 0;
}

static int snd_ctrl_latency_req_get(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *uvalue)
{
	struct sii9437 *sii9437 = snd_kcontrol_chip(kcontrol);
	mutex_lock(&(sii9437->fw_lock));
	uvalue->value.integer.value[0] = _sii9437_erx_latency_req_query(sii9437);
	mutex_unlock(&(sii9437->fw_lock));
	return 0;
}

#define SND_CTRL_PREFIX	"HDMI eARC "

static struct snd_kcontrol_new snd_ctrls_sii9437[] = {
	[CTRL_IDX_EARC_ENABLE] = {
		.iface		= SNDRV_CTL_ELEM_IFACE_MIXER,
		.name		= SND_CTRL_PREFIX "Enable",
		.access		= SNDRV_CTL_ELEM_ACCESS_READ |
				  SNDRV_CTL_ELEM_ACCESS_WRITE,
		.info		= snd_ctrl_earc_enable_info,
		.get		= snd_ctrl_earc_enable_get,
		.put		= snd_ctrl_earc_enable_put,
	},
	[CTRL_IDX_LINK_STATE] = {
		.iface		= SNDRV_CTL_ELEM_IFACE_MIXER,
		.name		= SND_CTRL_PREFIX "Link",
		.access		= SNDRV_CTL_ELEM_ACCESS_READ,
		.info		= snd_ctrl_link_state_info,
		.get		= snd_ctrl_link_state_get,
	},
	[CTRL_IDX_CHANNEL_COUNT] = {
		.iface		= SNDRV_CTL_ELEM_IFACE_MIXER,
		.name		= SND_CTRL_PREFIX "Channel Count",
		.access		= SNDRV_CTL_ELEM_ACCESS_READ |
				  SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info		= snd_ctrl_channel_count_info,
		.get		= snd_ctrl_channel_count_get,
	},
	[CTRL_IDX_SAMPLE_RATE] = {
		.iface		= SNDRV_CTL_ELEM_IFACE_MIXER,
		.name		= SND_CTRL_PREFIX "Sample Rate",
		.access		= SNDRV_CTL_ELEM_ACCESS_READ |
				  SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info		= snd_ctrl_sample_rate_info,
		.get		= snd_ctrl_sample_rate_get,
	},
	[CTRL_IDX_CSB_VALUE] = {
		.iface		= SNDRV_CTL_ELEM_IFACE_MIXER,
		.name		= SND_CTRL_PREFIX SNDRV_CTL_NAME_IEC958("",CAPTURE,DEFAULT),
		.access		= SNDRV_CTL_ELEM_ACCESS_READ |
				  SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info		= snd_ctrl_csb_info,
		.get		= snd_ctrl_csb_get_default,
	},
	[CTRL_IDX_CSB_MASK] = {
		.iface		= SNDRV_CTL_ELEM_IFACE_MIXER,
		.name		= SND_CTRL_PREFIX SNDRV_CTL_NAME_IEC958("",CAPTURE,MASK),
		.access		= SNDRV_CTL_ELEM_ACCESS_READ,
		.info		= snd_ctrl_csb_info,
		.get		= snd_ctrl_csb_get_mask,
	},
	[CTRL_IDX_CAP_DS] = {
		.iface		= SNDRV_CTL_ELEM_IFACE_MIXER,
		.name		= SND_CTRL_PREFIX "Capabilities Data Structure",
		.access		= SNDRV_CTL_ELEM_ACCESS_READ |
				  SNDRV_CTL_ELEM_ACCESS_WRITE,
		.info		= snd_ctrl_cap_ds_info,
		.get		= snd_ctrl_cap_ds_get,
		.put		= snd_ctrl_cap_ds_put,
	},
	[CTRL_IDX_LATENCY] = {
		.iface		= SNDRV_CTL_ELEM_IFACE_MIXER,
		.name		= SND_CTRL_PREFIX "Latency",
		.access		= SNDRV_CTL_ELEM_ACCESS_READ |
				  SNDRV_CTL_ELEM_ACCESS_WRITE,
		.info		= snd_ctrl_latency_info,
		.get		= snd_ctrl_latency_get,
		.put		= snd_ctrl_latency_put,
	},
	[CTRL_IDX_LATENCY_REQ] = {
		.iface		= SNDRV_CTL_ELEM_IFACE_MIXER,
		.name		= SND_CTRL_PREFIX "Latency Request",
		.access		= SNDRV_CTL_ELEM_ACCESS_READ,
		.info		= snd_ctrl_latency_info,
		.get		= snd_ctrl_latency_req_get,
	},
};

static struct snd_soc_dai_driver soc_dai_sii9437 = {
	.name			= DRV_NAME"-eARC",
	.capture = {
		.stream_name	= "HDMI eARC Capture",
		.channels_min	= 2,
		.channels_max	= 32,
		/* Rates are based on what can be described in an HDMI Short Audio Descriptor */
		.rates		= (SNDRV_PCM_RATE_32000  | SNDRV_PCM_RATE_44100  | SNDRV_PCM_RATE_48000 |	/* Base rates */
							   SNDRV_PCM_RATE_88200  | SNDRV_PCM_RATE_96000 |	/* Base rates x 2 */
							   SNDRV_PCM_RATE_176400 | SNDRV_PCM_RATE_192000),	/* Base rates x 4 */
		.formats	= SNDRV_PCM_FMTBIT_S32_LE,
	},
};

static irqreturn_t sii9437_isr_thread(int irq, void *data)
{
	uint32_t sii_ret;
	struct sii9437 *sii9437 = data;

	if (sii9437->fw_inst) {
		mutex_lock(&(sii9437->fw_lock));

		sii_ret = sii9437_handle(sii9437->fw_inst);
		if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
			dev_err(sii9437->dev, "sii9437_handle() failed err=%d\n", sii_ret);
		}

		mutex_unlock(&(sii9437->fw_lock));
	} else {
		dev_err_ratelimited(sii9437->dev, "ASSERT! IRQ before FW setup!\n");
	}
	return IRQ_HANDLED;
}

static enum hdmi_earc_link sii9437_get_link_state_locked(struct sii9437 *sii9437)
{
	enum sii_arc_mode mode;
	enum hdmi_earc_link state = HDMI_EARC_LINK_DISABLED;
	ASSERT_LOCKED(sii9437);

	mode = _sii9437_arc_mode_get(sii9437);

	switch (mode) {
		case SII_ARC_MODE__EARC:
			state = _sii9437_earc_link_query(sii9437) ? HDMI_EARC_LINK_ONLINE : HDMI_EARC_LINK_OFFLINE;

			if (sii9437->earc_timeout) {
				if (state == HDMI_EARC_LINK_ONLINE) {
					// We shouldn't get here
					dev_err(sii9437->dev, "conflicting states: online and timed out!\n");
				}

				state = HDMI_EARC_LINK_TIMEOUT;
			}
			break;

		case SII_ARC_MODE__ARC:
			state = HDMI_EARC_LINK_DISABLED;
			break;

		case SII_ARC_MODE__NONE:
		default:
			/* Leave state as disabled */
			dev_err(sii9437->dev, "arc mode %s(%d) invalid\n", _arc_mode_to_str(mode), mode);
			break;
	}

	return state;
}

/*
 * The Lattice FW can get stuck sometimes when HPD is toggling rapidly for periods
 * smaller than the HDMI spec specifies.  When it gets stuck the eARC Discovery
 * never times out leaving all ARC/eARC input blocked.  This watchdog triggers
 * after eARC should have been established or timed out.  If neither event has
 * occurred it forces the SII94347 into Legacy ARC passthrough mode.
 */
static void sii9437_hpd_watchdog_timeout(struct work_struct *work)
{
	struct sii9437 *sii9437 = container_of(work, struct sii9437, hpd_watchdog_work.work);
	bool_t hpd;
	enum hdmi_earc_link link;

	mutex_lock(&(sii9437->fw_lock));

	hpd = _sii9437_hpd_query(sii9437);
	link = sii9437_get_link_state_locked(sii9437);

	DEV_DBG(sii9437->dev, "hpd=%d link=%s\n", hpd, _earc_link_to_str(link));

	/* If HPD is still up, we should have selected some audio path by now */
	if (hpd && (link == HDMI_EARC_LINK_OFFLINE)) {
		/* Force a timeout */
		dev_err(sii9437->dev, "watchdog forcing earc timeout\n");
		sii9437->earc_timeout = TRUE;
		snd_ctl_notify(sii9437->snd_card, SNDRV_CTL_EVENT_MASK_VALUE, &(sii9437->snd_kctrls[CTRL_IDX_LINK_STATE]->id));
	}

	mutex_unlock(&(sii9437->fw_lock));
}

/* Needs to be longer than eARC Discovery timeout (300msec) */
#define FW_HPD_WATCHDOG_TIMEOUT (msecs_to_jiffies(1000))

static void sii9437_event_callback(uint32_t *event_flags)
{
	struct sii9437 *sii9437 = g_event_context;
	DEV_DBG(sii9437->dev, "event_flags=%02x\n", *event_flags);
	ASSERT_LOCKED(sii9437);

	/* HPD event */
	if (*event_flags & SII9437_EVENT_FLAGS__HPD_CHNG) {
		bool_t hpd = _sii9437_hpd_query(sii9437);
		bool_t earc_link = _sii9437_earc_link_query(sii9437);

		if (earc_link) {
			/* eARC HPD event */
			DEV_DBG(sii9437->dev, "hdmi earc hpd = %d\n", hpd);
			hdmictl_set_virtual_hpd(hpd);
		}
		else {
			/* Traditional HDMI HPD event */
			DEV_DBG(sii9437->dev, "hdmi hpd = %d\n", hpd);
			if (!hpd) {
				/* Cancel watchdog on HPD low */
				cancel_delayed_work(&(sii9437->hpd_watchdog_work));
			} else {
				/* Schedule watchdog on HPD hi */
				schedule_delayed_work(&(sii9437->hpd_watchdog_work), FW_HPD_WATCHDOG_TIMEOUT);
			}

			/* Reset timeout flag if set */
			if (sii9437->earc_timeout) {
				sii9437->earc_timeout = FALSE;
				snd_ctl_notify(sii9437->snd_card, SNDRV_CTL_EVENT_MASK_VALUE, &(sii9437->snd_kctrls[CTRL_IDX_LINK_STATE]->id));
			}
		}
	}

	/* eARC link event */
	if (*event_flags & SII9437_EVENT_FLAGS__EARC_LINK_CHNG) {
		DEV_DBG(sii9437->dev, "link = %d\n", _sii9437_earc_link_query(sii9437));
		sii9437->earc_timeout = FALSE;
		snd_ctl_notify(sii9437->snd_card, SNDRV_CTL_EVENT_MASK_VALUE, &(sii9437->snd_kctrls[CTRL_IDX_LINK_STATE]->id));
	}

	/* eARC discovery timeout event */
	if (*event_flags & SII9437_EVENT_FLAGS__EARC_DISC_TIMEOUT) {
		/* Fallback to Legacy ARC */
		DEV_DBG(sii9437->dev, "earc discovery timeout\n");
		sii9437->earc_timeout = TRUE;
		snd_ctl_notify(sii9437->snd_card, SNDRV_CTL_EVENT_MASK_VALUE, &(sii9437->snd_kctrls[CTRL_IDX_LINK_STATE]->id));
	}

	/* Latency request from eARC Tx */
	if (*event_flags & SII9437_EVENT_FLAGS__ERX_LATENCY_REQ_CHNG) {
		DEV_DBG(sii9437->dev, "latency request = %d\n", _sii9437_erx_latency_req_query(sii9437));
		snd_ctl_notify(sii9437->snd_card, SNDRV_CTL_EVENT_MASK_VALUE, &(sii9437->snd_kctrls[CTRL_IDX_LATENCY_REQ]->id));
	}

	/* Audio extraction mode change event */
	if (*event_flags & SII9437_EVENT_FLAGS__EXTRACT_MODE_CHNG) {
		enum sii9437_extraction_mode extract_mode = _sii9437_extraction_mode_query(sii9437);
		DEV_DBG(sii9437->dev, "extraction mode = %s\n", _extract_mode_to_str(extract_mode));

		/* SPDIF extraction should be disabled */
		if ((extract_mode == SII9437_EXTRACTION_MODE__SPDIF2) || (extract_mode == SII9437_EXTRACTION_MODE__SPDIF8)) {
			dev_err(sii9437->dev, "ASSERT: eARC extraction preference violation; SPDIF line active\n");
		}
	}

	/* Channel status change event */
	if (*event_flags & SII9437_EVENT_FLAGS__CHAN_STAT_CHNG) {
		uint32_t rate = _sii9437_sample_rate_query(sii9437);

		DEV_DBG(sii9437->dev, "sample rate = %u (%u)\n", rate, _convert_earc_sample_rate_to_alsa(rate));

		if (rate > MAX_EARC_SAMPLE_RATE) {
			dev_warn(sii9437->dev, "eARC sample rate exceeds maximum (%u)\n", rate);
		}

#ifdef DEBUG_SII9437
		// Log CSB change in debug only
		{
			uint8_t csb[CSB_SIZE];
			_sii9437_channel_status_query(sii9437, csb, sizeof(csb));
			DEV_DBG(sii9437->dev, "channel status = %*ph\n", (int)sizeof(csb), &csb);
			DEV_DBG(sii9437->dev, "mute bit = %s\n", (csb[18] & 0x20) ? "on" : "off");
		}
#endif // DEBUG_SII9437
	}

	/* FW assert event */
	if (*event_flags & SII9437_EVENT_FLAGS__ASSERT) {
		uint32_t assert;

		if (sii9437_assert_query(sii9437->fw_inst, &assert) != SII_RETURN_VALUE__SUCCESS) {
			dev_err(sii9437->dev, "sii9437_assert_query() failed\n");
		} else {
			dev_err(sii9437->dev, "firmware assert = %08x\n", assert);
		}
	}

	/* Error if we don't know what an event means (new FW ver?) */
	if (*event_flags &
	    ~(SII9437_EVENT_FLAGS__HPD_CHNG |
	      SII9437_EVENT_FLAGS__EARC_LINK_CHNG |
	      SII9437_EVENT_FLAGS__EARC_DISC_TIMEOUT |
	      SII9437_EVENT_FLAGS__ERX_LATENCY_REQ_CHNG |
	      SII9437_EVENT_FLAGS__EXTRACT_MODE_CHNG |
	      SII9437_EVENT_FLAGS__CHAN_STAT_CHNG |
	      SII9437_EVENT_FLAGS__ASSERT)) {
		dev_err(sii9437->dev, "Unknown event = %02x\n", *event_flags);
	}

	DEV_DBG(sii9437->dev, "earc state = %s\n", _earc_link_to_str(sii9437_get_link_state_locked(sii9437)));
}

static void sii9437_reset(struct sii9437 *sii9437)
{
	DEV_DBG(sii9437->dev, "\n");
	gpio_set_value(sii9437->reset_gpio, 0);
	msleep(1);
	gpio_set_value(sii9437->reset_gpio, 1);
	msleep(10);
}

static int sii9437_setup(struct sii9437 *sii9437)
{
	uint32_t sii_ret;
	struct sii9437_config config = {
		.dev_id = SII9437_INST_1,
		.callback_func = sii9437_event_callback,
		.log_fifo_size = 128,
		.i2c_addr_1 = SII9437_INST_1_I2C_ADDR_PAGE1,
		.i2c_addr_2 = SII9437_INST_1_I2C_ADDR_PAGE2};
	char fw_ver[32];
	uint32_t hw_id;
	uint8_t hw_rev;
	int err = 0;

	sii9437->earc_timeout = FALSE;

	sii9437_software_version_query(fw_ver, sizeof(fw_ver));
	dev_info(sii9437->dev, "sii9437 firmware v%s\n", fw_ver);

	/* Setup i2c pages and reset IC */
	sii_register_regmap_pages(sii9437->reg_main, sii9437->reg_rx, sii9437->reg_phy);
	sii9437_reset(sii9437);

	mutex_lock(&(sii9437->fw_lock));

	/* Start firmware */
	sii_ret = sii9437_create(&config, &(sii9437->fw_inst));
	if (sii_ret != SII_RETURN_VALUE__SUCCESS) {
		dev_err(sii9437->dev, "sii9437_create() failed (%d)\n", sii_ret);
		err = -EIO;
		goto failed_setup;
	}

	/* Setup initial configuration */

	/*
	 * Start in EARC mode regardless of HPD
	 * If HPD = 0 we wait for HPD
	 * If HPD = 1 then eARC will connect or timeout fairly quickly
	 */
	_sii9437_arc_mode_set(sii9437, SII_ARC_MODE__EARC);

	/* Prefer I2S for multi-channel audio (instead of SPDIF8 which is non-standard) */
	_sii9437_preferred_extraction_mode_set(sii9437, SII9437_PREF_EXTRACTION_MODE__I2S);

	/* Get chip and fw rev info */
	sii9437_chip_id_query(sii9437->fw_inst, &(hw_id));
	sii9437_chip_version_query(sii9437->fw_inst, &(hw_rev));

	dev_info(sii9437->dev, "sii9437 online (hw_id=%x hw_rev=%02x)\n", hw_id, hw_rev);

failed_setup:
	mutex_unlock(&(sii9437->fw_lock));
	return err;
}

static void procfs_remove(struct sii9437 *sii9437)
{
	if (sii9437->procfs_test_pll) {
		remove_proc_entry(PROCFS_TEST_PLL, sii9437->procfs_dir);
	}
	if (sii9437->procfs_test_i2s_map) {
		remove_proc_entry(PROCFS_TEST_I2S_MAP, sii9437->procfs_dir);
	}
	if (sii9437->procfs_log) {
		remove_proc_entry(PROCFS_LOG, sii9437->procfs_dir);
	}
	if (sii9437->procfs_dir) {
		remove_proc_entry(PROCFS_DIR, NULL);
	}
}

static int procfs_init(struct sii9437 *sii9437)
{
	sii9437->procfs_dir = proc_mkdir(PROCFS_DIR, NULL);
	if (!sii9437->procfs_dir) {
		dev_err(sii9437->dev, "failed to create procfs dir %s", PROCFS_DIR);
		goto failed_procfs;
	}

	sii9437->procfs_log = proc_create_data(PROCFS_LOG,
							(S_IRUSR | S_IRGRP | S_IROTH),
							sii9437->procfs_dir,
							&procfs_log_ops,
							sii9437);
	if (!sii9437->procfs_log) {
		dev_err(sii9437->dev, "failed to create procfs %s\n", PROCFS_LOG);
		goto failed_procfs;
	}

	sii9437->procfs_test_i2s_map = proc_create_data(PROCFS_TEST_I2S_MAP,
							(S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH),
							sii9437->procfs_dir,
							&procfs_test_i2s_map_ops,
							sii9437);
	if (!sii9437->procfs_test_i2s_map) {
		dev_err(sii9437->dev, "failed to create procfs %s\n", PROCFS_TEST_I2S_MAP);
		goto failed_procfs;
	}

	sii9437->procfs_test_pll = proc_create_data(PROCFS_TEST_PLL,
							(S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH),
							sii9437->procfs_dir,
							&procfs_test_pll_ops,
							sii9437);
	if (!sii9437->procfs_test_pll) {
		dev_err(sii9437->dev, "failed to create procfs %s\n", PROCFS_TEST_PLL);
		goto failed_procfs;
	}

	return 0;
failed_procfs:
	procfs_remove(sii9437);
	return -1;
}

int snd_codec_probe(struct snd_soc_codec *codec)
{
	int idx;
	int ret;
	struct sii9437 *sii9437 = snd_soc_codec_get_drvdata(codec);
	DEV_DBG(sii9437->dev, "snd_soc_codec = %p\n", codec);

	/* Get provisioned ALSA structures */
	sii9437->snd_codec = codec;
	sii9437->snd_card = codec->component.card->snd_card;
	DEV_DBG(sii9437->dev, "snd_soc_card = %s\n", codec->component.card->name);

	/* Break build if these arrays are not the same size */
	BUILD_BUG_ON(ARRAY_SIZE(snd_ctrls_sii9437) != ARRAY_SIZE(sii9437->snd_kctrls));

	/*
	 * Create the controls.
	 *
	 * Normally you can just instantiate all the controls in a static snd_kcontrol_new array which
	 * is referenced in snd_soc_codec_driver.component_driver.controls.  However to use
	 * snd_ctl_notify() you need the snd_kcontrol->id.  There appears to be no way to get from
	 * the static snd_kcontrol_new to the instantiated snd_kcontrol structure without resorting to
	 * brute force string searches of all of the ALSA controls.  So to use notify I just instantiate
	 * the controls one at a time and stach the snd_kcontrol pointers.
	 *
	 * This seems to be the way other ALSA drivers do it (sound/i2c/other/ak4113.c)...
	 */
	for (idx = 0; idx < ARRAY_SIZE(snd_ctrls_sii9437); idx++) {
		int err;

		/* Allocate the kcontrol structure */
		sii9437->snd_kctrls[idx] = snd_ctl_new1(&(snd_ctrls_sii9437[idx]), sii9437);
		if (!sii9437->snd_kctrls[idx]) {
			dev_err(sii9437->dev, "failed to create kcontrol '%s'\n", snd_ctrls_sii9437[idx].name);
			return -EIO;
		}

		/* Assign it to the card */
		err = snd_ctl_add(codec->component.card->snd_card, sii9437->snd_kctrls[idx]);
		if (err) {
			dev_err(sii9437->dev, "failed to add kcontrol '%s' (%d)\n", snd_ctrls_sii9437[idx].name, err);
			return -EIO;
		}

		DEV_DBG(sii9437->dev, "kcontrol '%s' assigned numid=%d name='%s'\n",
			snd_ctrls_sii9437[idx].name, sii9437->snd_kctrls[idx]->id.numid, sii9437->snd_kctrls[idx]->id.name);
	}

	/* Now that all the ALSA stuff is done we can enable the codec */

	/* Setup Lattice FW */
	ret = sii9437_setup(sii9437);
	if (ret) {
		return ret;
	}

	/* Request IRQ for interrupt gpio */
	sii9437->irq_num = gpio_to_irq(sii9437->irq_gpio);
	ret = devm_request_threaded_irq(sii9437->dev, sii9437->irq_num,
					NULL,
					sii9437_isr_thread,
					IRQF_TRIGGER_LOW | IRQF_ONESHOT,
					DRV_NAME, sii9437);
	if (ret) {
		dev_err(sii9437->dev, "irq request failed (err=%d)\n", ret);
		return ret;
	}

	return 0;
}

static struct snd_soc_codec_driver soc_codec_sii9437 = {
	.probe	= snd_codec_probe,
};

static int sii9437_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id)
{
	struct sii9437 *sii9437;
	int ret = 0;
	DEV_DBG(&(i2c_client->dev), "\n");

	/* Sanity check there is no other instance in use */
	if (g_event_context != NULL) {
		dev_err(&(i2c_client->dev), "device already active\n");
		return -EBUSY;
	}

	/* Setup private data structure */
	sii9437 = devm_kzalloc(&i2c_client->dev, sizeof(*sii9437), GFP_KERNEL);
	if (!sii9437) {
		dev_err(sii9437->dev, "mem allocation failed\n");
		return -ENOMEM;
	}
	i2c_set_clientdata(i2c_client, sii9437);
	sii9437->dev		= &(i2c_client->dev);
	sii9437->i2c_main	= i2c_client;
	g_event_context		= sii9437;
	dev_set_drvdata(sii9437->dev, sii9437);

	/* Allocate GPIOs */
	sii9437->mute_gpio = of_get_named_gpio(sii9437->i2c_main->dev.of_node, "mute-gpio", 0);
	if (!gpio_is_valid(sii9437->mute_gpio)) {
		dev_err(sii9437->dev, "mute gpio not found\n");
		goto failed_probe;
	}
	ret = devm_gpio_request_one(sii9437->dev, sii9437->mute_gpio, GPIOF_IN, "sii9437_mute");
	if (ret) {
		dev_err(sii9437->dev, "mute gpio (%d) request failed (err=%d)\n", sii9437->mute_gpio, ret);
		goto failed_probe;
	}
	DEV_DBG(sii9437->dev, "mute gpio (%d) acquired\n", sii9437->mute_gpio);

	sii9437->reset_gpio = of_get_named_gpio(sii9437->i2c_main->dev.of_node, "reset-gpio", 0);
	if (!gpio_is_valid(sii9437->reset_gpio)) {
		dev_err(sii9437->dev, "reset gpio not found\n");
		goto failed_probe;
	}
	ret = devm_gpio_request_one(sii9437->dev, sii9437->reset_gpio, GPIOF_OUT_INIT_LOW, "sii9437_reset");
	if (ret) {
		dev_err(sii9437->dev, "reset gpio (%d) request failed (err=%d)\n", sii9437->reset_gpio, ret);
		goto failed_probe;
	}
	DEV_DBG(sii9437->dev, "reset gpio (%d) acquired\n", sii9437->reset_gpio);

	sii9437->irq_gpio = of_get_named_gpio(sii9437->i2c_main->dev.of_node, "irq-gpio", 0);
	if (!gpio_is_valid(sii9437->irq_gpio)) {
		dev_err(sii9437->dev, "irq gpio not found\n");
		goto failed_probe;
	}
	ret = devm_gpio_request_one(sii9437->dev, sii9437->irq_gpio, GPIOF_IN, "sii9437_irq");
	if (ret) {
		dev_err(sii9437->dev, "irq gpio (%d) request failed (err=%d)\n", sii9437->irq_gpio, ret);
		goto failed_probe;
	}
	DEV_DBG(sii9437->dev, "irq gpio (%d) acquired (state=%d)\n", sii9437->irq_gpio, gpio_get_value(sii9437->irq_gpio));

	/* Create a regmap for the main page of i2c registers */
	sii9437->reg_main = devm_regmap_init_i2c(sii9437->i2c_main, &regmap_config_main);
	if (IS_ERR(sii9437->reg_main)) {
		ret = PTR_ERR(sii9437->reg_main);
		goto failed_probe;
	}

	/* Create a client and regmap for the rx page of i2c registers */
	sii9437->i2c_rx = i2c_new_dummy(sii9437->i2c_main->adapter, SII9437_INST_1_I2C_ADDR_PAGE1 >> 1);
	if (!sii9437->i2c_rx) {
		dev_err(sii9437->dev, "rx dummy device creation failed\n");
		ret = -ENODEV;
		goto failed_probe;
	}

	sii9437->reg_rx = devm_regmap_init_i2c(sii9437->i2c_rx, &regmap_config_rx);
	if (IS_ERR(sii9437->reg_rx)) {
		ret = PTR_ERR(sii9437->reg_rx);
		goto failed_probe;
	}

	/* Create a client and regmap for the phy page of i2c registers */
	sii9437->i2c_phy = i2c_new_dummy(sii9437->i2c_main->adapter, SII9437_INST_1_I2C_ADDR_PAGE2 >> 1);
	if (!sii9437->i2c_phy) {
		dev_err(sii9437->dev, "phy dummy device creation failed\n");
		ret = -ENODEV;
		goto failed_probe;
	}

	sii9437->reg_phy = devm_regmap_init_i2c(sii9437->i2c_phy, &regmap_config_phy);
	if (IS_ERR(sii9437->reg_phy)) {
		ret = PTR_ERR(sii9437->reg_phy);
		goto failed_probe;
	}

	/* Setup watchdog and mutex */
	INIT_DELAYED_WORK(&(sii9437->hpd_watchdog_work), &sii9437_hpd_watchdog_timeout);
	mutex_init(&(sii9437->fw_lock));

	/* Setup sysfs */
	ret = sysfs_create_group(&(sii9437->dev->kobj), &sii9437_attr_group);
	if (ret) {
		goto failed_probe;
	}

	/* Setup procfs */
	ret = procfs_init(sii9437);
	if (ret) {
		goto failed_probe;
	}

	/* Register driver as an ALSA  codec */
	ret = snd_soc_register_codec(sii9437->dev,
				     &soc_codec_sii9437,
				     &soc_dai_sii9437, 1);
	if (ret) {
		dev_err(sii9437->dev, "alsa codec registration failed (err=%d)\n", ret);
		goto failed_probe;
	}

	DEV_DBG(sii9437->dev, "probe complete!\n");
	return ret;
failed_probe:
	procfs_remove(sii9437);
	if (sii9437->i2c_phy) {
		i2c_unregister_device(sii9437->i2c_phy);
	}
	if (sii9437->i2c_rx) {
		i2c_unregister_device(sii9437->i2c_rx);
	}

	return ret;
}

static int
sii9437_i2c_remove(struct i2c_client *i2c_client)
{
	struct sii9437 *sii9437 = i2c_get_clientdata(i2c_client);

	snd_soc_unregister_codec(sii9437->dev);

	disable_irq(sii9437->irq_num);
	cancel_delayed_work_sync(&(sii9437->hpd_watchdog_work));

	procfs_remove(sii9437);
	sysfs_remove_group(&(sii9437->dev->kobj), &sii9437_attr_group);

	sii9437_delete(sii9437->fw_inst);
	sii_deregister_regmap_pages();

	i2c_unregister_device(sii9437->i2c_phy);
	i2c_unregister_device(sii9437->i2c_rx);

	g_event_context = NULL;

	return 0;
}

static const struct i2c_device_id sii9437_i2c_id[] = {
	{ DRV_NAME, 0 },
	{ }
};
MODULE_DEVICE_TABLE(sii9437_i2c, sii9437_i2c_id);

static struct of_device_id sii9437_ids[] = {
	{ .compatible = "lattice,sii9437" },
	{ /* sentinel */ }
};

static struct i2c_driver sii9437_i2c_driver = {
	.driver = {
		.name   = DRV_NAME,
		.owner  = THIS_MODULE,
		.of_match_table = sii9437_ids,
	},
	.id_table  = sii9437_i2c_id,
	.probe     = sii9437_i2c_probe,
	.remove    = sii9437_i2c_remove,
};

static void __exit sii9437_exit(void)
{
	PRINTK_DBG("\n");
	i2c_del_driver(&sii9437_i2c_driver);
	return;
}
module_exit(sii9437_exit);

static int __init sii9437_init(void)
{
	int ret;
	PRINTK_DBG("\n");

	ret = i2c_add_driver(&sii9437_i2c_driver);
	if (ret) {
		printk(KERN_ERR"init failed (%d)\n", ret);
	}
	return ret;
}
module_init(sii9437_init);

MODULE_LICENSE ("GPL");
MODULE_AUTHOR ("Sonos, Inc.");
MODULE_DESCRIPTION ("Lattice SII9437 HDMI eARC Module");
