/*
 * Sonos Wall Mount Driver
 *
 * Copyright (c) 2019-2020 Sonos Inc.
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/sysfs.h>
#include <linux/delay.h>
#include <linux/of_gpio.h>
#include <linux/sysfs.h>
#include "blackbox.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "sensors_dev.h"

#define WALL_MOUNT_POLL_TIME	msecs_to_jiffies(5000)
#define WALL_MOUNT_GPIO_NAME	"wall-mount"

struct wall_mount_entry {
	char *name;
	int gpio;
	enum of_gpio_flags gpio_flags;
	enum HWEVTQ_EventSource source;
};

static struct wall_mount_entry wme = {
	.name = WALL_MOUNT_GPIO_NAME,
	.source = HWEVTQSOURCE_ORIENTATION,
	.gpio = -EINVAL };

static struct delayed_work wall_mount_work;

static void wall_mount_get_gpio(struct wall_mount_entry *wme)
{
	struct device_node *np;

	np = of_find_node_by_name(NULL, "misc-gpio");
	if (np == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "wall_mount: sonos misc-gpio block missing from DTB");
		return;
	}
	wme->gpio = of_get_named_gpio_flags(np, wme->name, 0, &wme->gpio_flags);
}

static void wall_mount_poll(struct work_struct *work)
{
	static bool prev_state = false;
	static unsigned int polls = 0;
	int current_state;
	enum HWEVTQ_EventInfo orient_status;

	current_state = wall_mount_get_current();
	if (prev_state != current_state) {
		orient_status = current_state ? HWEVTQINFO_ORIENT_HORIZONTAL_WALL : HWEVTQINFO_ORIENT_HORIZONTAL;
		event_queue_send_event(wme.source, orient_status);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event(wme.source, orient_status);
#endif
		prev_state = current_state;
	}

	polls++;
	schedule_delayed_work(&wall_mount_work, WALL_MOUNT_POLL_TIME);
}

int wall_mount_get_current(void)
{
	int result = 0;
	if (gpio_is_valid(wme.gpio)) {
		result = gpio_get_value(wme.gpio);
	} else {
		return 0;
	}

	return !result;
}

void wall_mount_init(void)
{
	int error;

	wall_mount_get_gpio(&wme);
	if (!gpio_is_valid(wme.gpio)) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s gpio not found in DTB", wme.name);
		return;
	}

	error = gpio_request_one(wme.gpio, GPIOF_DIR_IN, wme.name);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "wall_mount: unable to init driver: err %d", error);
		return;
	}

	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "wall mount driver init success.");
	INIT_DELAYED_WORK(&wall_mount_work, wall_mount_poll);
	schedule_delayed_work(&wall_mount_work, WALL_MOUNT_POLL_TIME);
}

void wall_mount_exit(void)
{
	cancel_delayed_work_sync(&wall_mount_work);

	if (gpio_is_valid(wme.gpio)) {
		gpio_free(wme.gpio);
	}
}
