/*
 * Copyright (c) 2017-2019 Sonos Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 *
 * TMP102 thermal sensor - we use the native kernel driver to read temps from
 * hardware, but this shim formats the temp sensor data in a "Sonos" way
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/thermal.h>
#include <linux/err.h>
#include "blackbox.h"
#include "sensors_dev.h"
#include "sensors_hal.h"

#define ADJUST_TMP102_UNITS 1000

struct thermal_zone_device *tmp102_tz_amp = NULL, *tmp102_tz_cpu = NULL;

int tmp102shim_read_temp(int inst, int *temp)
{
	struct thermal_zone_device *tmp102_tz = NULL;
	int status=0;

	if (inst < THERMAL_SENSOR_NUM ) {
		if (inst==THERMAL_SENSOR_AMP) {
			tmp102_tz = tmp102_tz_amp;
		}
		else if (inst == THERMAL_SENSOR_CPU) {
			tmp102_tz = tmp102_tz_cpu;
		}
		if (tmp102_tz && (tmp102_tz->ops ) && (tmp102_tz->ops->get_temp)) {
			status = (tmp102_tz->ops->get_temp)(tmp102_tz, temp);
			*temp = (*temp)/ADJUST_TMP102_UNITS;
		} else {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Failure in tmp102shim to get read temp ptr.\n");
		}
	}
	else {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Failure in tmp102shim to read temp\n");
	}
	return(status);
}

void tmp102shim_init(void)
{
	int temp_reading;
	int status;
	struct thermal_zone_device *tzd;

	if (thermal_mgmt_map_device(THERMAL_SENSOR_CPU, "CPU", "tmp102", tmp102shim_read_temp)) {
		tzd = thermal_zone_get_zone_by_name("cputhermal");
		if (!IS_ERR(tzd)) {
			tmp102_tz_cpu = tzd;
			status = tmp102shim_read_temp(THERMAL_SENSOR_CPU, &temp_reading);
		}
	}

	if (thermal_mgmt_map_device(THERMAL_SENSOR_AMP, "AMP", "tmp102", tmp102shim_read_temp)) {
		tzd = thermal_zone_get_zone_by_name("ampthermal");
		if (!IS_ERR(tzd)) {
			tmp102_tz_amp = tzd;
			status = tmp102shim_read_temp(THERMAL_SENSOR_AMP, &temp_reading);
		}
	}

	return;
}

void tmp102shim_exit(void)
{
	tmp102_tz_amp = NULL;
	tmp102_tz_cpu = NULL;
	return;
}
