/*
 * Copyright (c) 2018-2021 Sonos Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */


#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/sysfs.h>
#include <linux/delay.h>
#include <linux/kthread.h>
#include <linux/debugfs.h>
#include <asm/byteorder.h>
#include <asm/uaccess.h>
#include "blackbox.h"
#include "sensors_dev.h"
#include "sensors_hal.h"
#include "mdp.h"

#include "event_queue_api.h"

#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif

#include "ti_opt3001_als_defs.h"

#define OPT3001_INT32_MAX 0x7FFFFFFF

extern struct manufacturing_data_page2 sys_mdp2;

static const struct sensors_als_light_lims opt3001_light_cond_tbl[] = {
		[LGT_PITCH_BLACK]		= {.min = 0,		.max = 999},
		[LGT_VERY_DARK]			= {.min = 1000,		.max = 4999},
		[LGT_DARK_INDOORS]		= {.min = 5000,		.max = 19999},
		[LGT_DIM_INDOORS]		= {.min = 20000,	.max = 39999},
		[LGT_NORMAL_INDOORS]		= {.min = 40000,	.max = 99999},
		[LGT_BRIGHT_INDOORS]		= {.min = 100000,	.max = 499999},
		[LGT_DIM_OUTDOORS]		= {.min = 500000,	.max = 999999},
		[LGT_CLOUDY_OUTDOORS]		= {.min = 1000000,	.max = 2999999},
		[LGT_DIRECT_SUNLIGHT]		= {.min = 3000000,	.max = 999999999},
};

struct opt3001_drv {
	struct dentry *debugfs;
	bool sim_mode;
	int32_t sim_result;
	struct i2c_client *client;
	struct task_struct *task;
};

static struct i2c_client *client_list[OPT3001_MAX_CLIENTS];
static int opt3001_driver_init = 0;
static int32_t g_comp_coeff = COMP_COEFF_DEFAULT;

static int32_t disable_events = 0;

static inline struct i2c_client *get_i2c_client(void)
{
	return client_list[0];
}

static int32_t opt3001_i2c_read_word(struct i2c_client *client, uint8_t reg)
{
	int32_t error;
	error = i2c_smbus_read_word_data(client, reg);
	if (error < 0) {
		return error;
	}

	return be16_to_cpu(error);
}

static int32_t opt3001_i2c_write_word(struct i2c_client *client, uint8_t reg, uint16_t value)
{
	int32_t error;
	error = i2c_smbus_write_word_data(client, reg, cpu_to_be16(value));
	return error;
}


static int32_t opt3001_get_lux_val(struct i2c_client *client)
{
	int32_t result;
	uint16_t r;
	uint8_t e;
	int64_t lux;
	struct opt3001_drv *pdata = i2c_get_clientdata(client);

	if (!pdata) {
		return -EINVAL;
	}

	if (pdata->sim_mode) {
		return pdata->sim_result;
	}

	result = opt3001_i2c_read_word(client, OPT3001_R_RESULT);
	if (result < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "ti_opt3001: invalid read operation on result reg: %04x", result);
		return result;
	}

	e = (result & OPT3001_EXPNT_BITS) >> 0xC;
	r = result & OPT3001_FRACT_BITS;

	result = (int32_t)(r << e);

	lux = ((int64_t)result * g_comp_coeff) / FIXED_POINT_DIVISOR;

	if (lux > OPT3001_INT32_MAX) {
		return OPT3001_INT32_MAX;
	} else {
		return (int32_t)lux;
	}
}

static int opt3001_get_light_cond(struct i2c_client *client, enum LIGHT_COND *step)
{
	int32_t result;
	int i;

	result = opt3001_get_lux_val(client);
	if (result < 0) {
		return -EFAULT;
	}

	for (i = 0; i < LGT_NUM_CONDS; i++) {
		if (result >= opt3001_light_cond_tbl[i].min &&
			result <= opt3001_light_cond_tbl[i].max)
		{
			*step = i;
			return 0;
		}
	}

	return -1;
}

static int opt3001_poll_thread(void *pdata)
{
	while (!kthread_should_stop()) {
		msleep_interruptible(OPT3001_POLLING_DELAY_MS);

		if (!disable_events) {
			event_queue_send_event(HWEVTQSOURCE_LIGHT_SENSOR, HWEVTQINFO_UPDATE);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(HWEVTQSOURCE_LIGHT_SENSOR, HWEVTQINFO_UPDATE);
#endif
		}
    }

	return 0;
}

static ssize_t print_reg(struct i2c_client *client, char *buf, uint8_t reg)
{
	int32_t ret;

	ret = opt3001_i2c_read_word(client, reg);
	if (ret < 0) {
		return scnprintf(buf, PAGE_SIZE, "err: %02x\n", ret);
	}

	return scnprintf(buf, PAGE_SIZE, "%04x\n", ret);
}

static ssize_t lux_val_show(struct device_driver *drv, char *buf)
{
	int32_t result;
	struct i2c_client *client = get_i2c_client();

	if (!opt3001_driver_init) {
		return scnprintf(buf, PAGE_SIZE, "ti_opt3001 device not ready\n");
	}

	result = opt3001_get_lux_val(client);
	if (result < 0) {
		return scnprintf(buf, PAGE_SIZE, "Invalid result register read: %02x\n", result);
	}

	return scnprintf(buf, PAGE_SIZE, "%d\n", result);
}

static ssize_t result_reg_show(struct device_driver *drv, char *buf)
{
	struct i2c_client *client = get_i2c_client();

	if (!opt3001_driver_init) {
		return scnprintf(buf, PAGE_SIZE, "ti_opt3001 device not ready\n");
	}

	return print_reg(client, buf, OPT3001_R_RESULT);
}

static ssize_t config_reg_show(struct device_driver *drv, char *buf)
{
	struct i2c_client *client = get_i2c_client();

	if (!opt3001_driver_init) {
		return scnprintf(buf, PAGE_SIZE, "ti_opt3001 device not ready\n");
	}

	return print_reg(client, buf, OPT3001_R_CONFIG);
}

static ssize_t regs_show(struct device_driver *drv, char *buf)
{
	int reg;
	uint16_t val[OPT3001_NUM_REGS] = {0};
	ssize_t count = 0;
	struct i2c_client *client = get_i2c_client();

	if (!opt3001_driver_init) {
		return scnprintf(buf, PAGE_SIZE, "ti_opt3001 device not ready\n");
	}

	for (reg = 0; reg < OPT3001_NUM_REGS; reg++) {
		int32_t ret = opt3001_i2c_read_word(client, reg);
		if (ret < 0) {
			int r;
			for (r = reg; r < OPT3001_NUM_REGS; r++) {
				val[r] = 0;
			}
			return scnprintf(buf, PAGE_SIZE, "i2c read error on reg %02x\n", reg);
		}
		val[reg] = ret;
	}

	for (reg = 0; reg < OPT3001_NUM_REGS; reg++) {
		if ((reg%16) == 0) {
			count += scnprintf(buf+count, PAGE_SIZE,  "%2x: ", reg);
		}
		count += scnprintf(buf+count, PAGE_SIZE, "%04x ", val[reg]);
	}
	count += scnprintf(buf+count, PAGE_SIZE, "\n");
	return count;
}

static ssize_t light_cond_show(struct device_driver *drv, char *buf)
{
	enum LIGHT_COND light_cond;
	int err;
	struct i2c_client *client = get_i2c_client();

	if (!opt3001_driver_init) {
		return scnprintf(buf, PAGE_SIZE, "ti_opt3001 device not ready\n");
	}

	err = opt3001_get_light_cond(client, &light_cond);
	if (err) {
		return scnprintf(buf, PAGE_SIZE, "ti_opt3001 unable to read light condition\n");
	}
	return scnprintf(buf, PAGE_SIZE, "%u\n", light_cond);
}

static DRIVER_ATTR_RO(light_cond);
static DRIVER_ATTR_RO(lux_val);
static DRIVER_ATTR_RO(result_reg);
static DRIVER_ATTR_RO(config_reg);
static DRIVER_ATTR_RO(regs);

static struct attribute *opt3001_sysfs_attrs[] = {
	&driver_attr_light_cond.attr,
	&driver_attr_lux_val.attr,
	&driver_attr_result_reg.attr,
	&driver_attr_config_reg.attr,
	&driver_attr_regs.attr,
	NULL,
};

static const struct attribute_group opt3001_attr_group = {
	.attrs = opt3001_sysfs_attrs
};

static const struct attribute_group *opt3001_attr_groups[] = {
	&opt3001_attr_group,
	NULL,
};

static int sim_mode_set(void *data, u64 val)
{
	struct opt3001_drv *pdata = data;

	if (val != 0 && val != 1) {
		return -EINVAL;
	}

	if (!pdata) {
		return -EFAULT;
	}

	pdata->sim_mode = val;

	return 0;
}

static int sim_mode_get(void *data, u64 *val)
{
	struct opt3001_drv *pdata = data;

	if (!pdata) {
		return -EFAULT;
	}

	*val = pdata->sim_mode;

	return 0;
}

DEFINE_SIMPLE_ATTRIBUTE(sim_mode_fops, sim_mode_get,
			sim_mode_set, "%llu\n");

static int sim_lux_set(void *data, u64 val)
{
	struct opt3001_drv *pdata = data;
	int32_t sval = val;

	if (sval < 0 || sval > 9999999) {
		return -EINVAL;
	}

	if (!pdata) {
		return -EFAULT;
	}

	pdata->sim_result = sval;

	return 0;

}

static int sim_lux_get(void *data, u64 *val)
{
	struct opt3001_drv *pdata = data;

	if (!pdata) {
		return -EFAULT;
	}

	*val = (u64) pdata->sim_result;

	return 0;
}

DEFINE_SIMPLE_ATTRIBUTE(sim_lux_fops, sim_lux_get,
			sim_lux_set, "%llu\n");

static int disable_events_set(void *data, u64 val)
{
	struct opt3001_drv *pdata = data;
        int32_t sval = val;

	if (!pdata) {
                return -EFAULT;
        }

        if (sval !=  0 && sval != 1) {
                return -EINVAL;
        }

        disable_events = sval;

        return 0;

}

static int disable_events_get(void *data, u64 *val)
{
	struct opt3001_drv *pdata = data;

	if (!pdata) {
                return -EFAULT;
        }

	*val = disable_events;
        return 0;
}

DEFINE_SIMPLE_ATTRIBUTE(disable_events_fops, disable_events_get,
			disable_events_set, "%llu\n");

static void debugfs_create_common(struct i2c_client *client)
{
	struct opt3001_drv *pdata = i2c_get_clientdata(client);

	if (pdata) {
		pdata->debugfs = debugfs_create_dir(OPT3001_DRIVER_NAME, NULL);

		debugfs_create_file("sim_mode", 0444, pdata->debugfs, pdata,
				&sim_mode_fops);

		debugfs_create_file("sim_lux", 0444, pdata->debugfs, pdata,
				&sim_lux_fops);

		debugfs_create_file("disable_events", 0444, pdata->debugfs,
				&disable_events, &disable_events_fops);
	}
}

static int opt3001_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id)
{
	int32_t ret;
	uint16_t value = 0;
	struct opt3001_drv *drv_data;

	drv_data = devm_kzalloc(&i2c_client->dev, sizeof(*drv_data), GFP_KERNEL);
	if (!drv_data) {
		return -ENOMEM;
	}

	ret = opt3001_i2c_read_word(i2c_client, OPT3001_R_MFGID);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "Error reading MFG_ID register");
		return -EIO;
	}

	value = ret;
	if (value != OPT3001_E_MFGID) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "Invalid MFG ID %x, expected %x", value, OPT3001_E_MFGID);
		return -EINVAL;
	}

	ret = opt3001_i2c_read_word(i2c_client, OPT3001_R_DEVID);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "Error reading DEV_ID register");
		return -EIO;
	}

	value = ret;
	if (value != OPT3001_E_DEVID) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "Invalid DEV ID %x, expected %x", value, OPT3001_E_DEVID);
		return -EINVAL;
	}

	ret = opt3001_i2c_write_word(i2c_client, OPT3001_R_CONFIG, OPT3001_DEFCFG);
	if (ret < 0) {
		bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Error writing CONFIG register");
		return -EIO;
	}

	drv_data->client = i2c_client;
	client_list[0] = i2c_client;
	i2c_set_clientdata(i2c_client, (void *) drv_data);
	debugfs_create_common(i2c_client);

	drv_data->task = kthread_run(opt3001_poll_thread, (void *) i2c_client, "als_poll");
	if (IS_ERR(drv_data->task)) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "ti_opt3001: unable to start kthread.");
		return -EIO;
	}

	if (sys_mdp2.mdp2_sigdata.variant == MDP_VARIANT_WHITE) {
		of_property_read_u32(i2c_client->dev.of_node, "ccoef-wht", &g_comp_coeff);
	} else if (sys_mdp2.mdp2_sigdata.variant == MDP_VARIANT_BLACK) {
		of_property_read_u32(i2c_client->dev.of_node, "ccoef-blk", &g_comp_coeff);
	} else if (sys_mdp2.mdp2_sigdata.variant == MDP_VARIANT_SHADOW) {
		of_property_read_u32(i2c_client->dev.of_node, "ccoef-blk", &g_comp_coeff);
	}

	if (g_comp_coeff != COMP_COEFF_DEFAULT) {
		bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_INFO, "assigned optical compensation coefficient: %d", g_comp_coeff);
	} else {
		bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_INFO, "assigned default optical compensation coefficient: %d", g_comp_coeff);
	}

	bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_INFO, "registered");
	opt3001_driver_init = 1;
	return 0;
}

bool opt3001_driver_ready(void)
{
	return (opt3001_driver_init == 1);
}

int opt3001_i2c_remove(struct i2c_client *i2c_client)
{
	struct opt3001_drv *pdata = i2c_get_clientdata(i2c_client);

	if (pdata) {
		debugfs_remove_recursive(pdata->debugfs);
		kthread_stop(pdata->task);
	}
	return 0;
}

static const struct i2c_device_id opt3001_i2c_id[] = {
	{ OPT3001_DRIVER_NAME, 0 },
	{  }
};
MODULE_DEVICE_TABLE(i2c, opt3001_i2c_id);

static struct of_device_id ti_opt3001_ids[] = {
	{ .compatible = "fsl,ti-opt3001" },
	{  }
};

static struct i2c_driver opt3001_i2c_driver = {
	.driver = {
		.name		= OPT3001_DRIVER_NAME,
		.owner		= THIS_MODULE,
		.groups		= opt3001_attr_groups,
		.of_match_table	= ti_opt3001_ids,
	},
	.id_table	= opt3001_i2c_id,
	.probe		= opt3001_i2c_probe,
	.remove		= opt3001_i2c_remove,
};

void opt3001_init(void)
{
	int error;
	error = i2c_add_driver(&opt3001_i2c_driver);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "ti_opt3001: i2c_add_driver failed with %d", error);
		return;
	}
}

void opt3001_exit(void)
{
	i2c_del_driver(&opt3001_i2c_driver);
}

int opt3001_handle_ioctl(enum sensors_als_ioctl_cmd cmd, void *result)
{
	struct i2c_client *client = get_i2c_client();

	if (client == NULL || result == NULL) {
		return -1;
	}

	if (cmd == SENSORS_ALS_IOCTL_GET_LUX_VAL) {
		int32_t *lux_value = result;
		*lux_value = opt3001_get_lux_val(client);
		return (*lux_value < 0) ? *lux_value : 0;
	} else if (cmd == SENSORS_ALS_IOCTL_GET_LIGHT_COND) {
		int ret;
		enum LIGHT_COND *light_cond = result;
		ret = opt3001_get_light_cond(client, light_cond);
		return ret;
	}
	return -EINVAL;
}
