/*
 * Copyright (c) 2018-2019 Sonos Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 *
 * Reads an ADC voltage and converts it to a temperature
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/thermal.h>
#include <linux/err.h>
#include <linux/sonos_kernel.h>
#include "blackbox.h"
#include "sensors_dev.h"
#include "sensors_hal.h"

struct thermistor_inst {
	char *name;
	int   adc_chan;
	u32   adc_conv;
	u32   slope;
	u32   zero_offset;
};
struct thermistor_inst thermistor_insts[] = {
	{.name = "POWER"}
};
#define THERMISTOR_NUM_INSTS (sizeof(thermistor_insts)/sizeof(struct thermistor_inst))

int thermistor_read_temp(int inst, int *temp)
{
	struct thermistor_inst *ti = &thermistor_insts[inst];
	int error, adc_val;
	error = read_adc_voltage(ti->adc_chan, &adc_val);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s: %s ADC read error %d", __func__, ti->name, error);
		*temp = 0;
	} else {
		int uvolts = adc_val * ti->adc_conv;

		*temp = (uvolts / ti->slope) + ti->zero_offset;
	}
	return error;
}

static struct device_node * thermistor_find_dtb_node(char *inst_name)
{
	struct device_node *np, *child;

	np = of_find_node_by_name(NULL, "thermistors");
	if (np == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "thermistors not in DTB");
		return NULL;
	}

	for_each_child_of_node(np, child) {
		if (strcmp(child->name, inst_name) == 0) {
			bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "found thermistor %s", inst_name);
			return child;
		}
	}
	bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "thermistor %s not found in DTB", inst_name);
	return NULL;
}

static int thermistor_get_dtb_data(struct thermistor_inst *ti)
{
	struct device_node *node;
	char *prop_name;
	int ret, swing, resolution;
	u32 val32;

	node = thermistor_find_dtb_node(ti->name);
	if (node == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "thermal inst %s not found in DTB", ti->name);
		return -ENOENT;
	}

	prop_name = "adc-channel";
	ret = of_property_read_u32_array(node, prop_name, &val32, 1);
	if (ret < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s missing from %s thermistor entry", prop_name, ti->name);
		return -ENOENT;
	}
	ti->adc_chan = val32;

	prop_name = "input-signal-swing";
	ret = of_property_read_u32_array(node, prop_name, &val32, 1);
	if (ret < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s missing from %s thermistor entry", prop_name, ti->name);
		return -ENOENT;
	}
	swing = val32;

	prop_name = "resolution-bits";
	ret = of_property_read_u32_array(node, prop_name, &val32, 1);
	if (ret < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s missing from %s thermistor entry", prop_name, ti->name);
		return -ENOENT;
	}
	resolution = 1 << val32;
	ti->adc_conv = swing / resolution;

	prop_name = "conversion-slope";
	ret = of_property_read_u32_array(node, prop_name, &val32, 1);
	if (ret < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s missing from %s thermistor entry", prop_name, ti->name);
		return -ENOENT;
	}
	ti->slope = val32;

	prop_name = "zero-offset";
	ret = of_property_read_u32_array(node, prop_name, &val32, 1);
	if (ret < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s missing from %s thermistor entry", prop_name, ti->name);
		return -ENOENT;
	}
	ti->zero_offset = val32;

	return 0;
}

void thermistor_init(void)
{
	int i, error;

	for (i = 0; i < THERMISTOR_NUM_INSTS; i++) {
		int temp;
		struct thermistor_inst *ti = &thermistor_insts[i];
		error = thermistor_get_dtb_data(ti);
		if (!error) {
			if (thermal_mgmt_map_device(i, ti->name, "sonos,thermistor", thermistor_read_temp)) {
			thermistor_read_temp(i, &temp);
			bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s: current temperature %d milli-Celsius", ti->name, temp);
			}
		}
	}
}

void thermistor_exit(void)
{
}
