/*
 * Copyright (c) 2016-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */


#include <linux/string.h>
#include <linux/workqueue.h>
#include <linux/jiffies.h>
#include <linux/of.h>
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#endif
#include <asm/uaccess.h>
#include <linux/mutex.h>
#include <linux/version.h>
#include "blackbox.h"
#include "sensors_hal.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "sensors_dev.h"
#include "mdp.h"
#include "sonos_lock.h"

#include "thermal_mgmt.h"

#define THERMAL_PROC_FILENAME "driver/temp-sensor"
#define THERMAL_SENSOR_DEBUG_INACTIVE 0
#define THERMAL_SENSOR_NAME_SIZE 32
struct thermal_sensor_data {
	struct thermal_mgmt_inst cfg;
	struct mutex therm_lock;
	int device_inst;
	int current_temp;
	int report_temp;
	int prev_temp;
	int debug_temp;
	int min_temp;
	int max_temp;
	unsigned int in_warn_count;
	unsigned int enter_warn_count;
	unsigned int backoff_warn_count;
	unsigned int in_fault_count;
	unsigned int enter_fault_count;
};

struct thermal_sensor_data thermal_sensors[THERMAL_SENSOR_NUM];
int thermal_status;

#define THERM_POLL_TIME		msecs_to_jiffies(5000)
unsigned long thermal_poll_time;
static struct delayed_work thermal_work;
static int hw_verification = -1;

struct thermal_mgmt_inst thermal_mgmt_insts[] = {
{
        .name = "CPU",
        .index = THERMAL_SENSOR_CPU,
        .source = HWEVTQSOURCE_CPU,
        .fault_mask = SYSTAT_CPU_FAULT_TEMPERATURE_STATUS,
        .warn_mask = SYSTAT_CPU_WARN_TEMPERATURE_STATUS,
},
{
        .name = "AMP",
        .index = THERMAL_SENSOR_AMP,
        .source = HWEVTQSOURCE_AMP,
        .fault_mask = SYSTAT_AUDIO_FAULT_TEMPERATURE_STATUS,
        .warn_mask = SYSTAT_AUDIO_WARN_TEMPERATURE_STATUS,
},
{
        .name = "SOC",
        .index = THERMAL_SENSOR_SOC,
        .source = HWEVTQSOURCE_SOC,
        .fault_mask = SYSTAT_SOC_FAULT_TEMPERATURE_STATUS,
        .warn_mask = SYSTAT_SOC_WARN_TEMPERATURE_STATUS,
},
{
        .name = "POWER",
        .index = THERMAL_SENSOR_POWER,
        .source = HWEVTQSOURCE_POWER,
        .fault_mask = SYSTAT_POWER_FAULT_TEMPERATURE_STATUS,
        .warn_mask = SYSTAT_POWER_WARN_TEMPERATURE_STATUS,
},
{
        .name = "MOTION",
        .index = THERMAL_SENSOR_MOTION,
        .source = HWEVTQSOURCE_MOTION_DETECTOR,
        .fault_mask = SYSTAT_MOTION_FAULT_TEMPERATURE_STATUS,
        .warn_mask = SYSTAT_MOTION_WARN_TEMPERATURE_STATUS,
}
};
#define MAX_THERMAL_SENSORS (sizeof(thermal_mgmt_insts) / sizeof(struct thermal_mgmt_inst))

extern int thermal_mgmt_get_battery_temp(void);
extern struct thermal_sensor_data *thermal_mgmt_find_device(char *inst_name);

int
thermal_mgmt_get_temp(int sensor)
{

	struct thermal_sensor_data *tsd = &thermal_sensors[sensor];
	struct thermal_mgmt_inst *cfg = &tsd->cfg;
	int ret;

	mutex_lock(&tsd->therm_lock);
	if ((tsd->device_inst < 0) || (cfg->name == NULL) || (tsd->current_temp <= SEN_RANGE)) {
		ret = 0;
	} else if (tsd->debug_temp != THERMAL_SENSOR_DEBUG_INACTIVE) {
		ret = tsd->debug_temp;
	} else {
		ret = tsd->current_temp;
	}
	mutex_unlock(&tsd->therm_lock);
	return( ret );
}
EXPORT_SYMBOL(thermal_mgmt_get_temp);

struct thermal_sensor_data *
thermal_mgmt_find_device(char *inst_name)
{
	int i;
	for (i = 0; i < THERMAL_SENSOR_NUM; i++) {
		struct thermal_sensor_data *tsd = &thermal_sensors[i];
		if (tsd->cfg.name == NULL) {
			continue;
		}
		if (strncmp(tsd->cfg.name, inst_name, THERMAL_SENSOR_NAME_SIZE) == 0) {
			return tsd;
		}
	}
	bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Temperature sensor %s does not exist", inst_name);
	return NULL;
}

static void thermal_mgmt_get_intval(struct device_node *node, char *prop_name, int *val, char *inst_name)
{
	u32 prop;
	int ret;
	ret = of_property_read_u32_array(node, prop_name, &prop, 1);
	if (ret < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s missing from %s thermal mgmt entry", prop_name, inst_name);
	} else {
		*val = prop;
	}
}

static struct device_node * thermal_mgmt_find_dtb_node(char *inst_name)
{
	struct device_node *np, *child;

	np = of_find_node_by_name(NULL, "thermal-mgmt");
	if (np == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "thermal mgmt not in DTB");
		return NULL;
	}

	for_each_child_of_node(np, child) {
		if (strcmp(child->name, inst_name) == 0) {
			bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "found thermal inst %s", inst_name);
			return child;
		}
	}
	bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "thermal mgmt inst %s not found in DTB", inst_name);
	return NULL;
}

int thermal_mgmt_map_device(int inst, char *inst_name, char *compatible, int (*thermal_read)(int, int *))
{
	int i;
	int ret = 0;
	struct thermal_mgmt_inst *ti;
	struct thermal_sensor_data *tsd;
	struct device_node *node;
	char *comp_string, comp_buf[64];

	node = thermal_mgmt_find_dtb_node(inst_name);
	if (node == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "thermal inst %s not found in DTB", inst_name);
		goto map_done;
	}

	comp_string = &comp_buf[0];
	if (of_property_read_string(node, "compatible", (const char **)&comp_string) < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "thermal inst %s compatible string not found in DTB", inst_name);
		goto map_done;
	}

	if (strncmp(comp_string, compatible, strlen(compatible))) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s from the dtb does not match %s", comp_string, compatible);
		goto map_done;
	}

	for (i = 0; i < MAX_THERMAL_SENSORS; i++) {
		ti = &thermal_mgmt_insts[i];
		if (strncmp(ti->name, inst_name, THERMAL_SENSOR_NAME_SIZE) == 0) {
			break;
		}
	}
	if (i >= MAX_THERMAL_SENSORS) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "thermal inst %s unknown", inst_name);
		goto map_done;
	}

	tsd = &thermal_sensors[ti->index];
	mutex_lock(&tsd->therm_lock);
	if ( tsd->device_inst == -1 )  {
		tsd->device_inst = inst;
		tsd->cfg = *ti;
		tsd->debug_temp = THERMAL_SENSOR_DEBUG_INACTIVE;
		tsd->max_temp = -273000;
		tsd->min_temp =  273000;
		tsd->cfg.thermal_read = thermal_read;
		thermal_mgmt_get_intval(node, "fault-temperature", &tsd->cfg.fault_temperature, inst_name);
		thermal_mgmt_get_intval(node, "warn-temperature", &tsd->cfg.warn_temperature, inst_name);
		ret = 1;
	} else {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Tried to overwrite thermal inst %s", inst_name);
	}
	mutex_unlock(&tsd->therm_lock);

map_done:
	return ret;
}

void thermal_mgmt_set_debug_temp(char *name, int temp)
{
	struct thermal_sensor_data *tsd = thermal_mgmt_find_device(name);
	if (tsd != NULL) {
		mutex_lock(&tsd->therm_lock);
		tsd->debug_temp = temp;
		mutex_unlock(&tsd->therm_lock);
	}
}

void thermal_mgmt_set_warn_temp(char *name, int temp)
{
	struct thermal_sensor_data *tsd = thermal_mgmt_find_device(name);
	if (tsd != NULL) {
		mutex_lock(&tsd->therm_lock);
		tsd->cfg.warn_temperature = temp;
		mutex_unlock(&tsd->therm_lock);
	}
}

void thermal_mgmt_set_fault_temp(char *name, int temp)
{
	struct thermal_sensor_data *tsd = thermal_mgmt_find_device(name);
	if (tsd != NULL) {
		mutex_lock(&tsd->therm_lock);
		tsd->cfg.fault_temperature = temp;
		mutex_unlock(&tsd->therm_lock);
	}
}

static void thermal_mgmt_perform(struct thermal_sensor_data *tsd)
{
	struct thermal_mgmt_inst *cfg = &tsd->cfg;

	if (tsd->report_temp >= cfg->fault_temperature) {
		tsd->in_fault_count++;
		if (tsd->prev_temp < cfg->fault_temperature) {
			tsd->enter_fault_count++;
			thermal_status |= cfg->fault_mask;
			thermal_status &= ~cfg->warn_mask;
			event_queue_send_event(cfg->source, HWEVTQINFO_TEMP_FAULT);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(cfg->source, HWEVTQINFO_TEMP_FAULT);
#endif
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s temperature fault detected [%d >= %d]", cfg->name, tsd->report_temp, cfg->fault_temperature);
		}
	} else if (tsd->report_temp >= cfg->warn_temperature) {
		tsd->in_warn_count++;
		if (tsd->prev_temp < cfg->warn_temperature) {
			tsd->enter_warn_count++;
			thermal_status |= cfg->warn_mask;
			event_queue_send_event(cfg->source, HWEVTQINFO_TEMP_WARNING);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(cfg->source, HWEVTQINFO_TEMP_WARNING);
#endif
			bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "%s temperature warning detected", cfg->name);
		} else if (tsd->prev_temp >= cfg->fault_temperature) {
			tsd->backoff_warn_count++;
			thermal_status |= cfg->warn_mask;
			thermal_status &= ~cfg->fault_mask;
			event_queue_send_event(cfg->source, HWEVTQINFO_TEMP_WARNING);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(cfg->source, HWEVTQINFO_TEMP_WARNING);
#endif
			bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "%s temperature warning (back off from fault)", cfg->name);
		}
	} else {
		if (tsd->prev_temp >= cfg->fault_temperature) {
			thermal_status &= ~cfg->fault_mask;
			event_queue_send_event(cfg->source, HWEVTQINFO_TEMP_OK);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(cfg->source, HWEVTQINFO_TEMP_OK);
#endif
			bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s temperature fault cleared", cfg->name);
		} else if (tsd->prev_temp >= cfg->warn_temperature) {
			thermal_status &= ~cfg->warn_mask;
			event_queue_send_event(cfg->source, HWEVTQINFO_TEMP_OK);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(cfg->source, HWEVTQINFO_TEMP_OK);
#endif
			bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s temperature warning cleared", cfg->name);
		}
	}
}

static void thermal_mgmt_poll(struct work_struct *work)
{
	int i;
	int hwtemp;
	int dinst;
	int (*tempcb)(int, int *);

	if (hw_verification < 0) {
		if (is_mdp_authorized(MDP_AUTH_FLAG_HWV_DEFAULTS)) {
			hw_verification = 1;
		} else {
			hw_verification = 0;
		}
	}

	for (i = 0; i < THERMAL_SENSOR_NUM; i++) {
		int error = 0;
		struct thermal_sensor_data *tsd = &thermal_sensors[i];
		mutex_lock(&tsd->therm_lock);
		if (tsd->device_inst < 0) {
			tsd->current_temp = SEN_NODEV;
			tsd->report_temp = SEN_NODEV;
			mutex_unlock(&tsd->therm_lock);
			continue;
		}
		tsd->prev_temp = tsd->report_temp;
		dinst = tsd->device_inst;
		tempcb = tsd->cfg.thermal_read;
		mutex_unlock(&tsd->therm_lock);

		if (hw_verification == 1) {
			hwtemp = 0;
		}
		error = (tempcb)(dinst, &hwtemp);

		mutex_lock(&tsd->therm_lock);
		if (error) {
			tsd->current_temp = SEN_ERROR;
			mutex_unlock(&tsd->therm_lock);
			continue;
		}

		tsd->current_temp = hwtemp;
		tsd->report_temp = (tsd->debug_temp == THERMAL_SENSOR_DEBUG_INACTIVE) ? tsd->current_temp : tsd->debug_temp;
		if (tsd->max_temp < tsd->report_temp) {
			tsd->max_temp = tsd->report_temp;
		}
		if (tsd->min_temp > tsd->report_temp) {
			tsd->min_temp = tsd->report_temp;
		}
		thermal_mgmt_perform(tsd);
		mutex_unlock(&tsd->therm_lock);
	}

	schedule_delayed_work(&thermal_work, thermal_poll_time);
}

int
thermal_mgmt_show(struct seq_file *m, void *v)
{
	int i;

	for (i = 0; i < THERMAL_SENSOR_NUM; i++) {
		struct thermal_sensor_data *tsd = &thermal_sensors[i];
		struct thermal_mgmt_inst *cfg = &tsd->cfg;

		mutex_lock(&tsd->therm_lock);
		if ((tsd->device_inst < 0) || (cfg->name == NULL)) {
			mutex_unlock(&tsd->therm_lock);
			continue;
		}
		if (tsd->current_temp <= SEN_RANGE) {
			seq_printf(m, "%s Temperature sensor not present\n", cfg->name);
			mutex_unlock(&tsd->therm_lock);
			continue;
		}
		seq_printf(m, "%7d%sCelsius: %s Temperature sensor (", tsd->current_temp,
                   (tsd->current_temp > 1000 ) ? " milli" : " ", cfg->name);
		if (tsd->debug_temp != THERMAL_SENSOR_DEBUG_INACTIVE) {
			seq_printf(m, "debug %d, ", tsd->debug_temp);
		}
		seq_printf(m, "fault %d, warn %d)\n", cfg->fault_temperature, cfg->warn_temperature);
		seq_printf(m, "%9u warning polls\n", tsd->in_warn_count);
		seq_printf(m, "%9u warning enters\n", tsd->enter_warn_count);
		seq_printf(m, "%9u fault->warning backoffs\n", tsd->backoff_warn_count);
		seq_printf(m, "%9u fault polls\n", tsd->in_fault_count);
		seq_printf(m, "%9u fault enters\n", tsd->enter_fault_count);
		seq_printf(m, "%9d min temperature\n", tsd->min_temp);
		seq_printf(m, "%9d max temperature\n", tsd->max_temp);
		seq_printf(m, "\n");
		mutex_unlock(&tsd->therm_lock);
	}
	return 0;
}
const static struct seq_operations thermalsensor_op = {
	.show =		 thermal_mgmt_show
};

static int
thermal_mgmt_open(struct inode *inode, struct file *file)
{
	return single_open(file, thermal_mgmt_show, PDE_DATA(inode));
}

static ssize_t
thermal_mgmt_proc_write(struct file *file, const char __user * buffer,
						size_t count, loff_t *data)
{
	int temp;
	char buf[40];
	char *peq;
	int error;

	if (count >= sizeof(buf))
		return -EIO;
	if (copy_from_user(buf, buffer, count))
		return -EFAULT;
	buf[count] = 0;

	peq = strchr(buf, '=');
	if (peq != NULL) {
		char *tempstring="temp-";
		char *faultstring="fault-";
		char *warnstring="warn-";
		*peq = 0;
		error = kstrtoint(peq+1, 10, &temp);
		if (error) {
			bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "%s: kstrtoint failed", __func__);
			return count;
		}
		if (strncmp(buf, tempstring, strlen(tempstring)) == 0) {
			thermal_mgmt_set_debug_temp(buf+strlen(tempstring), temp);
		} else if (strncmp(buf, faultstring, strlen(faultstring)) == 0) {
			thermal_mgmt_set_fault_temp(buf+strlen(faultstring), temp);
		} else if (strncmp(buf, warnstring, strlen(warnstring)) == 0) {
			thermal_mgmt_set_warn_temp(buf+strlen(warnstring), temp);
		}
	}
	return count;
}

static struct file_operations thermal_mgmt_proc_operations = {
	.owner		 = THIS_MODULE,
	.open		 = thermal_mgmt_open,
	.write		 = thermal_mgmt_proc_write,
	.read		 = seq_read,
	.llseek		 = seq_lseek,
	.release	 = single_release,
};

static int
thermal_mgmt_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create(THERMAL_PROC_FILENAME, 0666, NULL, &thermal_mgmt_proc_operations);
	if (entry == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "procfile creation failed");
		return -EIO;
	}
	return 0;
}

void thermal_mgmt_init(void)
{
	struct device_node *np;
	int i, ret;
	u32 val;
	for (i = 0; i < THERMAL_SENSOR_NUM; i++) {
		thermal_sensors[i].device_inst = -1;
		mutex_init(&thermal_sensors[i].therm_lock);
	}
	thermal_status = 0;
	thermal_mgmt_proc_init();
	INIT_DELAYED_WORK(&thermal_work, thermal_mgmt_poll);

	thermal_poll_time = THERM_POLL_TIME;
	np = of_find_node_by_name(NULL, "thermal-mgmt-poll");
	if (np != NULL) {
		ret = of_property_read_u32_array(np, "msec", &val, 1);
		if (ret >= 0) {
			bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "thermal poll time %umsec", val);
			thermal_poll_time = msecs_to_jiffies(val);
		}
	}

	schedule_delayed_work(&thermal_work, thermal_poll_time);
}

void thermal_mgmt_exit(void)
{
	cancel_delayed_work_sync(&thermal_work);
	remove_proc_entry(THERMAL_PROC_FILENAME, NULL);
	memset(thermal_sensors, 0, sizeof(thermal_sensors));
}

void thermal_mgmt_read(int *temps, int *status)
{
	int i, *temp=temps;
	for (i = 0; i < THERMAL_SENSOR_NUM; i++) {
		struct thermal_sensor_data *tsd = &thermal_sensors[i];
		mutex_lock(&tsd->therm_lock);
		*temp++ = tsd->report_temp;
		mutex_unlock(&tsd->therm_lock);
	}
	*status = thermal_status;
}
