/*
 * Copyright (c) 2016-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */

#include <generated/autoconf.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/of.h>
#include <linux/fs.h>
#include <linux/version.h>
#include <asm/uaccess.h>
#include <linux/cdev.h>
#include "blackbox.h"
#include "button_inst.h"
#include "sensors_hal.h"
#include "sensors_dev.h"
#include "hwevent_queue_user_api.h"
#include "gpio_buttons.h"
#include "sonos_device.h"

static int devno = -1;

static int device_init_success = 0;
int device_readers = 0;

struct file_operations sensors_fops;

#if defined(CONFIG_SONOS_BUTTONS)
struct button_inst buttons_table[BUTTONS_NUM_SUPPORTED];
#endif

#if defined(CONFIG_SONOS_LIS2HH12)
extern int lis2hh12_device_ready;
#endif
#if defined(CONFIG_SONOS_BMA420)
extern int bma420_device_ready;
#endif


struct sensors_func_entry {
	void (*init)(void);
	void (*exit)(void);
#if defined(CONFIG_SONOS_LIS2HH12) || defined(CONFIG_SONOS_BMA420)
	int (*handle_ioctl)(motion_device_t *);
#endif
};

struct sensors_func_entry sensors_func_table[] = {
#if defined(CONFIG_SONOS_THERMAL_MGMT)
	{thermal_mgmt_init, thermal_mgmt_exit},
#endif

#if defined(CONFIG_SONOS_S5851)
	{s5851_init, s5851_exit},
#endif

#if defined(CONFIG_SENSORS_TMP102)
	{tmp102shim_init, tmp102shim_exit},
#endif

#if defined(CONFIG_SONOS_MTK_ADC_THERMISTOR)
	{thermistor_init, thermistor_exit},
#endif

#if defined(CONFIG_SONOS_MMA8453Q)
	{mma8453q_init, mma8453q_exit},
#endif

#if defined(CONFIG_SONOS_LIS2HH12)
	{lis2hh12_init, lis2hh12_exit, lis2hh12_handle_ioctl},
#endif

#if defined(CONFIG_SONOS_BMA420)
	{bma420_init, bma420_exit, bma420_handle_ioctl},
#endif

#if defined(CONFIG_SONOS_FXL6408)
	{fxl6408_init, fxl6408_exit},
#endif

#if defined(CONFIG_SONOS_ORIENTATION)
	{orient_init, orient_exit},
#endif

#if defined(CONFIG_SONOS_WALL_MOUNT)
	{wall_mount_init, wall_mount_exit},
#endif

#if defined(CONFIG_SONOS_LINEIN_OVERVOLT)
	{linein_overvolt_init, linein_overvolt_exit},
#endif

#if defined(CONFIG_SONOS_CABLE_GPIO_DETECT)
	{cable_detect_init, cable_detect_exit},
#endif

#if defined(CONFIG_SONOS_BUTTONS)
	{button_init, button_exit},
	{buttons_sim_init, buttons_sim_exit},
#endif

#if defined(CONFIG_MTK_SOC_THERMAL)
	{mt8521_soc_temp_init, mt8521_soc_temp_exit},
#endif

#if defined(CONFIG_SONOS_IMX7ULP_CPU_THERMAL)
	{imx7ulp_cpu_temp_init, imx7ulp_cpu_temp_exit},
#endif

#if defined(CONFIG_SONOS_RPMSG_THERMAL)
	{sensors_rpmsg_init, sensors_rpmsg_exit},

	{rpmsg_temp_init, rpmsg_temp_exit},
#endif

#if defined(CONFIG_SONOS_TIOPT3001) && defined(SONOS_ARCH_ATTR_SUPPORTS_AMBIENT_LGT_SENSE)
	{opt3001_init, opt3001_exit},
#endif

#if defined(CONFIG_SONOS_LTR311) && defined(SONOS_ARCH_ATTR_SUPPORTS_AMBIENT_LGT_SENSE)
	{ltr311_init, ltr311_exit},
#endif

#if defined(CONFIG_SONOS_NXP3H2111_NFC)
	{nxp_nfc_init, nxp_nfc_exit},
#endif

#if defined(CONFIG_SONOS_SYMBOLIC_GPIO)
	{gpio_outs_init, gpio_outs_exit},
#endif

#if defined(CONFIG_SONOS_CCG3_MODULE)
	{battery_temp_init, battery_temp_exit},
#endif

#if defined(CONFIG_SONOS_NCU15)
	{ncu15_init, ncu15_exit},
#endif
};
#define SENSOR_FUNC_TABLE_SIZE (sizeof(sensors_func_table) / sizeof(struct sensors_func_entry))

struct cdev sensors_chr_dev;

extern int thermal_mgmt_get_temp(int);

static void sensors_devices_exit(void)
{
	int idx = SENSOR_FUNC_TABLE_SIZE - 1;

	for (; idx >= 0; idx--) {
		sensors_func_table[idx].exit();
	}
}

static void __exit sensors_exit(void)
{
	sonos_device_destroy(devno);
	cdev_del(&sensors_chr_dev);
	unregister_chrdev_region(devno, 1);
	sensors_devices_exit();
}
module_exit(sensors_exit);

#if defined(CONFIG_SONOS_LIS2HH12) || defined(CONFIG_SONOS_BMA420)
static void imu_fail_test(void) {
	int need_imu = false;

	#if defined(CONFIG_SONOS_BMA420)
	if (of_find_compatible_node(NULL, NULL, "sonos,bma420")) {
		if (bma420_device_ready) {
			return ;
		}
		need_imu = true;
	}
	#endif

	#if defined(CONFIG_SONOS_LIS2HH12)
	if (of_find_compatible_node(NULL, NULL, "stm,lis2hh12")) {
		if (lis2hh12_device_ready) {
			return ;
		}
		need_imu = true;
	}
	#endif

	if (need_imu) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "No IMU was detected.");
	}
}
#endif


#if defined(SONOS_ARCH_ATTR_SUPPORTS_AMBIENT_LGT_SENSE)
static void check_als(void) {
	int n_als_devices = 0;
	#if defined(CONFIG_SONOS_TIOPT3001)
		if (of_find_compatible_node(NULL, NULL, "fsl,ti-opt3001")) {
			n_als_devices++;
		}
	#endif

	#if defined(CONFIG_SONOS_LTR311)
		if (of_find_compatible_node(NULL, NULL, "ltr311")) {
			n_als_devices++;
		}
	#endif

	if (n_als_devices == 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "No ALS device is present.");
	} else if (n_als_devices > 1) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "Multiple ALS devices are present.");
	}
}
#endif

static int __init sensors_init(void)
{
	int error, idx;
	struct device *class_dev = NULL;

	devno = MKDEV(SENSORS_MAJOR_NUMBER, 0);
	for (idx = 0; idx < SENSOR_FUNC_TABLE_SIZE; idx++) {
		sensors_func_table[idx].init();
	}

#if defined(CONFIG_SONOS_LIS2HH12) || defined(CONFIG_SONOS_BMA420)
	imu_fail_test();
#endif

#if defined(SONOS_ARCH_ATTR_SUPPORTS_AMBIENT_LGT_SENSE)
	check_als();
#endif

#ifdef CONFIG_DEVTMPFS
	error = alloc_chrdev_region(&devno, 0, 1, SENSORS_DEVICE_NAME);
#else
	error = register_chrdev_region(devno, 1, SENSORS_DEVICE_NAME);
#endif
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Error %d registering device %s.", error, SENSORS_DEVICE_NAME);
		sensors_devices_exit();
		return error;
	}
	cdev_init(&sensors_chr_dev, &sensors_fops);
	sensors_chr_dev.owner = THIS_MODULE;

	error = cdev_add(&sensors_chr_dev, devno, 1);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Couldn't add character device (%d).", error);
		sensors_devices_exit();
		unregister_chrdev_region(devno, 1);
		return error;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, SENSORS_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		printk(KERN_ERR "Error creating Sonos class.\n");
		cdev_del(&sensors_chr_dev);
		unregister_chrdev_region(devno, 1);
		error = PTR_ERR(class_dev);
		return error;
	}

	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "Sensors module registered at %d, version %d", MAJOR(devno), SENSORS_VERSION);
	device_init_success = 1;
	return 0;
}
module_init(sensors_init);

static int sensors_device_open(struct inode *inode, struct file *file)
{
	if (file->f_mode & FMODE_READ) {
		++device_readers;
	}
	return (device_init_success ? 0 : -ENODEV);
};

static int sensors_device_release(struct inode *inode, struct file *file)
{
	if (file->f_mode & FMODE_READ) {
		--device_readers;
	}
	return 0;
}

static void sensors_device_read_buttons(enum HWEVTQ_EventInfo *buttons)
{
#ifdef CONFIG_SONOS_BUTTONS
	int i = 0;
	struct button_inst *bi = buttons_table;

	for (i = 0; i < BUTTONS_NUM_SUPPORTED; i++) {
		buttons[i] = (bi->source == HWEVTQSOURCE_NO_SOURCE) ? HWEVTQINFO_RELEASED : bi->button_read(i);
		bi++;
	}
#endif
}

#ifdef CONFIG_SONOS_WALL_MOUNT
static enum HWEVTQ_EventInfo get_wall_orient(enum HWEVTQ_EventInfo current_orient, bool wall_mounted)
{
	switch (current_orient) {
		case HWEVTQINFO_ORIENT_HORIZONTAL:
			return (wall_mounted ? HWEVTQINFO_ORIENT_HORIZONTAL_WALL : current_orient);
		case HWEVTQINFO_ORIENT_VERTICAL:
			return (wall_mounted ? HWEVTQINFO_ORIENT_VERTICAL_WALL : current_orient);
		case HWEVTQINFO_ORIENT_VERTICAL_LEFT:
			return (wall_mounted ? HWEVTQINFO_ORIENT_VERTICAL_LEFT_WALL : current_orient);
		case HWEVTQINFO_ORIENT_VERTICAL_RIGHT:
			return (wall_mounted ? HWEVTQINFO_ORIENT_VERTICAL_RIGHT_WALL : current_orient);
		default:
			return current_orient;
	}
}
#endif

enum HWEVTQ_EventInfo sensors_get_orient(void)
{
	enum HWEVTQ_EventInfo orient_state = HWEVTQINFO_ORIENT_HORIZONTAL;
#ifdef CONFIG_SONOS_ORIENTATION
	orient_state = orient_get_current();
#else
	#if defined(CONFIG_SONOS_LIS2HH12)
		if (lis2hh12_device_ready) {
			orient_state = lis2hh12_get_current();
		}
	#endif
	#if defined(CONFIG_SONOS_BMA420)
		if (bma420_device_ready) {
			orient_state = bma420_get_current();
		}
	#endif
#endif
#ifdef CONFIG_SONOS_WALL_MOUNT
	orient_state = get_wall_orient(orient_state, wall_mount_get_current());
#endif
	return orient_state;
}

enum HWEVTQ_EventInfo sensors_get_cable_state(char *cable_name)
{
#ifdef CONFIG_SONOS_CABLE_GPIO_DETECT
	return cable_detect_get_info(cable_name);
#else
	return HWEVTQINFO_DISCONNECTED;
#endif
}

#ifdef CONFIG_SONOS_HDMICTL_MODULE
enum HWEVTQ_EventInfo sensors_songle_state = HWEVTQINFO_DISCONNECTED;
EXPORT_SYMBOL(sensors_songle_state);
#endif

enum HWEVTQ_EventInfo sensors_get_songle(void)
{
#ifdef CONFIG_SONOS_HDMICTL_MODULE
	return READ_ONCE(sensors_songle_state);
#else
	return HWEVTQINFO_DISCONNECTED;
#endif
}

#ifdef CONFIG_SONOS_HDMICTL_MODULE
enum HWEVTQ_EventInfo sensors_hdmi_fault_state = HWEVTQINFO_NO_EVENT;
EXPORT_SYMBOL(sensors_hdmi_fault_state);
#endif

enum HWEVTQ_EventInfo sensors_get_hdmi_fault_state(void)
{
#ifdef CONFIG_SONOS_HDMICTL_MODULE
	return READ_ONCE(sensors_hdmi_fault_state);
#else
	return HWEVTQINFO_NO_EVENT;
#endif
}

static ssize_t sensors_device_read(struct file *filp, char *data, size_t len, loff_t *ppos)
{
	int ret;
	struct SENSORS_READ sensors_read_data;

	if (len < sizeof(struct SENSORS_READ)) {
		return -EINVAL;
	}

	thermal_mgmt_read(sensors_read_data.thermal_sensors, &sensors_read_data.thermal_status);
	sensors_read_data.orientation = sensors_get_orient();
	sensors_device_read_buttons(sensors_read_data.buttons);
	sensors_read_data.line_in = sensors_get_cable_state(CABLE_DETECT_NAME_LINEIN);
	sensors_read_data.songle = sensors_get_songle();
	sensors_read_data.subwoofer_out = sensors_get_cable_state(CABLE_DETECT_NAME_SUBWOOFER_OUT);
	sensors_read_data.avtrigger = sensors_get_cable_state(CABLE_DETECT_NAME_AVTRIGGER);
	sensors_read_data.line_out = sensors_get_cable_state(CABLE_DETECT_NAME_LINEOUT);
	sensors_read_data.spdif_out = sensors_get_cable_state(CABLE_DETECT_NAME_SPDIF_OUT);
	sensors_read_data.hdmi = sensors_get_hdmi_fault_state();

	if (copy_to_user(data, (char *)&sensors_read_data, sizeof(sensors_read_data))) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: copy_to_user failed", __func__);
		ret = -EFAULT;
	} else {
		ret = sizeof(sensors_read_data);
	}
	return ret;
}

static long sensors_device_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	switch (_IOC_NR(cmd)) {
	case 0:
	{
		uint32_t ver = SENSORS_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	case 1:
	{
#ifdef CONFIG_SONOS_BUTTONS
		int repeat_time;
		if (copy_from_user(&repeat_time, (uint32_t *)arg, sizeof(uint32_t))) {
			return -EACCES;
		}
		button_set_repeat_time(repeat_time);
		break;
#else
		return -ENODEV;
#endif
	}
	case 2:
	{
#ifdef CONFIG_SONOS_BUTTONS
		uint32_t time = button_get_repeat_time();
		if (copy_to_user((uint32_t *)arg, &time, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
#else
		return -ENODEV;
#endif
	}
#if defined(CONFIG_SONOS_LIS2HH12)
	case 3:
	{
		motion_device_t	motion;
		int ret = 0;
		if (copy_from_user(&motion, (uint32_t *)arg, sizeof(motion_device_t))) {
			return -EACCES;
		}

		ret = lis2hh12_handle_ioctl(&motion);
		if (ret) {
			return ret;
		}

		if (copy_to_user((uint32_t *)arg, &motion, sizeof(motion_device_t))) {
			return -EACCES;
		}
		break;
	}
#endif
        case 4:
        {
                int temp;
                temp = thermal_mgmt_get_temp(THERMAL_SENSOR_POWER);
                if (copy_to_user((int *)arg, &temp, sizeof(temp))) {
                        return -EACCES;
                }
                break;
        }
#if defined(SONOS_ARCH_ATTR_SUPPORTS_AMBIENT_LGT_SENSE)
	case 5:
	{
		int res = -ENODEV;
		int32_t lux_val;

		#if defined(CONFIG_SONOS_TIOPT3001)
		if (opt3001_driver_ready()) {
			res = opt3001_handle_ioctl(SENSORS_ALS_IOCTL_GET_LUX_VAL, &lux_val);
			if (res) {
				return res;
			}
		} else
		#endif
		#if defined(CONFIG_SONOS_LTR311)
		if (ltr311_driver_ready()) {
			res = ltr311_handle_ioctl(SENSORS_ALS_IOCTL_GET_LUX_VAL, &lux_val);
			if (res) {
				return res;
			}
		} else
		#endif
		{
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "IOCTL: requested lux_val, but no ALS device exists/is initialized.");
			return res;
		}

		if (copy_to_user((int16_t *)arg, &lux_val, sizeof(lux_val))) {
			return -EACCES;
		}
		break;
	}
	case 6:
	{
		int res = -ENODEV;
		int light_cond;

		#if defined(CONFIG_SONOS_TIOPT3001)
		if (opt3001_driver_ready()) {
			res = opt3001_handle_ioctl(SENSORS_ALS_IOCTL_GET_LIGHT_COND, &light_cond);
			if (res) {
				return res;
			}
		} else
		#endif
		#if defined(CONFIG_SONOS_LTR311)
		if (ltr311_driver_ready()) {
			res = ltr311_handle_ioctl(SENSORS_ALS_IOCTL_GET_LIGHT_COND, &light_cond);
			if (res) {
				return res;
			}
		} else
		#endif
		{
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "IOCTL: requested lux_val, but no ALS device exists/is initialized.");
			return res;
		}

		if (copy_to_user((int *)arg, &light_cond, sizeof(light_cond))) {
			return -EACCES;
		}
		break;
	}
#endif
	default:
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Unrecognized IOCTL command %u\n", _IOC_NR(cmd));
		return -EINVAL;
	}
	return 0;
}

struct file_operations sensors_fops =
{
	.unlocked_ioctl	= sensors_device_ioctl,
	.open		= sensors_device_open,
	.release	= sensors_device_release,
	.read		= sensors_device_read,
};

MODULE_LICENSE ("GPL");
MODULE_AUTHOR ("Sonos, Inc.");
MODULE_DESCRIPTION ("Sensors Module");
