/*
 * Copyright (c) 2016-2019 Sonos Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */


#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/delay.h>
#include "blackbox.h"
#include "sensors_dev.h"

#define S5851_MAX_DEVICES 5

int s5851_next_device = 0;
struct i2c_client *s5851_devices[S5851_MAX_DEVICES] = {NULL};

static int __s5851_read_temp(struct i2c_client *i2c_client, int *temp)
{
	int error;
	short short_temp;

	error = i2c_smbus_read_word_swapped(i2c_client, 0);
	if (error < 0) {
		*temp = 0;
		return -EIO;
	}

	short_temp = error;
	short_temp /= 256;
	*temp = short_temp;
	return 0;
}

int s5851_read_temp(int inst, int *temp)
{
	if ((inst < 0) || (inst >= s5851_next_device) || s5851_devices[inst] == NULL) {
		*temp = 0;
		return -EPERM;
	}
	return __s5851_read_temp(s5851_devices[inst], temp);
}

static void
s5851_init_device(struct i2c_client *i2c_client, char *name)
{
	int error;
	u8 reg = 1;
	u8 value = 0;
	error = i2c_smbus_write_byte_data(i2c_client, reg, value);
	if (error) {
		bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "%s sensor init error %d", name, error);
	} else {
		int temp;
		error = __s5851_read_temp(i2c_client, &temp);
		if (error) {
			bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "%s sensor read error %d", name, error);
		} else {
			bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_INFO, "%s: current temperature %d Celsius", name, temp);
		}
	}
}

static int
s5851_i2c_remove(struct i2c_client *i2c_client)
{
	int i;

	for (i = 0; i < s5851_next_device; i++) {
		if (s5851_devices[i] == i2c_client) {
			s5851_devices[i] = NULL;
			break;
		}
	}
	return 0;
}

static int
s5851_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id)
{
	int len;
	char *inst_name;

	inst_name = (char *)of_get_property(i2c_client->dev.of_node, "device_type", &len);
	if (inst_name == NULL) {
		bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "%s: device %x ignored", __func__, i2c_client->addr);
		return 0;
	}

	if (s5851_next_device < S5851_MAX_DEVICES) {
		if (thermal_mgmt_map_device(s5851_next_device, inst_name, "seiko,s5851", s5851_read_temp)) {
			s5851_devices[s5851_next_device++] = i2c_client;
			s5851_init_device(i2c_client, inst_name);
		}
	} else {
		bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "%s: exceeded max devices (%d), addr %x", __func__, S5851_MAX_DEVICES, i2c_client->addr);
	}

	return 0;
}

static const struct i2c_device_id s5851_i2c_id[] = {
	{ "s5851cpu", 0 },
	{ }
};
MODULE_DEVICE_TABLE(s5851_i2c, s5851_i2c_id);

static struct of_device_id s5851_ids[] = {
	{ .compatible = "seiko,s5851" },
	{  }
};

static struct i2c_driver s5851_i2c_driver = {
	.driver = {
		.name   = "thermalsensor",
		.owner  = THIS_MODULE,
		.of_match_table = s5851_ids,
	},
	.id_table  = s5851_i2c_id,
	.probe     = s5851_i2c_probe,
	.remove    = s5851_i2c_remove,
};

void s5851_init(void)
{
#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	int ret = i2c_add_driver(&s5851_i2c_driver);
	if (ret) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "s5851 i2c init failed");
	}
#endif
}

void s5851_exit(void)
{
	i2c_del_driver(&s5851_i2c_driver);
}
