/*
 * Encore Orientation
 *
 * Copyright (c) 2014-2020, Sonos, Inc.
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/workqueue.h>
#include <asm/uaccess.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/timer.h>
#include "mdp.h"
#include <linux/sonos_kernel.h>
#include "hwevent_queue_user_api.h"
#include "sensors_hal.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "sensors_dev.h"
#include "blackbox.h"

#define ORIENT_POLL_TIME    msecs_to_jiffies(100)
static struct delayed_work orient_work;

#define ORIENT_NUM_SAMPLES  32
#define ORIENT_1G_10BIT     256
#define ORIENT_M            40
#define ORIENT_N            10
#define ORIENT_INVALID      HWEVTQINFO_NO_EVENT

typedef struct orient_filter_s {
	char *pname;
	int accumulator;
	int denom;
	int index;
	int samples[ORIENT_NUM_SAMPLES];
	int min;
	int max;
} orient_filter_t;

typedef struct orient_gravity_vector_s {
	unsigned int low_gsquared;
	unsigned int high_gsquared;
	unsigned int min_gsquared;
	unsigned int max_gsquared;
	unsigned int rejects;
	unsigned int samples;
}orient_gravity_vector_t;

typedef struct orient_data_s {
	int cur_orient;
	int override;
	int report;
	unsigned int m;
	unsigned int n;
	unsigned int invalids;
	unsigned int polls;
	unsigned int next_update;
	orient_gravity_vector_t gravity_vector;
	orient_filter_t xfilt;
	orient_filter_t yfilt;
	orient_filter_t zfilt;
}orient_data_t;
orient_data_t orient_data;

char * orient_get_string(int pos)
{
	char *orient_string;

	switch (pos)
	{
	case ORIENT_INVALID:
		orient_string = "detect/invalid";
		break;
	case HWEVTQINFO_ORIENT_HORIZONTAL:
		orient_string = "horizontal";
		break;
	case HWEVTQINFO_ORIENT_VERTICAL_LEFT:
		orient_string = "left";
		break;
	case HWEVTQINFO_ORIENT_VERTICAL_RIGHT:
		orient_string = "right";
		break;
	default:
		orient_string = "unknown";
		break;
	}

	return orient_string;
}

static int orient_filter_get(orient_filter_t *pf)
{
	int axis = 0;

	if (pf->denom > 0) {
		axis = pf->accumulator / pf->denom;
	}
	return axis;
}

static inline int orient_calc_n_term(int n, int y, int G)
{
	int val, roundup=0, rem;

	val = (y * 100) / G;
	val = 100 - val;
	val = n * val;
	rem = val % 100;
	if (rem >= 60)
		roundup = 1;
	if (rem <= -60)
		roundup = -1;
	val = val / 100;
	val += roundup;

	return val;
}

static inline unsigned int orient_absolute_value(int v)
{
	unsigned int uv = v;
	if (v < 0) {
		uv = -v;
	}
	return uv;
}

enum HWEVTQ_EventInfo orient_get_current(void)
{
	int x, y, z;
	unsigned int xav, yav, zav, nterm;
	uint8_t horiz_thresh, vert_thresh;

	if (orient_data.gravity_vector.samples < ORIENT_NUM_SAMPLES) {
		return HWEVTQINFO_ORIENT_HORIZONTAL;
	}

	x = orient_filter_get(&orient_data.xfilt);
	xav = orient_absolute_value(x);
	y = orient_filter_get(&orient_data.yfilt);
	yav = orient_absolute_value(y);
	z = orient_filter_get(&orient_data.zfilt);
	zav = orient_absolute_value(z);

	nterm = orient_calc_n_term(orient_data.n, yav, ORIENT_1G_10BIT);

	vert_thresh = (xav > (zav + nterm + orient_data.m));
	horiz_thresh = (zav > (xav + nterm + orient_data.m));

	switch (orient_data.cur_orient) {
	case HWEVTQINFO_ORIENT_HORIZONTAL:
		if (vert_thresh) {
			if (x > 0) {
				orient_data.cur_orient = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
			} else {
				orient_data.cur_orient = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
			}
		}
		break;
	case HWEVTQINFO_ORIENT_VERTICAL_LEFT:
		if (horiz_thresh) {
			orient_data.cur_orient = HWEVTQINFO_ORIENT_HORIZONTAL;
		} else if (vert_thresh && x <= 0) {
			orient_data.cur_orient = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
		}
		break;
	case HWEVTQINFO_ORIENT_VERTICAL_RIGHT:
		if (horiz_thresh) {
			orient_data.cur_orient = HWEVTQINFO_ORIENT_HORIZONTAL;
		} else if (vert_thresh && x > 0) {
			orient_data.cur_orient = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
		}
		break;
	default:
		orient_data.invalids++;
		break;
	}

	if (orient_data.override != ORIENT_INVALID) {
		return orient_data.override;
	} else {
		return orient_data.cur_orient;
	}
}

static int orient_gravity_vector_valid(struct accel_data *pvector)
{
	orient_gravity_vector_t *pgv = &orient_data.gravity_vector;
	unsigned int gv2;
	int valid=1;

	gv2 = (pvector->x * pvector->x) + (pvector->y * pvector->y) + (pvector->z * pvector->z);

	if (gv2 < pgv->min_gsquared) {
		pgv->min_gsquared = gv2;
	}
	if (gv2 > pgv->max_gsquared) {
		pgv->max_gsquared = gv2;
	}

	if ((gv2 < pgv->low_gsquared) || (gv2 > pgv->high_gsquared)) {
		valid = 0;
		pgv->rejects++;
	} else {
		pgv->samples++;
	}
	return valid;
}

static void orient_filter_sample(orient_filter_t *pf, int axis_value)
{
	if (axis_value < pf->min) {
		pf->min = axis_value;
	}
	if (axis_value > pf->max) {
		pf->max = axis_value;
	}

	pf->accumulator -= pf->samples[pf->index];
	pf->accumulator += axis_value;

	pf->samples[pf->index++] = axis_value;
	if (pf->index >= ORIENT_NUM_SAMPLES) {
		pf->index = 0;
	}
	if (pf->denom < ORIENT_NUM_SAMPLES) {
		pf->denom++;
	}
}

static void orient_poll(struct work_struct *work)
{
	struct accel_data ad;
	int orient = ORIENT_INVALID;

	mma8453q_get_raw_accel(&ad);
	if (orient_gravity_vector_valid(&ad)) {
		orient_filter_sample(&orient_data.xfilt, ad.x);
		orient_filter_sample(&orient_data.yfilt, ad.y);
		orient_filter_sample(&orient_data.zfilt, ad.z);
	}
	orient = orient_get_current();
	if (orient != orient_data.report && orient_data.polls >= orient_data.next_update) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "Orientation changed from %s to %s.\n",
		       orient_get_string(orient_data.report), orient_get_string(orient));
		event_queue_send_event(HWEVTQSOURCE_ORIENTATION, orient);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event(HWEVTQSOURCE_ORIENTATION, orient);
#endif
		orient_data.report = orient;
		orient_data.next_update = orient_data.polls + 100;

		sonos_orientation_change_event(orient);
	}
	orient_data.polls++;
	schedule_delayed_work(&orient_work, ORIENT_POLL_TIME);
}

static int
orient_proc_show(struct seq_file *m, void *v)
{
	char *actual = orient_get_string(orient_data.cur_orient);

	if (orient_data.override == ORIENT_INVALID) {
		seq_printf(m, "Current orientation:  %s\n", actual);
	} else {
		seq_printf(m, "Current orientation:  %s (Simulation mode, actual %s)\n",
				orient_get_string(orient_data.override), actual);
	}
	seq_printf(m, "%9u m\n", orient_data.m);
	seq_printf(m, "%9u n\n", orient_data.n);
	seq_printf(m, "%9u polls\n", orient_data.polls);
	seq_printf(m, "%9u invalid\n", orient_data.invalids);
	seq_printf(m, "filters x:y:z: %d:%d:%d\n",
		   orient_filter_get(&orient_data.xfilt),
		   orient_filter_get(&orient_data.yfilt),
		   orient_filter_get(&orient_data.zfilt));
	seq_printf(m, "%9u low-g2\n", orient_data.gravity_vector.low_gsquared);
	seq_printf(m, "%9u high-g2\n", orient_data.gravity_vector.high_gsquared);
	seq_printf(m, "%9u min-g2\n", orient_data.gravity_vector.min_gsquared);
	seq_printf(m, "%9u max-g2\n", orient_data.gravity_vector.max_gsquared);
	seq_printf(m, "%9u rejects\n", orient_data.gravity_vector.rejects);
	seq_printf(m, "%9u samples\n", orient_data.gravity_vector.samples);

	return 0;
}

static int
orient_proc_write(struct file *file, const char __user * buffer,
                  size_t count, loff_t *data)
{
	char buf[200];
	char *keyword;
	int len;
	int temp;

	if (count >= sizeof(buf))
		return -EIO;
	else if (copy_from_user(buf, buffer, count))
		return -EFAULT;

	buf[count] = '\0';

	keyword = "orientation-reporting=";
	len = strlen(keyword);
	if (strncmp(buf, keyword, len) == 0) {
		char *detect = "detect";
		char *horizontal  = "horizontal";
		char *left = "left";
		char *right = "right";
		if (strncmp(&buf[len], detect, strlen(detect)) == 0) {
			orient_data.override = ORIENT_INVALID;
		} else if (strncmp(&buf[len], horizontal, strlen(horizontal)) == 0) {
			orient_data.override = HWEVTQINFO_ORIENT_HORIZONTAL;
		} else if (strncmp(&buf[len], left, strlen(left)) == 0) {
			orient_data.override = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
		} else if (strncmp(&buf[len], right, strlen(right)) == 0) {
			orient_data.override = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
		} else {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "illegal orientation override %s\n", &buf[len]);
		}
	}
	keyword = "m=";
	len = strlen(keyword);
	if (strncmp(buf, keyword, len) == 0) {
		if (kstrtoint(buf + len, 10, &temp)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "orient: kstrtoint failed!\n");
			return -EIO;
		}
		orient_data.m = temp;
	}
	keyword = "n=";
	len = strlen(keyword);
	if (strncmp(buf, keyword, len) == 0) {
		if (kstrtoint(buf + len, 10, &temp)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "orient: kstrtoint failed!\n");
			return -EIO;
		}
		orient_data.n = temp;
	}
	keyword = "low-g2=";
	len = strlen(keyword);
	if (strncmp(buf, keyword, len) == 0) {
		if (kstrtoint(buf + len, 10, &temp)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "orient: kstrtoint failed!\n");
			return -EIO;
		}
		orient_data.gravity_vector.low_gsquared = temp;
	}
	keyword = "high-g2=";
	len = strlen(keyword);
	if (strncmp(buf, keyword, len) == 0) {
		if (kstrtoint(buf + len, 10, &temp)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "orient: kstrtoint failed!\n");
			return -EIO;
		}
		orient_data.gravity_vector.high_gsquared = temp;
	}
	return count;
}

static int orient_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, orient_proc_show, PDE_DATA(inode));
}

static const struct file_operations orient_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= orient_proc_open,
	.write		= orient_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define ORIENT_PROCFS_FILE "driver/orientation"

static int orient_proc_init(void)
{
	struct proc_dir_entry *proc;

	proc = proc_create(ORIENT_PROCFS_FILE, 0666, NULL, &orient_proc_fops);
	if (proc == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "driver/orientation file not created\n");
		return -EIO;
	}

	return 0;
}

static void orient_proc_remove(void)
{
	remove_proc_entry(ORIENT_PROCFS_FILE, NULL);
}

static inline void orient_filter_init(orient_filter_t *pf, char *pname)
{
	memset(pf, 0, sizeof(*pf));
	pf->pname = pname;
	pf->min = ORIENT_1G_10BIT + ORIENT_1G_10BIT - 1;
	pf->max = -(ORIENT_1G_10BIT + ORIENT_1G_10BIT);
}

void orient_init(void)
{
	orient_gravity_vector_t *pgv = &orient_data.gravity_vector;
	unsigned int gsquared = ORIENT_1G_10BIT * ORIENT_1G_10BIT;

	pgv->low_gsquared  = (gsquared *  75) / 100;
	pgv->high_gsquared = (gsquared * 125) / 100;
	pgv->max_gsquared = 0;
	pgv->min_gsquared = 0xffffffff;

	orient_data.cur_orient = HWEVTQINFO_ORIENT_HORIZONTAL;
	orient_data.report = HWEVTQINFO_ORIENT_HORIZONTAL;
	orient_data.override = ORIENT_INVALID;
	orient_data.m = ORIENT_M;
	orient_data.n = ORIENT_N;

	orient_filter_init(&orient_data.xfilt, "Xpos");
	orient_filter_init(&orient_data.yfilt, "Ypos");
	orient_filter_init(&orient_data.zfilt, "Zpos");

	INIT_DELAYED_WORK(&orient_work, orient_poll);
	schedule_delayed_work(&orient_work, ORIENT_POLL_TIME);

	orient_proc_init();

	event_queue_send_event(HWEVTQSOURCE_ORIENTATION, orient_data.cur_orient);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	hwevtq_send_event(HWEVTQSOURCE_ORIENTATION, orient_data.cur_orient);
#endif
}

void orient_exit(void)
{
	cancel_delayed_work_sync(&orient_work);
	orient_proc_remove();
}
