/*
 * Copyright (c) 2019 - 2020 Sonos Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/sysfs.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/interrupt.h>
#include <linux/gpio.h>
#include <linux/of_platform.h>
#include <linux/of_irq.h>
#include <linux/of_gpio.h>
#include <linux/cdev.h>
#include <linux/uaccess.h>
#include <linux/fs.h>
#include "blackbox.h"
#include "sonos_device.h"
#include "sensors_dev.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "mdp.h"

#define NXP_NFC_DEFAULT_I2C_ADDRESS		(0xAA)
#define NXP_NFC_I2C_BLOCK_LEN			(16)
#define NXP_NFC_CHR_BUF_LEN			(1008)
#define NXP_NFC_MAX_FILE_IDX			((NXP_NFC_USERMEM_END_BLK) * NXP_NFC_I2C_BLOCK_LEN)
#define MINIMUM_I2C_DELAY_MS			(4)
#define WRBLK_CMD_STRLEN_MIN			(25)
#define WRBLK_CMD_STRLEN_MAX			(90)

#define NXP_NFC_DEVICE_NAME			"nxp_nfc"
#define NXP_DEVICE_NODE				"/dev/"NXP_NFC_DEVICE_NAME

#define NXP_NFC_CC_BLK				(0x00)
#define NXP_NFC_USERMEM_START_BLK		(0x01)
#define NXP_NFC_USERMEM_END_BLK			(0x38)
#define NXP_NFC_SRAM_START_BLK			(0xf8)
#define NXP_NFC_SRAM_END_BLK			(0xfb)
#define NXP_NFC_CONFIG_REG_BLK			(0x3a)
#define NXP_NFC_SESSION_REG_BLK			(0xfe)

#define CONFIG_REG_NCREG			(0x00)
#define CONFIG_REG_LAST_NDEF_BLK		(0x01)
#define CONFIG_REG_SRAM_MIRROR_BLK		(0x02)
#define CONFIG_REG_WDT_LS			(0x03)
#define CONFIG_REG_WDT_MS			(0x04)
#define CONFIG_REG_I2C_CLK_STR			(0x05)
#define CONFIG_REG_REG_LOCK			(0x06)
#define CONFIG_REG_RFU				(0x07)

#define SESSION_REG_NCREG			(0x00)
#define SESSION_REG_LAST_NDEF_BLK		(0x01)
#define SESSION_REG_SRAM_MIRROR_BLK		(0x02)
#define SESSION_REG_WDT_LS			(0x03)
#define SESSION_REG_WDT_MS			(0x04)
#define SESSION_REG_I2C_CLK_STR			(0x05)
#define SESSION_REG_NS_REG			(0x06)
#define SESSION_REG_RFU				(0x07)
#define SESSION_NUM_REGS			(0x07)

#define NCREG_DEFAULT				(0x29)

static struct nxp_nfc_dev_t {
	struct i2c_client *client;
	uint64_t serial_num;
	uint32_t cc;
	uint16_t slock;
	int32_t device_ready;
	bool allow_write;
} nfc_data;

static int32_t driver_loaded = false;
static dev_t devno = -1;
static struct cdev nfc_chr_dev;

static int nxp_nfc_i2c_addr;

#define ENABLE_VOUT_DATA_COLLECTION
#if defined(ENABLE_VOUT_DATA_COLLECTION)
#include <linux/sonos_kernel.h>
#define VOUT_COLLECTION_INTERVAL	50
#define VOUT_NOISE_FILTER		40
#define VOUT_SAMPLE_RATE		4
struct delayed_work nxp_nfc_vout_work;
static int vout_adc_port;
static int nfc_nxp_vout_scan_count = 0;
#endif
static struct gpio field_detect = {.gpio = -EINVAL, .flags = IRQF_TRIGGER_FALLING, .label = "NFC field detect"} ;

static void nxp_nfc_i2c_address_is_wrong(struct i2c_client *client)
{

	if (client->addr == (NXP_NFC_DEFAULT_I2C_ADDRESS >> 1)) {
		client->addr = nxp_nfc_i2c_addr >> 1;
	} else {
		client->addr = NXP_NFC_DEFAULT_I2C_ADDRESS >> 1;
	}
	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "NFC: uses-alternate-address, i2c access failed - changing i2c_client address to 0x%x (%x/%x)", \
			client->addr, (client->addr << 1), ((client->addr << 1) | 1) );
}

static int32_t nxp_nfc_i2c_write_block(struct i2c_client *client, uint8_t block_id, uint8_t *buf)
{
	int32_t error;
	int retry_count = 0;

retry_i2c_write:
	mdelay(MINIMUM_I2C_DELAY_MS);
	error = i2c_smbus_write_i2c_block_data(client, block_id, NXP_NFC_I2C_BLOCK_LEN, buf);
	if (error < 0) {
		if (error == -ENXIO  && (nxp_nfc_i2c_addr != NXP_NFC_DEFAULT_I2C_ADDRESS)) {
			nxp_nfc_i2c_address_is_wrong(client);
			if (retry_count++ < 3) {
				goto retry_i2c_write;
			}
		}
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: %s failed %d blk: %x", __func__, error, block_id);
	}
	return error;
}

static int32_t nxp_nfc_i2c_read_block(struct i2c_client *client, uint8_t block_id, uint8_t *rsp_buf)
{
	int32_t error;
	int retry_count = 0;

retry_i2c_read:
	mdelay(MINIMUM_I2C_DELAY_MS);
	error = i2c_smbus_read_i2c_block_data(client, block_id, NXP_NFC_I2C_BLOCK_LEN, rsp_buf);
	if (error < 0) {
		if (error == -ENXIO  && (nxp_nfc_i2c_addr != NXP_NFC_DEFAULT_I2C_ADDRESS)) {
			nxp_nfc_i2c_address_is_wrong(client);
			if (retry_count++ < 3) {
				goto retry_i2c_read;
			}
		}
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: %s failed %d blk: %x", __func__, error, block_id);
	}
	return error;
}

static int32_t nxp_nfc_i2c_read_session_reg(struct i2c_client *client, uint8_t reg, uint8_t *val)
{
	int32_t error;
	uint8_t wr_msg[2] = {NXP_NFC_SESSION_REG_BLK, reg};
	uint8_t rd_msg[1];
	int retry_count = 0;

	struct i2c_msg msg[2] = {
		{
			.addr = client->addr,
			.len = 2,
			.buf = wr_msg,
		},
		{
			.addr = client->addr,
			.flags = I2C_M_RD,
			.len = 1,
			.buf = rd_msg,

		}
	};

retry_i2c_transfer:
	mdelay(MINIMUM_I2C_DELAY_MS);
	error = i2c_transfer(client->adapter, msg, 2);
	if (error < 0) {
		if (error == -ENXIO  && (nxp_nfc_i2c_addr != NXP_NFC_DEFAULT_I2C_ADDRESS)) {
			nxp_nfc_i2c_address_is_wrong(client);
			if (retry_count++ < 3) {
				goto retry_i2c_transfer;
			}
		}
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: %s i2c_transfer. reg: 0x%02x err: %d", __func__, reg, error);
	} else {
		*val = (uint8_t) rd_msg[0];
	}
	return error;
}

static int32_t nxp_nfc_i2c_write_session_reg(struct i2c_client *client, uint8_t reg, uint8_t mask, uint8_t val)
{
	int32_t error;
	uint8_t buf[4] = {NXP_NFC_SESSION_REG_BLK, 0, 0, 0};
	int retry_count = 0;

	buf[1] = reg;
	buf[2] = mask;
	buf[3] = val;

retry_i2c_send:
	mdelay(MINIMUM_I2C_DELAY_MS);
	error = i2c_master_send(client, buf, 4);
	if (error < 0) {
		if (error == -ENXIO  && (nxp_nfc_i2c_addr != NXP_NFC_DEFAULT_I2C_ADDRESS)) {
			nxp_nfc_i2c_address_is_wrong(client);
			if (retry_count++ < 3) {
				goto retry_i2c_send;
			}
		}
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: %s i2c_master_send failed. err: %d", __func__, error);
	}
	return error;
}

static int32_t write_last_ndef_blk(struct i2c_client* client, uint8_t last_blk)
{
	int32_t ret;
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN];
	memset(block_buf, 0, NXP_NFC_I2C_BLOCK_LEN);

	ret = nxp_nfc_i2c_read_block(client, NXP_NFC_CONFIG_REG_BLK, block_buf);
	if (ret < 0) {
		bb_log_dev(&(client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to read config regs.");
		return -EIO;
	}
	block_buf[CONFIG_REG_NCREG] = NCREG_DEFAULT;
	block_buf[CONFIG_REG_LAST_NDEF_BLK] = last_blk;
	ret = nxp_nfc_i2c_write_block(client, NXP_NFC_CONFIG_REG_BLK, block_buf);
	if (ret < 0) {
		bb_log_dev(&(client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to write config regs.");
		return -EIO;
	}

	ret = nxp_nfc_i2c_write_session_reg(client, SESSION_REG_LAST_NDEF_BLK, 0xff, last_blk);
	if (ret < 0) {
		bb_log_dev(&(client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to write session reg.");
		return -EIO;
	}
	return 0;
}

static int32_t nxp_nfc_erase_tag(struct i2c_client *client)
{
	uint8_t block_idx;
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN] = {0};
	int32_t err = 0;

	for(block_idx = NXP_NFC_USERMEM_START_BLK; block_idx <= NXP_NFC_USERMEM_END_BLK; block_idx++) {
		err = nxp_nfc_i2c_write_block(client, block_idx, block_buf);
		if (err) {
			break;
		}
	}
	return err;
}

static int32_t nxp_nfc_format_tag(struct i2c_client *client)
{
	uint8_t null_block[NXP_NFC_I2C_BLOCK_LEN] = {0x03, 0x03, 0xD0, 0x00, 0x00, 0xFF};
	int32_t err = 0;

	if ((err = nxp_nfc_erase_tag(client))) {
		return err;
	}
	if ((err = nxp_nfc_i2c_write_block(client, NXP_NFC_USERMEM_START_BLK, null_block)) < 0) {
		return err;
	}

	err = write_last_ndef_blk(nfc_data.client, NXP_NFC_USERMEM_START_BLK);

	return err;
}

static void nxp_nfc_parse_slock(uint8_t *rsp_buf, uint16_t *slock)
{
	*slock = (uint32_t)rsp_buf[11];
	*slock |= ((uint32_t)rsp_buf[10] << 8);
}

static void nxp_nfc_parse_cc(uint8_t *rsp_buf, uint32_t *cc)
{
	*cc = (uint32_t)rsp_buf[15];
	*cc |= ((uint32_t)rsp_buf[14] << 8);
	*cc |= ((uint32_t)rsp_buf[13] << 16);
	*cc |= ((uint32_t)rsp_buf[12] << 24);
}

static void nxp_nfc_parse_sn(uint8_t *rsp_buf, uint64_t *serial)
{
	*serial = (uint64_t)rsp_buf[6];
	*serial |= ((uint64_t)rsp_buf[5] << 8);
	*serial |= ((uint64_t)rsp_buf[4] << 16);
	*serial |= ((uint64_t)rsp_buf[3] << 24);
	*serial |= ((uint64_t)rsp_buf[2] << 32);
	*serial |= ((uint64_t)rsp_buf[1] << 40);
	*serial |= ((uint64_t)rsp_buf[0] << 48);
}

int32_t nxp_nfc_write_slock(struct i2c_client *client, uint16_t slock)
{
	int32_t ret;
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN];
	uint8_t block_idx;

	memset(block_buf, 0, NXP_NFC_I2C_BLOCK_LEN);
	block_idx = NXP_NFC_CC_BLK;
	ret = nxp_nfc_i2c_read_block(client, block_idx, block_buf);
	if (ret < 0) {
		bb_log_dev(&(client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to read block %x", block_idx);
		return -EIO;
	}

	block_buf[0] = nxp_nfc_i2c_addr;

	block_buf[10] = (slock & 0xff00) >> 8;
	block_buf[11] = (slock & 0x00ff);
	nxp_nfc_parse_slock(block_buf, &nfc_data.slock);

	ret = nxp_nfc_i2c_write_block(client, block_idx, block_buf);
	if (ret < 0) {
		bb_log_dev(&(client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to write block %x", block_idx);
		return -EIO;
	}

	return 0;
}

static int32_t nxp_nfc_write_cc(struct i2c_client *client, uint32_t cc)
{
	int32_t ret;
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN];
	uint8_t block_idx;

	memset(block_buf, 0, NXP_NFC_I2C_BLOCK_LEN);
	block_idx = NXP_NFC_CC_BLK;

	ret = nxp_nfc_i2c_read_block(client, block_idx, block_buf);
	if (ret < 0) {
		bb_log_dev(&(client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to read block %x", block_idx);
		return -EIO;
	}

	block_buf[0] = nxp_nfc_i2c_addr;

	block_buf[12] = (cc & 0xff000000) >> 24;
	block_buf[13] = (cc & 0x00ff0000) >> 16;
	block_buf[14] = (cc & 0x0000ff00) >> 8;
	block_buf[15] = (cc & 0x000000ff);
	nxp_nfc_parse_cc(block_buf, &nfc_data.cc);

	ret = nxp_nfc_i2c_write_block(client, block_idx, block_buf);
	if (ret < 0) {
		bb_log_dev(&(client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to write block %x", block_idx);
		return -EIO;
	}

	return 0;
}

static int32_t nxp_nfc_write_cc_block(struct i2c_client *client, uint32_t cc, uint16_t slock)
{
	int32_t ret;
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN];
	uint8_t block_idx;

	memset(block_buf, 0, NXP_NFC_I2C_BLOCK_LEN);
	block_idx = NXP_NFC_CC_BLK;
	ret = nxp_nfc_i2c_read_block(client, block_idx, block_buf);
	if (ret < 0) {
		bb_log_dev(&(client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to read block %x", block_idx);
		return -EIO;
	}

	nxp_nfc_parse_sn(block_buf, &nfc_data.serial_num);

	block_buf[0] = nxp_nfc_i2c_addr;

	block_buf[10] = (slock & 0xff00) >> 8;
	block_buf[11] = (slock & 0x00ff);

	block_buf[12] = (cc & 0xff000000) >> 24;
	block_buf[13] = (cc & 0x00ff0000) >> 16;
	block_buf[14] = (cc & 0x0000ff00) >> 8;
	block_buf[15] = (cc & 0x000000ff);
	nxp_nfc_parse_cc(block_buf, &nfc_data.cc);
	nxp_nfc_parse_slock(block_buf, &nfc_data.slock);

	ret = nxp_nfc_i2c_write_block(client, block_idx, block_buf);
	if (ret < 0) {
		bb_log_dev(&(client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to write block %x", block_idx);
		return -EIO;
	}

	return 0;
}

static ssize_t cc_show(struct device_driver *drv, char *buf)
{
	int32_t ret;
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN];
	uint8_t block_idx;

	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}

	memset(block_buf, 0, NXP_NFC_I2C_BLOCK_LEN);
	block_idx = NXP_NFC_CC_BLK;
	ret = nxp_nfc_i2c_read_block(nfc_data.client, block_idx, block_buf);
	if (ret < 0) {
		bb_log_dev(&(nfc_data.client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to read block %x", block_idx);
		return -EIO;
	}

	nxp_nfc_parse_cc(block_buf, &nfc_data.cc);

	return scnprintf(buf, PAGE_SIZE, "%08x\n", nfc_data.cc);
}

static ssize_t cc_store(struct device_driver *drv, const char *buf, size_t count)
{
	uint32_t val;

	if (!nfc_data.device_ready) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc device not ready");
		return -EFAULT;
	}
	if (kstrtouint(buf, 16, &val) !=0 ) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: Failed to parse the value.");
		return -EFAULT;
	}
	nxp_nfc_write_cc(nfc_data.client, val);

	return count;
}
static DRIVER_ATTR_RW(cc);

static ssize_t slock_show(struct device_driver *drv, char *buf)
{
	int32_t ret;
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN];
	uint8_t block_idx;

	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}

	memset(block_buf, 0, NXP_NFC_I2C_BLOCK_LEN);
	block_idx = NXP_NFC_CC_BLK;
	ret = nxp_nfc_i2c_read_block(nfc_data.client, block_idx, block_buf);
	if (ret < 0) {
		bb_log_dev(&(nfc_data.client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to read block %x", block_idx);
		return -EIO;
	}

	nxp_nfc_parse_slock(block_buf, &nfc_data.slock);

	return scnprintf(buf, PAGE_SIZE, "%04x\n", nfc_data.slock);
}

static ssize_t slock_store(struct device_driver *drv, const char *buf, size_t count)
{
	uint32_t val;

	if (!nfc_data.device_ready) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc device not ready");
		return -EFAULT;
	}

	if (kstrtouint(buf, 16, &val) !=0 ) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: Failed to parse the value.");
		return -EFAULT;
	}
	nxp_nfc_write_slock(nfc_data.client, (uint16_t)val);

	return count;
}
static DRIVER_ATTR_RW(slock);

static ssize_t serial_show(struct device_driver *drv, char *buf)
{
	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}

	return scnprintf(buf, PAGE_SIZE, "%014llx\n", nfc_data.serial_num);
}
static DRIVER_ATTR_RO(serial);

static ssize_t session_regs_show(struct device_driver *drv, char *buf)
{
	uint8_t idx;
	uint8_t val;
	int32_t ret;
	ssize_t count = 0;

	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}

	for (idx = 0; idx < SESSION_NUM_REGS; idx++) {
		ret = nxp_nfc_i2c_read_session_reg(nfc_data.client, idx, &val);
		if (ret < 0) {
			return scnprintf(buf, PAGE_SIZE, "unable to read session register.\n");
		}

		count += scnprintf(buf+count, PAGE_SIZE, "%02x ", val);
	}
	count += scnprintf(buf+count, PAGE_SIZE, "\n");

	return count;
}
static DRIVER_ATTR_RO(session_regs);

static ssize_t config_regs_show(struct device_driver *drv, char *buf)
{
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN];
	uint8_t block_idx;
	uint8_t byte;
	int32_t ret;
	ssize_t count = 0;

	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}

	memset(block_buf, 0, NXP_NFC_I2C_BLOCK_LEN);
	block_idx = NXP_NFC_CONFIG_REG_BLK;
	ret = nxp_nfc_i2c_read_block(nfc_data.client, block_idx, block_buf);
	if (ret < 0) {
		return scnprintf(buf, PAGE_SIZE, "unable to read block 0x%x\n", block_idx);
	}

	for (byte = 0; byte < NXP_NFC_I2C_BLOCK_LEN; byte++) {
		if ((byte%16) == 0) {
			count += scnprintf(buf+count, PAGE_SIZE,  "%02x: ", byte);
		}
		count += scnprintf(buf+count, PAGE_SIZE, "%02x ", block_buf[byte]);
	}
	count += scnprintf(buf+count, PAGE_SIZE, "\n");
	return count;
}
static DRIVER_ATTR_RO(config_regs);

static ssize_t dump_sram_show(struct device_driver *drv, char *buf)
{
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN];
	uint8_t block_idx;
	uint8_t byte;
	int32_t ret;
	ssize_t count = 0;

	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}

	for (block_idx = NXP_NFC_SRAM_START_BLK; block_idx <= NXP_NFC_SRAM_END_BLK; block_idx++) {
		memset(block_buf, 0, NXP_NFC_I2C_BLOCK_LEN);
		ret = nxp_nfc_i2c_read_block(nfc_data.client, block_idx, block_buf);
		if (ret < 0) {
			return scnprintf(buf, PAGE_SIZE, "unable to read block 0x%x\n", block_idx);
		}

		for (byte = 0; byte < NXP_NFC_I2C_BLOCK_LEN; byte++) {
			if ((byte%16) == 0) {
				count += scnprintf(buf+count, PAGE_SIZE,  "%02x: ", block_idx);
			}
			count += scnprintf(buf+count, PAGE_SIZE, "%02x ", block_buf[byte]);
		}
		count += scnprintf(buf+count, PAGE_SIZE, "\n");
	}
	return count;
}
static DRIVER_ATTR_RO(dump_sram);

static ssize_t dump_ndef_show(struct device_driver *drv, char *buf)
{
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN];
	uint8_t byte;
	int32_t ret;
	ssize_t count = 0;
	static uint8_t block_idx = 0;

	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}

	if (block_idx) {
		while (block_idx <= NXP_NFC_USERMEM_END_BLK) {
			if (count + 53 > PAGE_SIZE) {
				return count;
			}
			memset(block_buf, 0, NXP_NFC_I2C_BLOCK_LEN);
			ret = nxp_nfc_i2c_read_block(nfc_data.client, block_idx, block_buf);
			if (ret < 0) {
				return scnprintf(buf, PAGE_SIZE, "unable to read block 0x%x\n", block_idx);
			}

			for (byte = 0; byte < NXP_NFC_I2C_BLOCK_LEN; byte++) {
				if ((byte%16) == 0) {
					count += scnprintf(buf+count, PAGE_SIZE,  "%02x: ", block_idx);
				}
				count += scnprintf(buf+count, PAGE_SIZE, "%02x ", block_buf[byte]);
			}
			count += scnprintf(buf+count, PAGE_SIZE, "\n");
			block_idx++;
		}
	} else {
		for (block_idx = NXP_NFC_USERMEM_START_BLK; block_idx <= NXP_NFC_USERMEM_END_BLK; block_idx++) {
			if (count + 53 > PAGE_SIZE) {
				return count;
			}
			memset(block_buf, 0, NXP_NFC_I2C_BLOCK_LEN);
			ret = nxp_nfc_i2c_read_block(nfc_data.client, block_idx, block_buf);
			if (ret < 0) {
				return scnprintf(buf, PAGE_SIZE, "unable to read block 0x%x\n", block_idx);
			}

			for (byte = 0; byte < NXP_NFC_I2C_BLOCK_LEN; byte++) {
				if ((byte%16) == 0) {
					count += scnprintf(buf+count, PAGE_SIZE,  "%02x: ", block_idx);
				}
				count += scnprintf(buf+count, PAGE_SIZE, "%02x ", block_buf[byte]);
			}
			count += scnprintf(buf+count, PAGE_SIZE, "\n");
		}
	}
	block_idx = 0;
	return count;
}
static DRIVER_ATTR_RO(dump_ndef);

static ssize_t nsreg_show(struct device_driver *drv, char *buf)
{
	uint8_t regval;
	int32_t error;

	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}

	error = nxp_nfc_i2c_read_session_reg(nfc_data.client, SESSION_REG_NS_REG, &regval);
	if (error < 0) {
		return scnprintf(buf, PAGE_SIZE, "unable to read session register\n");
	}
	return scnprintf(buf, PAGE_SIZE, "%02x\n", regval);
}
static DRIVER_ATTR_RO(nsreg);

static ssize_t last_ndef_blk_show(struct device_driver *drv, char *buf)
{
	uint8_t regval;
	int32_t error;

	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}

	error = nxp_nfc_i2c_read_session_reg(nfc_data.client, SESSION_REG_LAST_NDEF_BLK, &regval);
	if (error < 0) {
		return scnprintf(buf, PAGE_SIZE, "unable to read session register\n");
	}
	return scnprintf(buf, PAGE_SIZE, "%02x\n", regval);
}
static DRIVER_ATTR_RO(last_ndef_blk);

static ssize_t write_enable_show(struct device_driver *drv, char *buf)
{
	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}
	return scnprintf(buf, PAGE_SIZE, "%d\n", nfc_data.allow_write);
}

static ssize_t write_enable_store(struct device_driver *drv, const char *buf, size_t count)
{
	int val;
	if (!nfc_data.device_ready) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc device not ready");
		return -EFAULT;
	}
	if (kstrtoint(buf, 0, &val) != 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: Failed to parse the value.");
		return -EFAULT;
	}
	nfc_data.allow_write = !!val;
	return count;
}
static DRIVER_ATTR_RW(write_enable);

#if defined(ENABLE_VOUT_DATA_COLLECTION)
static ssize_t vout_scan_show(struct device_driver *drv, char *buf)
{
	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}
	return scnprintf(buf, PAGE_SIZE, "%d\n", nfc_nxp_vout_scan_count);
}

static ssize_t vout_scan_store(struct device_driver *drv, const char *buf, size_t count)
{
	int val;
	if (!nfc_data.device_ready) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc device not ready");
		return -EFAULT;
	}
	if (kstrtoint(buf, 0, &val) != 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: Failed to parse the value.");
		return -EFAULT;
	}
	nfc_nxp_vout_scan_count = val;
	return count;
}
static DRIVER_ATTR_RW(vout_scan);
#endif

static ssize_t cmd_show(struct device_driver *drv, char *buf)
{
	if (!nfc_data.device_ready) {
		return scnprintf(buf, PAGE_SIZE, "nxp_nfc device not ready\n");
	}
	return scnprintf(buf, PAGE_SIZE, "Supported Commands:\nformat\nerase\n"
			"wrblk0x## <0xbyteval0> <0xbyteval1> ... <0xbyteval15>\n"
			"\teg:wrblk0x37 0x00 0x01 0x02 0x03 0x04 0x05 0x06 0x07 0x08 0x09 0x0a 0x0b 0x0c 0x0d 0x0e 0x0f\n"
			"\tnote: only the first 4 bytes of block 0x38 can be written.\n");
}

static ssize_t cmd_store(struct device_driver *drv, const char *buf, size_t count)
{
	const char *format = "format\n";
	const char *erase = "erase\n";
	const char *wrblk = "wrblk";
	int32_t wrblk_len = strlen(wrblk);
	uint8_t block_buf[NXP_NFC_I2C_BLOCK_LEN] = {0};
	int32_t block_id = 1;
	uint8_t args_rd;

	if (!nfc_data.device_ready) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc device not ready");
		return count;
	}
	if (!nfc_data.allow_write) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: write mode is not currently enabled.");
		return count;
	}
	if (strcmp(format, buf) == 0) {
		if (nxp_nfc_format_tag(nfc_data.client)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: unable to format tag.");
			return count;
		}
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "nxp_nfc: tag formatted successfully.");
		return count;
	}
	if (strcmp(erase, buf) == 0) {
		if (nxp_nfc_erase_tag(nfc_data.client)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: unable to erase tag.");
			return count;
		}
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "nxp_nfc: tag erased successfully.");
		return count;
	}
	if (strncmp(wrblk, buf, wrblk_len) == 0) {
		if (count < WRBLK_CMD_STRLEN_MIN || count > WRBLK_CMD_STRLEN_MAX) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: invalid number of characters for wrblk cmd.");
			return -EFAULT;
		}
		args_rd = sscanf(buf, "wrblk0x%2x 0x%2hhx 0x%2hhx 0x%2hhx 0x%2hhx 0x%2hhx 0x%2hhx 0x%2hhx 0x%2hhx "
				"0x%2hhx 0x%2hhx 0x%2hhx 0x%2hhx 0x%2hhx 0x%2hhx 0x%2hhx 0x%2hhx\n",
				&block_id, &block_buf[0],  &block_buf[1],  &block_buf[2], &block_buf[3],
				&block_buf[4],  &block_buf[5], &block_buf[6],  &block_buf[7],  &block_buf[8],
				&block_buf[9],  &block_buf[10],  &block_buf[11], &block_buf[12],  &block_buf[13],
				&block_buf[14], &block_buf[15]);
		if (args_rd) {
			if (block_id == 0) {
				bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: wrblk cmd cannot be used to write block 0");
				return -EFAULT;
			} else if (block_id == NXP_NFC_USERMEM_END_BLK && args_rd != 5) {
				bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: block 0x38 only supports writing 4 bytes");
				return -EFAULT;
			} else if (block_id == NXP_NFC_USERMEM_END_BLK && args_rd == 5) {
			} else if (args_rd != 17) {
				bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: wrblk cmd invalid number of arguments: %d", args_rd);
				return -EFAULT;
			}
			if ((nxp_nfc_i2c_write_block(nfc_data.client, block_id, block_buf)) == 0) {
				bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "nxp_nfc: wrblk successful write to block: %d", block_id);
				return count;
			}
		} else {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: wrblk cmd no arguments were parsed");
		}
		return -EFAULT;
	}
	bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: invalid command.");
	return count;
}
static DRIVER_ATTR_RW(cmd);

static struct attribute *nxp_nfc_attrs[] = {
#if defined(ENABLE_VOUT_DATA_COLLECTION)
	&driver_attr_vout_scan.attr,
#endif
	&driver_attr_write_enable.attr,
	&driver_attr_cmd.attr,
	&driver_attr_serial.attr,
	&driver_attr_cc.attr,
	&driver_attr_config_regs.attr,
	&driver_attr_session_regs.attr,
	&driver_attr_dump_sram.attr,
	&driver_attr_dump_ndef.attr,
	&driver_attr_nsreg.attr,
	&driver_attr_last_ndef_blk.attr,
	&driver_attr_slock.attr,
	NULL,
};

static const struct attribute_group nxp_nfc_attr_group = {
	.attrs = nxp_nfc_attrs
};

static const struct attribute_group *nxp_nfc_attr_groups[] = {
	&nxp_nfc_attr_group,
	NULL,
};

static int32_t nxp_nfc_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id)
{
	int32_t ret;

	nfc_data.client = i2c_client;

	ret = nxp_nfc_write_cc_block(i2c_client, 0xe1106d0f, 0x0100);
	if (ret < 0) {
		bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to init the cc block ");
		return -EIO;
	}


	ret = nxp_nfc_i2c_write_session_reg(i2c_client, SESSION_REG_NCREG, 0xff, NCREG_DEFAULT);
	if (ret < 0) {
		bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to write session reg.");
		return -EIO;
	}

	ret = nxp_nfc_i2c_write_session_reg(i2c_client, SESSION_REG_SRAM_MIRROR_BLK, 0xff, 0);
	if (ret < 0) {
		bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Unable to write session reg.");
		return -EIO;
	}

	bb_log_dev(&(i2c_client->dev), BB_MOD_SENSORS, BB_LVL_INFO, "SN: %llx registered", nfc_data.serial_num);
	nfc_data.device_ready = 1;
	return 0;
}

int32_t nxp_nfc_i2c_remove(struct i2c_client *i2c_client)
{
	return 0;
}

static const struct i2c_device_id nxp_nfc_i2c_id[] = {
	{ "nxp_nt3h2111", 0 },
	{  }
};
MODULE_DEVICE_TABLE(i2c, nxp_nfc_i2c_id);

static struct of_device_id ti_nxp_nfc_ids[] = {
	{ .compatible = "fsl,nxp-nt3h2111" },
	{  }
};

static struct i2c_driver nxp_nfc_i2c_driver = {
	.driver = {
		.name		= "nxp_nt3h2111",
		.owner		= THIS_MODULE,
		.of_match_table	= ti_nxp_nfc_ids,
		.groups		= nxp_nfc_attr_groups,
	},
	.id_table	= nxp_nfc_i2c_id,
	.probe		= nxp_nfc_i2c_probe,
	.remove		= nxp_nfc_i2c_remove,
};


bool nxp_nfc_is_field_detect_active(void)
{
	bool ret = false;

	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "gpio_get_value(field_detect.gpio) = %d", gpio_get_value(field_detect.gpio));
	if (!gpio_get_value(field_detect.gpio)) {
		ret = true;
	}

	return ret;
}

#if defined(ENABLE_VOUT_DATA_COLLECTION)
void nxp_nfc_vout_collection(struct work_struct *work)
{
	int mvolts;
	int zero_count = 0;
	int total=0;
	int count=0;

	if (vout_adc_port < 0) return;

	if (nfc_nxp_vout_scan_count) {
		printk(KERN_ERR "scanning nfc_vout_adcin\n");
		printk(KERN_ERR "  sample interval:  %d ms\n", VOUT_SAMPLE_RATE);
		printk(KERN_ERR "  filtering below %d mvolts\n", VOUT_NOISE_FILTER);
	}
	while(nfc_nxp_vout_scan_count > 0) {
		(void)read_adc_voltage(vout_adc_port, &mvolts);
		if (mvolts <= VOUT_NOISE_FILTER) {
			if (!zero_count && count) {
				printk(KERN_ERR "  %d average [%d/%d] over last %d ms\n", (int)(total / count), total, count, count * VOUT_SAMPLE_RATE);
			}
			zero_count++;
			count = 0;
			total = 0;
		} else {
			total += mvolts;
			count++;
			if (zero_count) {
				printk(KERN_ERR "%d mvolts [%d] after %d samples of 0\n", mvolts, total, zero_count);
			} else {
				printk(KERN_ERR "%d mvolts [%d]\n", mvolts, total);
			}
			zero_count = 0;
		}
		nfc_nxp_vout_scan_count--;
		if (!nfc_nxp_vout_scan_count) {
			printk(KERN_ERR "sampling done - last %d samples were 0\n", zero_count);
		}
		mdelay(VOUT_SAMPLE_RATE);
	}

	schedule_delayed_work(&nxp_nfc_vout_work, VOUT_COLLECTION_INTERVAL);
	(void)mvolts;
}
#endif

irqreturn_t nxp_nfc_isr(int irq, void *data)
{
	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "nxp_nfc_isr handled");
	event_queue_send_event_defer(HWEVTQSOURCE_NFC, HWEVTQINFO_UPDATE);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	hwevtq_send_event_defer(HWEVTQSOURCE_NFC, HWEVTQINFO_UPDATE);
#endif
	return IRQ_HANDLED;
}

static int32_t nfc_cdev_open(struct inode *inodep, struct file *filp)
{
	loff_t *idx;

	filp->private_data = kmalloc(sizeof(loff_t), GFP_KERNEL);

	if(!filp->private_data) {
		bb_log_dev(&(nfc_data.client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Failed to allocate memory");
		return -ENOMEM;
	}
	idx = (loff_t*)filp->private_data;
	*idx = 0;

	return 0;
}

static int32_t validate_size(size_t * size)
{
	size_t tmp_size = *size;

	if((tmp_size < NXP_NFC_I2C_BLOCK_LEN || ((tmp_size % NXP_NFC_I2C_BLOCK_LEN) != 0))) {
		bb_log_dev(&(nfc_data.client->dev), BB_MOD_SENSORS, BB_LVL_INFO, "Invalid size specified: %zu.", tmp_size);
		return -EINVAL;
	}

	if(tmp_size > NXP_NFC_CHR_BUF_LEN) {
		tmp_size = NXP_NFC_CHR_BUF_LEN;
	}

	*size = tmp_size;
	return 0;
}

static int32_t validate_idx(loff_t cur_idx, loff_t offset, size_t * _start_idx)
{
	size_t start_idx = *_start_idx;

	start_idx = cur_idx + NXP_NFC_I2C_BLOCK_LEN;
	pr_debug("%s: start_idx %zu cur_idx: %lld offset: %lld\n",__func__, start_idx, cur_idx, offset);

	if (offset != 0) {
		start_idx += offset;
		pr_debug("%s: offset value is nonzero %lld start_idx %zu\n",__func__, offset, start_idx);
	}

	if (start_idx >= NXP_NFC_MAX_FILE_IDX) {
		pr_debug("%s: start_idx has reached end, start_idx %zu \n",__func__, start_idx);
		return -ESPIPE;
	}

	if((start_idx % NXP_NFC_I2C_BLOCK_LEN) != 0) {
		return -EINVAL;
	}

	*_start_idx = start_idx;

	return 0;
}

static ssize_t nfc_cdev_read(struct file *filp, char __user *buffer, size_t size, loff_t *offset)
{
	uint8_t kbuf[NXP_NFC_CHR_BUF_LEN];
	loff_t *cur_idx;
	uint8_t block_idx = 0;
	size_t start_idx;
	size_t bytes_to_return = 0;
	int32_t start_block = 0;
	int32_t end_block = 0;
	int32_t ret;

	memset(kbuf, 0, NXP_NFC_CHR_BUF_LEN);

	cur_idx = (loff_t*)filp->private_data;

	if ((ret = validate_idx(*cur_idx, *offset, &start_idx)) != 0) {
		if (ret == -ESPIPE) {
			return 0;
		}
		return ret;
	}

	if ((ret = validate_size(&size)) != 0) {
		return ret;
	}

	memset(kbuf, 0, NXP_NFC_CHR_BUF_LEN);

	start_block = (start_idx / NXP_NFC_I2C_BLOCK_LEN);
	end_block = start_block + (size / NXP_NFC_I2C_BLOCK_LEN);
	end_block = end_block % (NXP_NFC_USERMEM_END_BLK + 1);

	for (block_idx = start_block; block_idx < end_block; block_idx++) {
		if ((ret = nxp_nfc_i2c_read_block(nfc_data.client, block_idx, kbuf+bytes_to_return)) < 0) {
			return ret;
		}
		bytes_to_return += NXP_NFC_I2C_BLOCK_LEN;
	}

	if (bytes_to_return == 0) {
		bb_log_dev(&(nfc_data.client->dev), BB_MOD_SENSORS, BB_LVL_INFO, "%s called, but no bytes were written to user space", __func__);
		return 0;
	}

	ret = copy_to_user(buffer, kbuf, bytes_to_return);
	if(ret) {
		bb_log_dev(&(nfc_data.client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Failed to copy buffer from kernel to userspace");
		return ret;
	}

	*cur_idx += bytes_to_return;

	return bytes_to_return;
}

static ssize_t nfc_cdev_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset)
{
	uint8_t kbuf[NXP_NFC_CHR_BUF_LEN];
	loff_t *cur_idx;
	int32_t start_block = 0;
	int32_t end_block = 0;
	uint8_t block_idx = 0;
	size_t start_idx;
	size_t bytes_to_return = 0;
	int32_t ret;

	memset(kbuf, 0, NXP_NFC_CHR_BUF_LEN);

	cur_idx = (loff_t*)filp->private_data;

	if ((ret = validate_idx(*cur_idx, *offset, &start_idx)) != 0) {
		if (ret == -ESPIPE) {
			return 0;
		}
		return ret;
	}

	if ((ret = validate_size(&size)) != 0) {
		return ret;
	}

	if (copy_from_user(kbuf, buffer, size)) {
		bb_log_dev(&(nfc_data.client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Failed to copy buffer from userspace to kernel");
		return -1;
	}

	start_block = (start_idx / NXP_NFC_I2C_BLOCK_LEN);
	end_block = start_block + (size / NXP_NFC_I2C_BLOCK_LEN);
	end_block = end_block % (NXP_NFC_USERMEM_END_BLK + 1);

	for (block_idx = start_block; block_idx < end_block; block_idx++) {
		if ((ret = nxp_nfc_i2c_write_block(nfc_data.client, block_idx, kbuf+bytes_to_return)) < 0) {
			return ret;
		}

		bytes_to_return += NXP_NFC_I2C_BLOCK_LEN;
	}

	if (bytes_to_return == 0) {
		bb_log_dev(&(nfc_data.client->dev), BB_MOD_SENSORS, BB_LVL_INFO, "%s was called but no bytes were written to the device", __func__);
		return 0;
	}

	if ((ret = write_last_ndef_blk(nfc_data.client, end_block > 1 ? end_block - 1 : 1)) < 0) {
		return ret;
	}

	*cur_idx += bytes_to_return;

	return bytes_to_return;
}

static loff_t nfc_cdev_llseek(struct file *filp, loff_t offset, int32_t seek_type)
{
	loff_t *cur_idx;
	loff_t tmp_idx;

	cur_idx = (loff_t*)filp->private_data;
	tmp_idx = *cur_idx;

	if(seek_type == SEEK_CUR) {
		tmp_idx += offset;
	} else if (seek_type == SEEK_SET) {
		tmp_idx = offset;
	} else if (seek_type == SEEK_END) {
		tmp_idx = NXP_NFC_MAX_FILE_IDX;
	} else {
		return -EINVAL;
	}

	*cur_idx = tmp_idx;
	return tmp_idx;
}

static int32_t nfc_cdev_release(struct inode *inodep, struct file *filp)
{
	kfree(filp->private_data);
	return 0;
}

struct file_operations nfc_cdev_fops =
{
	.open		= nfc_cdev_open,
	.read		= nfc_cdev_read,
	.write		= nfc_cdev_write,
	.llseek		= nfc_cdev_llseek,
	.release	= nfc_cdev_release,
};

void nxp_nfc_init(void)
{
	int32_t error;
	u32 alt_address;
        struct device_node *dev_node;
	char *status_check, chkbuf[10];
	struct device *class_dev = NULL;
        dev_node = of_find_compatible_node(NULL, NULL, "fsl,nxp-nt3h2111");

	nxp_nfc_i2c_addr = NXP_NFC_DEFAULT_I2C_ADDRESS;

	if (!dev_node) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "No nxp-nt3h2111 in this device");
		return;
	}

#if defined(SONOS_ARCH_ATTR_SOC_IS_IMX6)
	if (((sys_mdp.mdp_model == MDP_MODEL_TITAN) && (sys_mdp.mdp_revision < MDP_REVISION_TITAN_PROTO1B))) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "nxp-nt3h2111 in the dtb, but not accessible on this revision");
		return;
	}
#endif

	status_check = chkbuf;
	of_property_read_string(dev_node, "status", (const char **)&status_check);
	if (strncmp(status_check, "okay", 4)) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "nxp-nt3h2111 not enabled");
		return;
	}

	error = alloc_chrdev_region(&devno, 0, 1, NXP_NFC_DEVICE_NAME);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Error %d allocating device %s.", error, NXP_NFC_DEVICE_NAME);
		return;
	}

	if (!of_property_read_u32(dev_node, "uses-alternate-address", &alt_address)) {
		nxp_nfc_i2c_addr = alt_address << 1;
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "alternate address - NFC device will use i2c address 0x%x (%x/%x)", \
				(nxp_nfc_i2c_addr >> 1), nxp_nfc_i2c_addr, (nxp_nfc_i2c_addr | 1) );
	}

	if ((field_detect.gpio = of_get_named_gpio(dev_node, "nfc-field-detect", 0)) >= 0) {
		if (gpio_is_valid(field_detect.gpio)) {
			error = gpio_request_one(field_detect.gpio, GPIOF_DIR_IN, "nfc-field-detect");
			if (error) {
				bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Could not get the nfc-field-detect interrupt pin %d", field_detect.gpio);
			} else {
				error = request_irq(gpio_to_irq(field_detect.gpio), nxp_nfc_isr, field_detect.flags, field_detect.label, NULL);
				if (error) {
					bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Could not get field-detect interrupt %d", gpio_to_irq(field_detect.gpio));
				}
			}
		}
	}

#if defined(ENABLE_VOUT_DATA_COLLECTION)
	if (of_property_read_u32(dev_node, "nfc-adc-port", &vout_adc_port)) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "No nfc adc monitor port.");
		vout_adc_port = -1;
	} else {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "vout_adc_port %d will show nfc VOUT values", vout_adc_port);
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "init delayed work");
		INIT_DELAYED_WORK(&nxp_nfc_vout_work, nxp_nfc_vout_collection);
		schedule_delayed_work(&nxp_nfc_vout_work, msecs_to_jiffies(5000));
	}
#endif

	error = i2c_add_driver(&nxp_nfc_i2c_driver);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "nxp_nfc: i2c_add_driver failed with %d", error);
	}

	cdev_init(&nfc_chr_dev, &nfc_cdev_fops);
	nfc_chr_dev.owner = THIS_MODULE;

	error = cdev_add(&nfc_chr_dev, devno, 1);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s Couldn't add character device (%d).", __func__, error);
		unregister_chrdev_region(devno, 1);
		return;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, NXP_NFC_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		error = PTR_ERR(class_dev);
		bb_log(BB_MOD_APPLE, BB_LVL_ERR, "Error creating %s sonos device. error %d", NXP_NFC_DEVICE_NAME, error);
		return;
	}

	driver_loaded = true;
}

void nxp_nfc_exit(void)
{
#if defined (ENABLE_VOUT_DATA_COLLECTION)
	cancel_delayed_work_sync(&nxp_nfc_vout_work);
#endif
	if (driver_loaded) {
		i2c_del_driver(&nxp_nfc_i2c_driver);
		sonos_device_destroy(devno);
		cdev_del(&nfc_chr_dev);
		driver_loaded = false;
	}

	if (gpio_is_valid(field_detect.gpio)) {
		free_irq(gpio_to_irq(field_detect.gpio), NULL);
		gpio_free(field_detect.gpio);
	}
}
