/*
 * Copyright (c) 2021 Sonos Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 *
 * Reads an ADC voltage and converts it to a temperature, based on the
 * presence of an NCU15 NTC device.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/thermal.h>
#include <linux/err.h>
#include <linux/sonos_kernel.h>
#include "blackbox.h"
#include "sensors_dev.h"
#include "sensors_hal.h"

#define MAX_RANGES	20
#define NEGATIVE	0
#define POSITIVE	1

struct ncu15_inst {
	char	*name;
	int	adc_chan;
	int	fault_temperature;
	int	warn_temperature;
	int	hTemp;
	int	polarity;
	int	num_ranges;
	int	range_max[MAX_RANGES];
	int	a[MAX_RANGES];
	int	b[MAX_RANGES];
};

struct ncu15_inst ncu15_insts[] = {
	{.name = "AMP"},
	{.name = "CPU"}
};

int ma_index = -1;
int ma_mV[4];
int mV_total;

int ncu15_build_map_from_node(struct device_node *map, struct ncu15_inst *ti)
{
	int	ret = 0;
	int	val;
	int	i, index;

	of_property_read_u32(map, "num-ranges", &val);
	if (val < 1 || val > MAX_RANGES) {
		return -EINVAL;
	}

	ti->num_ranges = val;

	if (of_property_read_bool(map, "negative")) {
		ti->polarity = NEGATIVE;
	} else {
		ti->polarity = POSITIVE;
	}

	for (i = 0;i < ti->num_ranges;i++) {
		index = (3 * i);
		of_property_read_u32_index(map, "map-values", index, &ti->range_max[i]);
		index = (3 * (i+1)) - 2;
		of_property_read_u32_index(map, "map-values", index, &ti->a[i]);
		index = (3 * (i+1)) - 1;
		of_property_read_u32_index(map, "map-values", index, &ti->b[i]);

		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "up to %d mVolts - a = %d, b = %d", ti->range_max[i], ti->a[i], ti->b[i]);
	}

	return ret;
}

#define AMPTEMP_NUM_INSTS (sizeof(ncu15_insts)/sizeof(struct ncu15_inst))


void ncu15_mv_to_temp(int mVolts, int *retTemp, struct ncu15_inst *ti)
{
	int i;


	for (i = 0; i < ti->num_ranges; i++) {
		if (mVolts < ti->range_max[i]) {
			if (ti->polarity  == NEGATIVE) {
				*retTemp = ((ti->b[i] - (ti->a[i] * mVolts))/10000);
			} else {
				*retTemp = ((ti->b[i] + (ti->a[i] * mVolts))/10000);
			}
			break;
		}
	}
}

int ncu15_read_temp(int inst, int *temp)
{
	struct ncu15_inst *ti = &ncu15_insts[inst];
	int error, mVolts;

	error = sonos_sar_adc_iio_info_read_raw(ti->adc_chan, &mVolts);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s: %s ADC read error %d", __func__, ti->name, error);
		*temp = 0;
	} else {
		if (ma_index == -1) {
			ma_index = 0;
			ma_mV[0] = ma_mV[1] = ma_mV[2] = ma_mV[3] = mVolts;
			mV_total = 4 * mVolts;
		} else {
			mV_total -= ma_mV[ma_index];
			mV_total += mVolts;
			ma_mV[ma_index] = mVolts;
			ma_index = (ma_index + 1) % 4;
		}

		ncu15_mv_to_temp((mV_total / 4), temp, ti);
		bb_log_dbg(BB_MOD_SENSORS, "NCU15 mapping %d mV to %d degrees C", mVolts, *temp);
	}

#if 0
	{
		int i;
		static int done = 0;
		if (done++ == 5) {
			for (i = 150;i < 3800;i++) {
				int map_temp = 0;
				ncu15_mv_to_temp(i, &map_temp, ti);
				printk(KERN_ERR "%04d  %05d\n", i, map_temp);
			}
		}
	}
#endif
	return (0);
}

static struct device_node * ncu15_find_dtb_entry(char *inst_name)
{
	struct device_node *np;

	np = of_find_compatible_node(NULL, NULL, "sonos,temp-ncu15");

	if (np == NULL) {
		return NULL;
	}
	if (strcmp(np->name, inst_name) == 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "ncu15_%s", inst_name);
		return np;
	}
	return NULL;
}

static int ncu15_get_device(struct ncu15_inst *ti)
{
	struct device_node *node, *child;
	char *prop_name;
	int ret;
	u32 val32;

	node = ncu15_find_dtb_entry(ti->name);
	if (node == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "No sonos-ncu15 compatible device on this system");
		return -ENOENT;
	}

	prop_name = "adc-channel";
	ret = of_property_read_u32_array(node, prop_name, &val32, 1);
	ti->adc_chan = val32;
	if (ret < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s missing from %s ncu15_ entry", prop_name, ti->name);
		return -ENOENT;
	}
	ti->adc_chan = val32;

	prop_name = "fault-temperature";
	ret = of_property_read_u32_array(node, prop_name, &val32, 1);
	if (ret < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s missing from %s ncu15_ entry", prop_name, ti->name);
		return -ENOENT;
	}
	ti->fault_temperature= val32;

	prop_name = "warn-temperature";
	ret = of_property_read_u32_array(node, prop_name, &val32, 1);
	if (ret < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s missing from %s ncu15_ entry", prop_name, ti->name);
		return -ENOENT;
	}
	ti->warn_temperature= val32;

	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s temp-sensor - ncu15 on adc channel %d  (%d/%d)",
				ti->name, ti->adc_chan, ti->fault_temperature, ti->warn_temperature);

	child = of_get_child_by_name(node, "temperature-map");
	if (child == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "no temperature-map node for this device");
		return -ENOENT;
	}

	ret = ncu15_build_map_from_node(child, ti);

	return ret;
}

void ncu15_init(void)
{
	int i=0, error;
	int temp;
	struct ncu15_inst *ti = &ncu15_insts[i];

	error = ncu15_get_device(ti);
	if (!error) {
	    if (thermal_mgmt_map_device(i, ti->name, "sonos,temp-ncu15", ncu15_read_temp)) {
		ncu15_read_temp(i, &temp);
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s: current temperature %d Celsius", ti->name, temp);
		}
	}
}

void ncu15_exit(void)
{
}

