/*
 * Copyright (c) 2016 Sonos Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */


#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <asm/uaccess.h>
#include "blackbox.h"
#include "sensors_dev.h"

#define MMA8453Q_STATUS                         0x00
#define MMA8453Q_STATUS_ZYXOW                   0x80
#define MMA8453Q_STATUS_ZOW                     0x40
#define MMA8453Q_STATUS_YOW                     0x20
#define MMA8453Q_STATUS_XOW                     0x10
#define MMA8453Q_STATUS_ZYXDR                   0x08
#define MMA8453Q_STATUS_ZDR                     0x04
#define MMA8453Q_STATUS_YDR                     0x02
#define MMA8453Q_STATUS_XDR                     0x01
#define MMA8453Q_OUT_X_MSB                      0x01
#define MMA8453Q_OUT_X_LSB                      0x02
#define MMA8453Q_OUT_Y_MSB                      0x03
#define MMA8453Q_OUT_Y_LSB                      0x04
#define MMA8453Q_OUT_Z_MSB                      0x05
#define MMA8453Q_OUT_Z_LSB                      0x06
#define MMA8453Q_SYSMOD                         0x0b
#define MMA8453Q_SYSMOD_SYSMOD1                 0x02
#define MMA8453Q_SYSMOD_SYSMOD0                 0x01
#define MMA8453Q_INT_SOURCE                     0x0c
#define MMA8453Q_INT_SOURCE_SRC_ASLP            0x80
#define MMA8453Q_INT_SOURCE_SRC_TRANS           0x20
#define MMA8453Q_INT_SOURCE_SRC_LNDPRT          0x10
#define MMA8453Q_INT_SOURCE_SRC_PULSE           0x08
#define MMA8453Q_INT_SOURCE_SRC_FF_MT           0x04
#define MMA8453Q_INT_SOURCE_SRC_DRDY            0x01
#define MMA8453Q_WHO_AM_I                       0x0d
#define MMA8453Q_WHO_AM_I_ID                    0x3a
#define MMA8453Q_XYZ_DATA_CFG                   0x0e
#define MMA8453Q_XYZ_DATA_CFG_FS_MASK           0x03
#define MMA8453Q_HP_FILTER_CUTOFF               0x0f
#define MMA8453Q_HP_FILTER_CUTOFF_PULSE_HPF_BYP 0x02
#define MMA8453Q_HP_FILTER_CUTOFF_PULSE_LPF_EN  0x01
#define MMA8453Q_HP_FILTER_CUTOFF_SEL_MASK      0x03
#define MMA8453Q_PL_STATUS                      0x10
#define MMA8453Q_PL_STATUS_NEWLP                0x80
#define MMA8453Q_PL_STATUS_LO                   0x40
#define MMA8453Q_PL_STATUS_LAPO_MASK            0x06
#define MMA8453Q_PL_STATUS_BAFRO                0x01
#define MMA8453Q_PL_CFG                         0x11
#define MMA8453Q_PL_CFG_DBCNTM                  0x80
#define MMA8453Q_PL_CFG_PL_EN                   0x40
#define MMA8453Q_PL_COUNT                       0x12
#define MMA8453Q_PL_BF_ZCOMP                    0x13
#define MMA8453Q_PL_BF_ZCOMP_BKFR_MASK          0xc0
#define MMA8453Q_PL_BF_ZCOMP_ZLOCK_MASK         0x07
#define MMA8453Q_P_L_THIS_REG                   0x14
#define MMA8453Q_P_L_THIS_REG_P_L_HTS_MASK      0xf8
#define MMA8453Q_P_L_THIS_REG_HYS_MASK          0x07
#define MMA8453Q_FF_MT_CFG                      0x15
#define MMA8453Q_FF_MT_CFG_ELE                  0x80
#define MMA8453Q_FF_MT_CFG_OAE                  0x40
#define MMA8453Q_FF_MT_CFG_ZEFE                 0x20
#define MMA8453Q_FF_MT_CFG_YEFE                 0x10
#define MMA8453Q_FF_MT_CFG_XEFE                 0x08
#define MMA8453Q_FF_MT_SRC                      0x16
#define MMA8453Q_FF_MT_SRC_EA                   0x80
#define MMA8453Q_FF_MT_SRC_ZHE                  0x20
#define MMA8453Q_FF_MT_SRC_ZHP                  0x10
#define MMA8453Q_FF_MT_SRC_YHE                  0x08
#define MMA8453Q_FF_MT_SRC_YHP                  0x04
#define MMA8453Q_FF_MT_SRC_XHE                  0x02
#define MMA8453Q_FF_MT_SRC_XHP                  0x01
#define MMA8453Q_FF_MT_THS                      0x17
#define MMA8453Q_FF_MT_THS_DBCNTM               0x80
#define MMA8453Q_FF_MT_THS_THS_MASK             0x7f
#define MMA8453Q_FF_MT_COUNT                    0x18
#define MMA8453Q_TRANSIENT_CFG                  0x1d
#define MMA8453Q_TRANSIENT_CFG_ELE              0x10
#define MMA8453Q_TRANSIENT_CFG_ZTEFE            0x08
#define MMA8453Q_TRANSIENT_CFG_YTEFE            0x04
#define MMA8453Q_TRANSIENT_CFG_XTEFE            0x02
#define MMA8453Q_TRANSIENT_CFG_HPF_BYP          0x01
#define MMA8453Q_TRANSIENT_SRC                  0x1e
#define MMA8453Q_TRANSIENT_SRC_EA               0x40
#define MMA8453Q_TRANSIENT_SRC_ZTRANSE          0x20
#define MMA8453Q_TRANSIENT_SRC_Z_TRANS_POL      0x10
#define MMA8453Q_TRANSIENT_SRC_YTRANSE          0x08
#define MMA8453Q_TRANSIENT_SRC_Y_TRANS_POL      0x04
#define MMA8453Q_TRANSIENT_SRC_XTRANSE          0x02
#define MMA8453Q_TRANSIENT_SRC_X_TRANS_POL      0x01
#define MMA8453Q_TRANSIENT_THS                  0x1f
#define MMA8453Q_TRANSIENT_THS_DBCNTM           0x80
#define MMA8453Q_TRANSIENT_THS_THS_MASK         0x7f
#define MMA8453Q_TRANSIENT_COUNT                0x20
#define MMA8453Q_PULSE_CFG                      0x21
#define MMA8453Q_PULSE_CFG_DPA                  0x80
#define MMA8453Q_PULSE_CFG_ELE                  0x40
#define MMA8453Q_PULSE_CFG_ZDPEFE               0x20
#define MMA8453Q_PULSE_CFG_ZSPEFE               0x10
#define MMA8453Q_PULSE_CFG_YDPEFE               0x08
#define MMA8453Q_PULSE_CFG_YSPEFE               0x04
#define MMA8453Q_PULSE_CFG_XDPEFE               0x02
#define MMA8453Q_PULSE_CFG_XSPEFE               0x01
#define MMA8453Q_PULSE_SRC                      0x22
#define MMA8453Q_PULSE_SRC_EA                   0x80
#define MMA8453Q_PULSE_SRC_AXZ                  0x40
#define MMA8453Q_PULSE_SRC_AXY                  0x20
#define MMA8453Q_PULSE_SRC_AXX                  0x10
#define MMA8453Q_PULSE_SRC_DPE                  0x08
#define MMA8453Q_PULSE_SRC_POLZ                 0x04
#define MMA8453Q_PULSE_SRC_POLY                 0x02
#define MMA8453Q_PULSE_SRC_POLX                 0x01
#define MMA8453Q_PULSE_THSX                     0x23
#define MMA8453Q_PULSE_THSY                     0x24
#define MMA8453Q_PULSE_THSZ                     0x25
#define MMA8453Q_PULSE_TMLT                     0x26
#define MMA8453Q_PULSE_LTCY                     0x27
#define MMA8453Q_PULSE_WIND                     0x28
#define MMA8453Q_ASLP_COUNT                     0x29
#define MMA8453Q_CTRL_REG1                      0x2a
#define MMA8453Q_CTRL_REG1_ASLP_RATE_MASK       0xc0
#define MMA8453Q_CTRL_REG1_DR_MASK              0x38
#define MMA8453Q_CTRL_REG1_DR_SHFT              3
#define MMA8453Q_CTRL_REG1_LNOISE               0x04
#define MMA8453Q_CTRL_REG1_F_READ               0x02
#define MMA8453Q_CTRL_REG1_ACTIVE               0x01
#define MMA8453Q_CTRL_REG2                      0x2b
#define MMA8453Q_CTRL_REG2_ST                   0x80
#define MMA8453Q_CTRL_REG2_RST                  0x40
#define MMA8453Q_CTRL_REG2_SMODS_MASK           0x18
#define MMA8453Q_CTRL_REG2_SLPE                 0x04
#define MMA8453Q_CTRL_REG2_MODS_MASK            0x03
#define MMA8453Q_CTRL_REG3                      0x2c
#define MMA8453Q_CTRL_REG3_WAKE_TRANS           0x40
#define MMA8453Q_CTRL_REG3_WAKE_LNDPRT          0x20
#define MMA8453Q_CTRL_REG3_WAKE_PULSE           0x10
#define MMA8453Q_CTRL_REG3_WAKE_FF_MT           0x08
#define MMA8453Q_CTRL_REG3_IPOL                 0x02
#define MMA8453Q_CTRL_REG3_PP_OD                0x01
#define MMA8453Q_CTRL_REG4                      0x2d
#define MMA8453Q_CTRL_REG4_INT_EN_ASLP          0x80
#define MMA8453Q_CTRL_REG4_INT_EN_TRANS         0x20
#define MMA8453Q_CTRL_REG4_INT_EN_LNDPRT        0x10
#define MMA8453Q_CTRL_REG4_INT_EN_PULSE         0x08
#define MMA8453Q_CTRL_REG4_INT_EN_FF_MT         0x04
#define MMA8453Q_CTRL_REG4_INT_EN_DRDY          0x01
#define MMA8453Q_CTRL_REG5                      0x2e
#define MMA8453Q_CTRL_REG5_INT_CFG_ASLP         0x80
#define MMA8453Q_CTRL_REG5_INT_CFG_TRANS        0x20
#define MMA8453Q_CTRL_REG5_INT_CFG_LNDPRT       0x10
#define MMA8453Q_CTRL_REG5_INT_CFG_PULSE        0x08
#define MMA8453Q_CTRL_REG5_INT_CFG_FF_MT        0x04
#define MMA8453Q_CTRL_REG5_INT_CFG_DRDY         0x01
#define MMA8453Q_OFF_X                          0x2f
#define MMA8453Q_OFF_Y                          0x30
#define MMA8453Q_OFF_Z                          0x31

struct i2c_client *mma8453q_client = NULL;
int mma8453q_device_ready = 0;

int mma8453q_get_raw_accel(struct accel_data *ad)
{
	int reg;
	char raw_data[7];

	if (mma8453q_client == NULL) {
		return -EINVAL;
	}

	for (reg = 1; reg < 7; reg++) {
		s32 ret = i2c_smbus_read_byte_data(mma8453q_client, reg);
		if (ret < 0) {
			return -EIO;
		}
		raw_data[reg] = ret;
	}
	memset(ad, 0, sizeof(ad));
	ad->x = raw_data[1] << 8 | raw_data[2];
	ad->x /= 64;
	ad->y = raw_data[3] << 8 | raw_data[4];
	ad->y /= 64;
	ad->z = raw_data[5] << 8 | raw_data[6];
	ad->z /= 64;

	return 0;
}

static int mma8453q_proc_read(struct seq_file *m, void *v)
{
	int reg;
	uint8_t val[0x31] = {0};
	struct accel_data ad;

	if (!mma8453q_device_ready) {
		seq_printf(m, "mma8453q device not ready\n");
		return 0;
	}
	if (mma8453q_client == NULL) {
		seq_printf(m, "MMA8453Q: I2C mma8453q_client not initialized.\n");
		return 0;
	}
	for (reg = 0; reg < 0x31; reg++) {
		s32 ret = i2c_smbus_read_byte_data(mma8453q_client, reg);
		if (ret < 0) {
			int r;
			for (r = reg; r < 0x31; r++) {
				val[r] = 0;
			}
			seq_printf(m, "MMA8453Q: i2c read error on reg %02x\n", reg);
			break;
		}
		val[reg] = ret;
	}
	for (reg = 0; reg < 0x31; reg++) {
		if (reg && ((reg%16) == 0))
			seq_printf(m, "\n");
		if (((reg%16) == 0))
			seq_printf(m, "%2x: ", reg);
		if (reg && ((reg%4) == 0) &&  ((reg%16) != 0))
			seq_printf(m, "  ");
		seq_printf(m, "%02x ", val[reg]);
	}
	seq_printf(m, "\n");

	if (mma8453q_get_raw_accel(&ad) < 0) {
		seq_printf(m, "error reading acceleration\n");
	}
	else {
		seq_printf(m, "x=%4d y=%4d z=%4d\n", ad.x, ad.y, ad.z);
	}

	return 0;
}

static ssize_t mma8453q_proc_write(struct file *file, const char __user * buffer,
			size_t count, loff_t *data)
{
	int num_regs = 0x31;
	char buf[200];
	char *peq;
	int result = 0, error;
	int temp;
	u32  regnum;
	uint8_t  val;

	if (!mma8453q_device_ready) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "mma8453q device not ready");
		result = -EIO;
	} else if (count >= sizeof(buf)) {
		result = -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} else {
		buf[count] = '\0';
		peq = strchr(buf, '=');
		if (peq != NULL) {
			*peq = 0;
			if (kstrtoint(peq+1, 0, &temp)) {
				bb_log_dev(&(mma8453q_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "ksrtol (register address) failed!");
			}
			val = temp;
			if (strncmp(buf, "reg", 3) == 0 && buf[3] != '=') {
				if (kstrtoint(buf+3, 0, &temp)) {
					bb_log_dev(&(mma8453q_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "kstrtol (register value) failed!");
				}
				regnum = temp;
				if ((regnum < num_regs) && (mma8453q_client != NULL)) {
					error = i2c_smbus_write_byte_data(mma8453q_client, regnum, val);
					if (error) {
						bb_log_dev(&(mma8453q_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "register write failed with %d", error);
						return error;
					}
				}
			}
		}
		result = count;
	}

	return result;
}

static int mma8453q_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, mma8453q_proc_read, PDE_DATA(inode));
}

static const struct file_operations mma8453q_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= mma8453q_proc_open,
	.write		= mma8453q_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define MMA_PROCFS_FILE "driver/mma8453q"

static int mma8453q_proc_init(void)
{
	struct proc_dir_entry *proc;

	proc = proc_create_data(MMA_PROCFS_FILE, 0666, NULL, &mma8453q_proc_fops, NULL);
	if (!proc) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Couldn't create /proc/driver/mma8453q");
		return -EIO;
	}

	return 0;
}

static void mma8453q_proc_remove(void)
{
	remove_proc_entry(MMA_PROCFS_FILE, NULL);
}

static int mma8453q_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id)
{
	s32 ret;
	char value = 0;
	mma8453q_client = i2c_client;

	ret = i2c_smbus_read_byte_data(mma8453q_client, MMA8453Q_WHO_AM_I);
	if (ret < 0) {
		bb_log_dev(&(mma8453q_client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Error reading ID register");
		return -EIO;
	} else {
		value = ret;
		if (value != MMA8453Q_WHO_AM_I_ID) {
			bb_log_dev(&(mma8453q_client->dev), BB_MOD_SENSORS, BB_LVL_ERR, "Invalid id %x, expected %x", value, MMA8453Q_WHO_AM_I_ID);
			return -EINVAL;
		}

		value = 0;
		i2c_smbus_write_byte_data(mma8453q_client, MMA8453Q_CTRL_REG1, value);
		value = (4 << MMA8453Q_CTRL_REG1_DR_SHFT) & MMA8453Q_CTRL_REG1_DR_MASK;
		i2c_smbus_write_byte_data(mma8453q_client, MMA8453Q_CTRL_REG1, value);
		value |= MMA8453Q_CTRL_REG1_ACTIVE;
		i2c_smbus_write_byte_data(mma8453q_client, MMA8453Q_CTRL_REG1, value);
		bb_log_dev(&(mma8453q_client->dev), BB_MOD_SENSORS, BB_LVL_INFO, "registered");
	}
	mma8453q_device_ready = 1;
	return 0;
}

int mma8453q_i2c_remove(struct i2c_client *i2c_client)
{
	return 0;
}

static const struct i2c_device_id mma8453q_i2c_id[] = {
	{ "mma8453q", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, mma8453q_i2c_id);

static struct of_device_id mma8453q_ids[] = {
	{ .compatible = "fsl,mma8453q" },
	{ }
};

static struct i2c_driver mma8453q_i2c_driver = {
	.driver = {
		.name		= "mma8453q",
		.owner		= THIS_MODULE,
		.of_match_table	= mma8453q_ids,
	},
	.id_table	= mma8453q_i2c_id,
	.probe		= mma8453q_i2c_probe,
	.remove		= mma8453q_i2c_remove,
};

void mma8453q_init(void)
{
	int error;
	error = i2c_add_driver(&mma8453q_i2c_driver);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "mma8453q: i2c_add_driver failed with %d", error);
		return;
	}
	mma8453q_proc_init();
}

void mma8453q_exit(void)
{
	mma8453q_device_ready = 0;
	mma8453q_proc_remove();
	i2c_del_driver(&mma8453q_i2c_driver);
}
