/*
 * Copyright (c) 2021 Sonos Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */


#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/debugfs.h>
#include <linux/delay.h>
#include "blackbox.h"
#include "sensors_dev.h"
#include "sensors_hal.h"
#include "mdp.h"

#include "event_queue_api.h"

#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif

#include "ltr311_als_defs.h"
#define LTR311_INT32_MAX 0x7FFFFFFF

static inline struct i2c_client *get_i2c_client(void);

static int32_t ltr311_i2c_write_reg(struct i2c_client *client, uint8_t reg, uint8_t value);
static int32_t ltr311_i2c_read_reg(struct i2c_client *client, uint8_t reg);

static int32_t ltr311_get_lux_val(struct i2c_client *client);
static int ltr311_get_light_cond(struct i2c_client *client, enum LIGHT_COND *step);

static int sim_mode_set(void *data, u64 val);
static int sim_mode_get(void *data, u64 *val);
static int sim_lux_set(void *data, u64 val);
static int sim_lux_get(void *data, u64 *val);
static int disable_events_set(void *data, u64 val);
static int disable_events_get(void *data, u64 *val);
static void debugfs_create_common(struct i2c_client *client);

static int ltr311_poll_thread(void *i2c_client);

static int ltr311_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id);
static int ltr311_i2c_remove(struct i2c_client *i2c_client);

struct ltr311_drv {
	struct dentry *debugfs;
	int driver_init;
	bool sim_mode;
	int32_t sim_result;
	int32_t disable_events;
	struct i2c_client *client;
	struct task_struct *task;
};

extern struct manufacturing_data_page2 sys_mdp2;

static struct i2c_client *client_list[LTR311_MAX_CLIENTS];

static int32_t comp_coeff = LTR311_COMP_COEFF_DEFAULT;
static int32_t integration_factor = 1;

const struct sensors_als_light_lims ltr311_light_cond_tbl[] = {
		[LGT_PITCH_BLACK]		= {.min = 0,		.max = 999},
		[LGT_VERY_DARK]			= {.min = 1000,		.max = 4999},
		[LGT_DARK_INDOORS]		= {.min = 5000,		.max = 19999},
		[LGT_DIM_INDOORS]		= {.min = 20000,	.max = 39999},
		[LGT_NORMAL_INDOORS]		= {.min = 40000,	.max = 99999},
		[LGT_BRIGHT_INDOORS]		= {.min = 100000,	.max = 499999},
		[LGT_DIM_OUTDOORS]		= {.min = 500000,	.max = 999999},
		[LGT_CLOUDY_OUTDOORS]		= {.min = 1000000,	.max = 2999999},
		[LGT_DIRECT_SUNLIGHT]		= {.min = 3000000,	.max = 999999999},
};

static const struct i2c_device_id ltr311_i2c_id[] = {
	{LTR311_DRIVER_NAME, 0},
	{  }
};
MODULE_DEVICE_TABLE(i2c, ltr311_i2c_id);

static struct of_device_id ltr311_ids[] = {
	{ .compatible = "ltr311" },
	{  }
};

static struct i2c_driver ltr311_i2c_driver = {
	.driver = {
		.name = LTR311_DRIVER_NAME,
		.owner = THIS_MODULE,
		.of_match_table = ltr311_ids,
	},
	.id_table = ltr311_i2c_id,
	.probe = ltr311_i2c_probe,
	.remove = ltr311_i2c_remove,
};

DEFINE_SIMPLE_ATTRIBUTE(sim_mode_fops, sim_mode_get,
			sim_mode_set, "%llu\n");

DEFINE_SIMPLE_ATTRIBUTE(sim_lux_fops, sim_lux_get,
			sim_lux_set, "%llu\n");

DEFINE_SIMPLE_ATTRIBUTE(disable_events_fops, disable_events_get,
			disable_events_set, "%llu\n");



static inline struct i2c_client *get_i2c_client(void)
{
	return client_list[0];
}

static int32_t ltr311_i2c_write_reg(struct i2c_client *client, uint8_t reg, uint8_t value)
{
	int32_t ret;
	ret = i2c_smbus_write_byte_data(client, reg, value);
	return ret;
}

static int32_t ltr311_i2c_read_reg(struct i2c_client *client, uint8_t reg)
{
	int32_t ret;
	ret = i2c_smbus_read_byte_data(client, reg);
	if (ret < 0) {
		return ret;
	}

	return ret;
}

static int32_t ltr311_get_lux_val(struct i2c_client *client)
{
	struct ltr311_drv *drv;

	int32_t res;
	int32_t als_val, ir_val;

	int64_t num, denom;
	int64_t lux;

	if (!client) {
		return -EINVAL;
	}

	drv = i2c_get_clientdata(client);
	if (!drv) {
		return -EINVAL;
	}

	if (drv->sim_mode) {
		return drv->sim_result;
	}

	res = ltr311_i2c_read_reg(client, LTR311_R_ALS_DATA_LSB);
	if (res < 0) {
		bb_log_dev(&client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "invalid result on register %x read: %x", LTR311_R_ALS_DATA_LSB, res);
		return res;
	}
	als_val = res;

	res = ltr311_i2c_read_reg(client, LTR311_R_ALS_DATA_MSB);
	if (res < 0) {
		bb_log_dev(&client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "invalid result on register %x read: %x", LTR311_R_ALS_DATA_MSB, res);
		return res;
	}
	als_val |= (res << 8);

	res = ltr311_i2c_read_reg(client, LTR311_R_ALS_IR_DATA_LSB);
	if (res < 0) {
		bb_log_dev(&client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "invalid result on register %x read: %x", LTR311_R_ALS_IR_DATA_LSB, res);
		return res;
	}
	ir_val = res;

	res = ltr311_i2c_read_reg(client, LTR311_R_ALS_IR_DATA_MSB);
	if (res < 0) {
		bb_log_dev(&client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "invalid result on register %x read: %x", LTR311_R_ALS_IR_DATA_MSB, res);
		return res;
	}
	ir_val |= (res << 8);

	num = ((LTR311_LUX_CALC_ALS_VAL_COEFF*als_val) - (LTR311_LUX_CALC_IR_VAL_COEFF*ir_val*ir_val/als_val));
	denom = (1 << LTR311_ALS_GAIN_FACTOR) * integration_factor;

	lux = ((num * comp_coeff) / denom) / 100;

	if (lux > LTR311_INT32_MAX) {
		return LTR311_INT32_MAX;
	} else {
		return (int32_t)lux;
	}
}

static int ltr311_get_light_cond(struct i2c_client *client, enum LIGHT_COND *step)
{
	int32_t result;
	int i;

	result = ltr311_get_lux_val(client);
	if (result < 0) {
		return -EFAULT;
	}

	for (i=0; i < LGT_NUM_CONDS; i++) {
		if (result >= ltr311_light_cond_tbl[i].min &&
			result <= ltr311_light_cond_tbl[i].max)
		{
			*step = i;
			return 0;
		}
	}

	return -1;
}

static int sim_mode_set(void *data, u64 val)
{
	struct ltr311_drv *drv = data;

	if (val != 0 && val != 1) {
		return -EINVAL;
	}

	if (!drv) {
		return -EFAULT;
	}

	drv->sim_mode = val;

	return 0;
}

static int sim_mode_get(void *data, u64 *val)
{
	struct ltr311_drv *drv = data;

	if (!drv) {
		return -EFAULT;
	}

	*val = drv->sim_mode;

	return 0;
}

static int sim_lux_set(void *data, u64 val)
{
	struct ltr311_drv *drv = data;
	int32_t sval = val;

	if (sval < 0 || sval > 9999999) {
		return -EINVAL;
	}

	if (!drv) {
		return -EFAULT;
	}

	drv->sim_result = sval;

	return 0;

}

static int sim_lux_get(void *data, u64 *val)
{
	struct ltr311_drv *drv = data;

	if (!drv) {
		return -EFAULT;
	}

	*val = (u64) drv->sim_result;

	return 0;
}

static int disable_events_set(void *data, u64 val)
{
	struct ltr311_drv *drv = data;
		int32_t sval = val;

	if (!drv) {
				return -EFAULT;
		}

		if (sval !=  0 && sval != 1) {
				return -EINVAL;
		}

		drv->disable_events = sval;

		return 0;

}

static int disable_events_get(void *data, u64 *val)
{
	struct ltr311_drv *drv = data;

	if (!drv) {
				return -EFAULT;
		}

	*val = drv->disable_events;
		return 0;
}

static void debugfs_create_common(struct i2c_client *client)
{
	struct ltr311_drv *drv = i2c_get_clientdata(client);

	if (drv) {
		drv->debugfs = debugfs_create_dir(LTR311_DRIVER_NAME, NULL);

		debugfs_create_file("sim_mode", 0444, drv->debugfs, drv,
				&sim_mode_fops);

		debugfs_create_file("sim_lux", 0444, drv->debugfs, drv,
				&sim_lux_fops);

		debugfs_create_file("disable_events", 0444, drv->debugfs,
								&drv->disable_events, &disable_events_fops);
	}
}

static int ltr311_poll_thread(void *i2c_client)
{
	struct ltr311_drv *drv = i2c_get_clientdata(i2c_client);

	while (!kthread_should_stop()) {
		msleep_interruptible(LTR311_POLLING_DELAY_MS);

		if (!drv->disable_events) {
			event_queue_send_event(HWEVTQSOURCE_LIGHT_SENSOR, HWEVTQINFO_UPDATE);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(HWEVTQSOURCE_LIGHT_SENSOR, HWEVTQINFO_UPDATE);
#endif
		}
	}
	return 0;
}

static int ltr311_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id)
{
	int32_t ret;
	struct ltr311_drv *drv_data;

	drv_data = devm_kzalloc(&i2c_client->dev, sizeof(*drv_data), GFP_KERNEL);
	if (!drv_data) {
		return -ENOMEM;
	}

	drv_data->disable_events = 0;
	drv_data->driver_init = 0;

	ret = ltr311_i2c_read_reg(i2c_client, LTR311_R_MANUFAC_ID);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "Error reading manufacturer ID.");
		return -EIO;
	}
	if (ret != LTR311_R_MANUFAC_ID_EXPECTED) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "Invalid manufacturer ID. Expected %d, received %d.", LTR311_R_MANUFAC_ID_EXPECTED, ret);
		return -EINVAL;
	}

	ret = ltr311_i2c_read_reg(i2c_client, LTR311_R_PART_ID);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "Error reading part ID.");
		return -EIO;
	}
	if (ret != LTR311_R_PART_ID_EXPECTED) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "Invalid part ID. Expected %d, received %d.", LTR311_R_PART_ID_EXPECTED, ret);
		return -EINVAL;
	}

	ret = ltr311_i2c_write_reg(i2c_client, LTR311_R_RESET, 0x1);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "failed to reset device");
		return -EIO;
	}

	ret = ltr311_i2c_write_reg(i2c_client, LTR311_R_ALS_AVERAGING, 0x00);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "failed configuration");
		return -EIO;
	}
	ret = ltr311_i2c_write_reg(i2c_client, LTR311_R_ALS_TIME_SCALE, 0xE0 | ((LTR311_ALS_INT_TIME_SCALE & 0x3) << 2) | (LTR311_ALS_MEAS_RATE_SCALE & 0x3));
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "failed configuration");
		return -EIO;
	}
	ret = ltr311_i2c_write_reg(i2c_client, LTR311_R_ALS_INT_TIME_STEPS, LTR311_ALS_INT_TIME_STEP);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "failed configuration");
		return -EIO;
	}
	ret = ltr311_i2c_write_reg(i2c_client, LTR311_R_ALS_MRR_STEPS, LTR311_ALS_MEAS_TIME_STEP);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "failed configuration");
		return -EIO;
	}
	ret = ltr311_i2c_write_reg(i2c_client, LTR311_R_IR_ENABLE, 0x08);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "failed configuration");
		return -EIO;
	}
	ret = ltr311_i2c_write_reg(i2c_client, LTR311_R_PD_ENABLE, 0x4C);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "failed configuration");
		return -EIO;
	}
	ret = ltr311_i2c_write_reg(i2c_client, LTR311_R_INTERRUPT, 0x00);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "failed configuration");
		return -EIO;
	}

	if ((LTR311_ALS_INT_TIME_SCALE & 0x3) == 0x0) {
		integration_factor = LTR311_ALS_INT_TIME_00*(LTR311_ALS_INT_TIME_STEP+1)/100;
	}
	else if ((LTR311_ALS_INT_TIME_SCALE & 0x3) == 0x1) {
		integration_factor = LTR311_ALS_INT_TIME_01*(LTR311_ALS_INT_TIME_STEP+1)/100;
	}
	else if ((LTR311_ALS_INT_TIME_SCALE & 0x2) == 0x2) {
		integration_factor = LTR311_ALS_INT_TIME_1X*(LTR311_ALS_INT_TIME_STEP+1)/100;
	}
	else {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "integration period settings are invalid.");
		return -EINVAL;
	}


	ret = ltr311_i2c_write_reg(i2c_client, LTR311_R_ALS_CONTR, 0x40 | (LTR311_ALS_GAIN_FACTOR << 2) | 0x01);
	if (ret < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "failed to start device");
		return -EIO;
	}

	client_list[0] = i2c_client;
	drv_data->client = i2c_client;
	i2c_set_clientdata(i2c_client, (void *) drv_data);
	debugfs_create_common(i2c_client);

	drv_data->task = kthread_run(ltr311_poll_thread, (void *) i2c_client, "ltr311_poll");
	if (IS_ERR(drv_data->task)) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_ERR, "unable to start kthread.");
		return -EIO;
	}

	if (sys_mdp2.mdp2_sigdata.variant == MDP_VARIANT_WHITE) {
		of_property_read_u32(i2c_client->dev.of_node, "ccoef-wht", &comp_coeff);
	} else if (sys_mdp2.mdp2_sigdata.variant == MDP_VARIANT_BLACK) {
		of_property_read_u32(i2c_client->dev.of_node, "ccoef-blk", &comp_coeff);
	} else if (sys_mdp2.mdp2_sigdata.variant == MDP_VARIANT_SHADOW) {
		of_property_read_u32(i2c_client->dev.of_node, "ccoef-blk", &comp_coeff);
	}

	if (comp_coeff != LTR311_COMP_COEFF_DEFAULT) {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_INFO, "assigned optical compensation coefficient: %d", comp_coeff);
	} else {
		bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_INFO, "assigned default optical compensation coefficient: %d", comp_coeff);
	}

	drv_data->driver_init = 1;
	bb_log_dev(&i2c_client->dev, BB_MOD_SENSORS, BB_LVL_INFO, "probe complete!");
	return 0;
}

static int ltr311_i2c_remove(struct i2c_client *i2c_client)
{
	struct ltr311_drv *drv = i2c_get_clientdata(i2c_client);
	if (drv) {
		debugfs_remove_recursive(drv->debugfs);
		kthread_stop(drv->task);
	}

	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "ltr311: removed!");
	return 0;
}


bool ltr311_driver_ready(void)
{
	struct i2c_client *client;
	struct ltr311_drv *drv;

	client = get_i2c_client();
    if (!client) {
        return false;
    }

	drv = i2c_get_clientdata(client);
    if (!drv) {
        return false;
    }

	return (drv->driver_init == 1);
}

void ltr311_init(void)
{
	int error;
	error = i2c_add_driver(&ltr311_i2c_driver);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "ltr311: i2c_add_driver failed with %d", error);
		return;
	}
	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "ltr311: loaded driver!");
}

void ltr311_exit(void)
{
	i2c_del_driver(&ltr311_i2c_driver);
}

int ltr311_handle_ioctl(enum sensors_als_ioctl_cmd cmd, void *result)
{
	struct i2c_client *client = get_i2c_client();

	if (client == NULL || result == NULL) {
		return -1;
	}

	if (cmd == SENSORS_ALS_IOCTL_GET_LUX_VAL) {
		int32_t *lux_value = result;
		*lux_value = ltr311_get_lux_val(client);
		return (*lux_value < 0) ? *lux_value : 0;
	} else if (cmd == SENSORS_ALS_IOCTL_GET_LIGHT_COND) {
		int ret;
		enum LIGHT_COND *light_cond = result;
		ret = ltr311_get_light_cond(client, light_cond);
		return ret;
	}
	return -EINVAL;
}
