/*
 * Copyright (c) 2019-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/jiffies.h>
#include <asm/uaccess.h>
#include "blackbox.h"
#include "sensors_hal.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "sensors_dev.h"
#include "blackbox.h"

#define LIS2HH12_WHO_AM_I	0x0f
#define LIS2HH12_CTRL_1		0x20
#define LIS2HH12_CTRL_3		0x22
#define LIS2HH12_CTRL_4		0x23
#define LIS2HH12_STATUS		0x27
#define LIS2HH12_OUT_X_L	0x28
#define LIS2HH12_OUT_X_H	0x29
#define LIS2HH12_OUT_Y_L	0x2a
#define LIS2HH12_OUT_Y_H	0x2b
#define LIS2HH12_OUT_Z_L	0x2c
#define LIS2HH12_OUT_Z_H	0x2d

#define LIS2HH12_WHO_AM_I_ID	0x41
#define LIS2HH12_DATA_AVAILABLE	0x7
#define LIS2HH12_ODR_ENABLE(O)	((O << 4) | (1 << 3) | 0x7)
#define LIS2HH12_CLEAR_STATUS	0

struct i2c_client *lis2hh12_client = NULL;
int lis2hh12_device_ready = 0;
static int output_data_rate = 3;
static int print_timing = 0;
static int print_data = 0;
static int orientation_mode = 0;
static int motion_detection_mode = 0;
static int music_impacts_y_axis = 0;
static int music_impacts_z_axis = 0;
static int reinit_on_resume = 0;
static enum HWEVTQ_EventInfo last_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
static enum HWEVTQ_EventInfo current_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
static enum HWEVTQ_EventInfo hwevtq_last_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
static enum HWEVTQ_EventInfo hwevtq_current_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
#endif
typedef int (*lis2hh12_callback)(void);
void lis2hh12_device_init(void);

#define LIS2HH12_POLL_TIME	250
#define MOTION_THRESHOLD	70
#define SETTLE_THRESHOLD	26

#define H_TO_L		200
#define L_TO_H		50
#define H_TO_R		-200
#define R_TO_H		-50

struct delayed_work poll_work;
static void lis2hh12_poll(struct work_struct *work);

static motion_device_t motion_device;
static int last_read = 0;
static int polling_iterations = 0;
static int16_t x_total = 0, x_ma = 0;
static int16_t y_total = 0, y_ma = 0;
static int16_t z_total = 0, z_ma = 0;
static int diff[LIS2HH12_FIFO_DEPTH];
static int diff_total = 0;
static int moving = 0;
static bool orientation_cmd = false;

static inline unsigned int absval8(int8_t v)
{
        unsigned int uv = v;
        if (v < 0) {
                uv = -v;
        }
        return uv;
}
static inline unsigned int absval16(int16_t v)
{
        unsigned int uv = v;
        if (v < 0) {
                uv = -v;
        }
        return uv;
}
static inline unsigned int absval32(int32_t v)
{
        unsigned int uv = v;
        if (v < 0) {
                uv = -v;
        }
        return uv;
}

int lis2hh12_get_temp(int i1, int *lis_temp)
{
	int16_t current_temp;
	(void) i1;

	current_temp = (i2c_smbus_read_byte_data(lis2hh12_client, 0x0c) << 8) | (i2c_smbus_read_byte_data(lis2hh12_client, 0x0b));
	current_temp >>= 5;
	{
		int negative = current_temp & 0x400;
		current_temp >>= 3;
		if (negative) current_temp |= 0x8000;
		current_temp += 25;
	}

	*lis_temp = current_temp;

	return (0);
}

int lis2hh12_get_raw_accel(struct accel_data *ad)
{
	int reg;
	char raw_data[7];
	s32 ret;

	if (lis2hh12_client == NULL) {
		return -EINVAL;
	}

	ret = i2c_smbus_read_byte_data(lis2hh12_client, LIS2HH12_STATUS);
	if (ret < 0) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12: error 0x%x reading status\n", ret);
	} else {
		if (!(ret & LIS2HH12_DATA_AVAILABLE)) {
			return 0;
		}
	}

	for (reg = LIS2HH12_OUT_X_L; reg <= LIS2HH12_OUT_Z_H; reg++) {
		ret = i2c_smbus_read_byte_data(lis2hh12_client, reg);
		if (ret < 0) {
#if defined(ACCELEROMETER_DEBUG)
			if (print_data) printk(KERN_ERR "%s: -EIO\n", __func__);
#endif
			return -EIO;
		}
		raw_data[reg-LIS2HH12_OUT_X_L] = ret;
		schedule();
	}

	memset(ad, 0, sizeof(struct accel_data));
	ad->x = raw_data[1] << 8 | raw_data[0];
	ad->x /= 64;
	ad->y = raw_data[3] << 8 | raw_data[2];
	ad->y /= 64;
	ad->z = raw_data[5] << 8 | raw_data[4];
	ad->z /= 64;

	ret = i2c_smbus_write_byte_data(lis2hh12_client, LIS2HH12_STATUS, LIS2HH12_CLEAR_STATUS);

	return 0;
}

static int lis2hh12_get_data(motion_device_t *md)
{
	int i;
	int next_read = last_read+1;

	for (i = 0; i < LIS2HH12_FIFO_DEPTH;i++) {
		if (next_read >= LIS2HH12_FIFO_DEPTH) {
			next_read = 0;
		}
		md->x_val[i] = motion_device.x_val[next_read];
		md->y_val[i] = motion_device.y_val[next_read];
		md->z_val[i] = motion_device.z_val[next_read];
		next_read++;
	}
	return 0;
}

int lis2hh12_handle_ioctl(motion_device_t *md)
{

	int error;

	if (md->op_flag & MOTION_REG_WRITE_FLAG) {
		error = i2c_smbus_write_byte_data(lis2hh12_client, md->cfg_reg, md->cfg_data);
		if (error) {
			bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "lis2hh12: attempt to write register 0x%x failed [%d]", md->cfg_reg, error);
		}
	}

	if (md->op_flag & MOTION_REG_READ_FLAG) {
		error = i2c_smbus_read_byte_data(lis2hh12_client, md->cfg_reg);
		if (error < 0) {
			bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "lis2hh12: attempt to read register 0x%x failed [%d]", md->cfg_reg, error);
		} else {
			md->cfg_data = error;
		}
	}

	if (md->op_flag & MOTION_DATA_FETCH) {
		lis2hh12_get_data(md);
	}

	return 0;
}

static int lis2hh12_proc_read(struct seq_file *m, void *v)
{
	int reg;
	int current_temp;
	uint8_t val[0x40] = {0};
	struct accel_data ad;

	if (!lis2hh12_device_ready) {
		seq_printf(m, "lis2hh12 device not ready\n");
		return 0;
	}
	if (lis2hh12_client == NULL) {
		seq_printf(m, "LIS2HH12: I2C lis2hh12_client not initialized.\n");
		return 0;
	}
	for (reg = 0; reg < 0x3f; reg++) {
		s32 ret = i2c_smbus_read_byte_data(lis2hh12_client, reg);
		if (ret < 0) {
			int r;
			for (r = reg; r < 0x3f; r++) {
				val[r] = 0;
			}
			seq_printf(m, "LIS2HH12: i2c read error on reg %02x\n", reg);
			break;
		}
		val[reg] = ret;
	}
	for (reg = 0; reg < 0x3f; reg++) {
		if (reg && ((reg%16) == 0))
			seq_printf(m, "\n");
		if (((reg%16) == 0))
			seq_printf(m, "%2x: ", reg);
		if (reg && ((reg%4) == 0) &&  ((reg%16) != 0))
			seq_printf(m, "  ");
		seq_printf(m, "%02x ", val[reg]);
	}
	seq_printf(m, "\n");

	if (lis2hh12_get_raw_accel(&ad) < 0) {
		seq_printf(m, "error reading acceleration\n");
	}
	else {
		seq_printf(m, "x=%4d y=%4d z=%4d\n", ad.x, ad.y, ad.z);
	}

	reg = lis2hh12_get_temp(reg, &current_temp);

	seq_printf(m, "Current temp:  %d C\n", current_temp);

	if (orientation_mode) {
		if (orientation_cmd) {
			seq_printf(m, "Current orientation:  %s (Simulation mode, actual %s)\n", (last_orientation == HWEVTQINFO_ORIENT_HORIZONTAL) ? "horizontal" : \
							  (last_orientation == HWEVTQINFO_ORIENT_VERTICAL_RIGHT) ? "right" : "left", \
							  (current_orientation == HWEVTQINFO_ORIENT_HORIZONTAL) ? "horizontal" : \
							  (current_orientation == HWEVTQINFO_ORIENT_VERTICAL_RIGHT) ? "right" : "left");
		} else {
			seq_printf(m, "Current orientation:  %s\n", (last_orientation == HWEVTQINFO_ORIENT_HORIZONTAL) ? "horizontal" : \
							  (last_orientation == HWEVTQINFO_ORIENT_VERTICAL_RIGHT ? "right" : "left"));
		}

	}

	return 0;
}

static ssize_t lis2hh12_proc_write(struct file *file, const char __user * buffer,
			size_t count, loff_t *data)
{
	int num_regs = 0x31;
	char buf[200];
	char *peq;
	int result = 0, error;
	int temp;
	u32  regnum;
	uint8_t  val;

	if (!lis2hh12_device_ready) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "lis2hh12 device not ready");
		result = -EIO;
	} else if (count >= sizeof(buf)) {
		result = -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;

	} else if (strncmp(buf, "orientation-reporting", strlen("orientation-reporting")) == 0) {
		peq = strchr(buf, '=') + 1;
		if (strncmp(peq, "detect", strlen("detect")) == 0) {
			orientation_cmd = false;
		} else if (strncmp(peq, "horizontal", strlen("horizontal")) == 0) {
			orientation_cmd = true;
			last_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_last_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
#endif
		} else if (strncmp(peq, "right", strlen("right")) == 0) {
			orientation_cmd = true;
			last_orientation = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                        hwevtq_last_orientation = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
#endif
		} else if (strncmp(peq, "left", strlen("left")) == 0) {
			orientation_cmd = true;
			last_orientation = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                        hwevtq_last_orientation = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
#endif
		}
		if (orientation_cmd) {
			event_queue_send_event(HWEVTQSOURCE_ORIENTATION, last_orientation);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(HWEVTQSOURCE_ORIENTATION, hwevtq_last_orientation);
#endif
		}
		result = count;

	} else if (strncmp(buf, "detect", strlen("detect")) == 0) {
		orientation_cmd = false;
		result = count;
	} else if (strncmp(buf, "horizontal", strlen("horizontal")) == 0) {
		orientation_cmd = true;
		last_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
                event_queue_send_event(HWEVTQSOURCE_ORIENTATION, last_orientation);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_last_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
		hwevtq_send_event(HWEVTQSOURCE_ORIENTATION, hwevtq_last_orientation);
#endif
		result = count;
	} else if (strncmp(buf, "right", strlen("right")) == 0) {
		orientation_cmd = true;
		last_orientation = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
		event_queue_send_event(HWEVTQSOURCE_ORIENTATION, last_orientation);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_last_orientation = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
		hwevtq_send_event(HWEVTQSOURCE_ORIENTATION, hwevtq_last_orientation);
#endif
		result = count;
	} else if (strncmp(buf, "left", strlen("left")) == 0) {
		orientation_cmd = true;
		last_orientation = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
		event_queue_send_event(HWEVTQSOURCE_ORIENTATION, last_orientation);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_last_orientation = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
		hwevtq_send_event(HWEVTQSOURCE_ORIENTATION, hwevtq_last_orientation);
#endif
		result = count;
	} else if (strncmp(buf, "time", strlen("time")) == 0) {
		print_timing = 20;
		result = count;
	} else if (strncmp(buf, "data", strlen("data")) == 0) {
		print_data = 50;
		result = count;
	} else {
		buf[count] = '\0';
		peq = strchr(buf, '=');
		if (peq != NULL) {
			*peq = 0;
			if (kstrtoint(peq+1, 0, &temp)) {
				bb_log_dev(&(lis2hh12_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "ksrtol (register address) failed!");
			}
			val = temp;
			if (strncmp(buf, "reg", 3) == 0 && buf[3] != '=') {
				if (kstrtoint(buf+3, 0, &temp)) {
					bb_log_dev(&(lis2hh12_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "kstrtol (register value) failed!");
				}
				regnum = temp;
				if ((regnum < num_regs) && (lis2hh12_client != NULL)) {
					error = i2c_smbus_write_byte_data(lis2hh12_client, regnum, val);
					if (error) {
						bb_log_dev(&(lis2hh12_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "register write failed with %d", error);
						return error;
					}
				}
			}
		}
		result = count;
	}

	return result;
}

enum HWEVTQ_EventInfo lis2hh12_get_current(void)
{
	if (orientation_mode) {
		return last_orientation;
	} else {
		return HWEVTQINFO_ORIENT_HORIZONTAL;
	}
}


static int lis2hh12_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, lis2hh12_proc_read, PDE_DATA(inode));
}

static const struct file_operations lis2hh12_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= lis2hh12_proc_open,
	.write		= lis2hh12_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define ACCELEROMETER_GENERIC_NAME "driver/accel"
#define LIS2HH12_PROCFS_FILE "driver/lis2hh12"
#define LIS2HH12_ACCEL_LINK "/proc/driver/lis2hh12"

static int lis2hh12_proc_init(void)
{
	struct proc_dir_entry *proc;

	proc = proc_create_data(LIS2HH12_PROCFS_FILE, 0666, NULL, &lis2hh12_proc_fops, NULL);
	if (!proc) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Couldn't create /proc/driver/lis2hh12");
		return -EIO;
	}

	proc_symlink(ACCELEROMETER_GENERIC_NAME, NULL, LIS2HH12_ACCEL_LINK);
	return 0;
}

static void lis2hh12_proc_remove(void)
{
	remove_proc_entry(ACCELEROMETER_GENERIC_NAME, NULL);
	remove_proc_entry(LIS2HH12_PROCFS_FILE, NULL);
}



static void lis2hh12_new_values(struct accel_data *ad, int next)
{
	int last_x_ma = x_ma;
	int last_y_ma = y_ma;
	int last_z_ma = z_ma;

	x_total = x_total - absval16(motion_device.x_val[next]%1024);
	x_total = x_total + absval16(ad->x)%1024;
	motion_device.x_val[next] = ad->x;
	x_ma = x_total / LIS2HH12_FIFO_DEPTH;
	y_total = y_total - absval16(motion_device.y_val[next]%1024);
	y_total = y_total + absval16(ad->y)%1024;
	motion_device.y_val[next] = ad->y;
	y_ma = y_total / LIS2HH12_FIFO_DEPTH;
	z_total = z_total - absval16(motion_device.z_val[next]%1024);
	z_total = z_total + absval16(ad->z)%1024;
	motion_device.z_val[next] = ad->z;
	z_ma = z_total / LIS2HH12_FIFO_DEPTH;

	diff_total = diff_total - diff[next];
	if (music_impacts_z_axis) {
		diff[next] = absval16(last_x_ma - x_ma) + absval16(last_y_ma - y_ma);
	} else {
		diff[next] = absval16(last_x_ma - x_ma) + absval16(last_z_ma - z_ma);
	}
	diff_total = diff_total + diff[next];

#if defined(ACCELEROMETER_DEBUG)
	if (print_data) {
		printk(KERN_ERR "diff - %d, diff_total = %d  >> x_ma/last = %d/%d, y_ma/last = %d/%d, z_ma/last = %d/%d\n", \
				 diff[next], diff_total, x_ma, last_x_ma, y_ma, last_y_ma, z_ma, last_z_ma);
	}
#endif

	if (orientation_mode) {
		enum HWEVTQ_EventInfo new_orientation = HWEVTQINFO_NO_EVENT;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		enum HWEVTQ_EventInfo hwevtq_new_orientation = HWEVTQINFO_NO_EVENT;
#endif
		switch(last_orientation) {
			case HWEVTQINFO_ORIENT_HORIZONTAL:
				if (ad->y > H_TO_L) {
					current_orientation = new_orientation = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
					hwevtq_current_orientation = hwevtq_new_orientation = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
#endif
				} else if (ad->y < H_TO_R) {
					current_orientation = new_orientation = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                        hwevtq_current_orientation = hwevtq_new_orientation = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
#endif
				} else {
					current_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
					hwevtq_current_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
#endif
				}
				break;
			case HWEVTQINFO_ORIENT_VERTICAL_LEFT:
				if (ad->y < H_TO_R) {
					current_orientation = new_orientation = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                        hwevtq_current_orientation = hwevtq_new_orientation = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
#endif
				} else if (ad->y < L_TO_H) {
					current_orientation = new_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                        hwevtq_current_orientation = hwevtq_new_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
#endif
				} else {
					current_orientation = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
					hwevtq_current_orientation = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
#endif
				}
				break;
			case HWEVTQINFO_ORIENT_VERTICAL_RIGHT:
				if (ad->y > H_TO_L) {
					current_orientation = new_orientation = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                        hwevtq_current_orientation = hwevtq_new_orientation = HWEVTQINFO_ORIENT_VERTICAL_LEFT;
#endif
				} else if (ad->y > R_TO_H) {
					current_orientation = new_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                        hwevtq_current_orientation = hwevtq_new_orientation = HWEVTQINFO_ORIENT_HORIZONTAL;
#endif
				} else {
					current_orientation = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
					hwevtq_current_orientation = HWEVTQINFO_ORIENT_VERTICAL_RIGHT;
#endif
				}
				break;
			default:
				break;
		}
		if (new_orientation != HWEVTQINFO_NO_EVENT && diff_total < SETTLE_THRESHOLD && !orientation_cmd) {
			event_queue_send_event(HWEVTQSOURCE_ORIENTATION, new_orientation);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(HWEVTQSOURCE_ORIENTATION, hwevtq_new_orientation);
#endif
			bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "Orientation changed from %s to %s.", \
					(last_orientation == HWEVTQINFO_ORIENT_HORIZONTAL) ? "horizontal" : \
					(last_orientation == HWEVTQINFO_ORIENT_VERTICAL_LEFT) ? "left" : "right", \
					(new_orientation == HWEVTQINFO_ORIENT_HORIZONTAL) ? "horizontal" : \
					(new_orientation == HWEVTQINFO_ORIENT_VERTICAL_LEFT) ? "left" : "right");
			last_orientation = new_orientation;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_last_orientation = hwevtq_new_orientation;
#endif
		}
	}
	if (motion_detection_mode) {
		if (diff_total > MOTION_THRESHOLD) {
			if (!moving) {
				moving = 1;
				event_queue_send_event(HWEVTQSOURCE_MOTION_DETECTOR, HWEVTQINFO_MOTION_DETECTED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event(HWEVTQSOURCE_MOTION_DETECTOR,
						HWEVTQINFO_MOTION_DETECTED);
#endif
				bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12: Motion Detected\n");
			}
		} else if (diff_total < SETTLE_THRESHOLD) {
			if (moving) {
				moving = 0;
				event_queue_send_event(HWEVTQSOURCE_MOTION_DETECTOR, HWEVTQINFO_MOTION_SETTLED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event(HWEVTQSOURCE_MOTION_DETECTOR,
						HWEVTQINFO_MOTION_SETTLED);
#endif
				bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12: Motion Settled\n");
			}
		}
	}
}

static void lis2hh12_poll(struct work_struct *work)
{
	struct accel_data ad;
	int i;
	int next_read;
	u64 start;
	int count=0;

	start = get_jiffies_64();
	for(i = 0;i < LIS2HH12_FIFO_DEPTH;i++) {
		lis2hh12_get_raw_accel(&ad) ;
		count += 6;

		next_read = last_read + 1;
		if (next_read >= LIS2HH12_FIFO_DEPTH) {
			next_read = 0;
		}

		lis2hh12_new_values(&ad, next_read);

		last_read = next_read;
#if defined(ACCELEROMETER_DEBUG)
		if (print_data) {
			printk(KERN_ERR "x = %5d, y = %5d, z = %5d\n", ad.x, ad.y, ad.z);
			print_data--;
		}
#endif
	}
#if defined(ACCELEROMETER_DEBUG)
	if (print_timing) {
		printk(KERN_ERR "read %d bytes in %lld ms\n", count, (u64)jiffies_to_msecs((get_jiffies_64() - start)));
		print_timing--;
	}
#endif
	schedule_delayed_work(&poll_work, LIS2HH12_POLL_TIME);
	polling_iterations++;
}

static int lis2hh12_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id)
{
	s32 ret;
	char value = 0;
	lis2hh12_client = i2c_client;

	bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "Probe for accelerometer...");

	ret = i2c_smbus_read_byte_data(lis2hh12_client, LIS2HH12_WHO_AM_I);
	if (ret < 0) {
		return 0;
	} else {
		value = ret;
		if (value != LIS2HH12_WHO_AM_I_ID) {
			bb_log_dev(&(lis2hh12_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "Invalid id %x, expected %x", value, LIS2HH12_WHO_AM_I_ID);
			return -EINVAL;
		}
	}
	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12:  WHO_AM_I = 0x%x - driver loaded", LIS2HH12_WHO_AM_I_ID);
	lis2hh12_device_ready = 1;
	return 0;
}

int lis2hh12_i2c_remove(struct i2c_client *i2c_client)
{
	return 0;
}

static int lis2hh12_i2c_suspend(struct device *dev)
{
	return 0;
}
static int lis2hh12_i2c_resume(struct device *dev)
{
	if (reinit_on_resume) {
		lis2hh12_device_init();
	}
	return 0;
}

const struct dev_pm_ops lis2hh12_i2c_pm_ops = {
	.suspend = lis2hh12_i2c_suspend, \
	.resume = lis2hh12_i2c_resume,
};

static const struct i2c_device_id lis2hh12_i2c_id[] = {
	{ "lis2hh12", 0 },
	{ }
};
MODULE_DEVICE_TABLE(lis2hh12_i2c, lis2hh12_i2c_id);

static struct of_device_id lis2hh12_ids[] = {
	{ .compatible = "stm,lis2hh12" },
	{ }
};

static struct i2c_driver lis2hh12_i2c_driver = {
	.driver = {
		.name		= "lis2hh12",
		.owner		= THIS_MODULE,
		.pm             = &lis2hh12_i2c_pm_ops,
		.of_match_table	= lis2hh12_ids,
	},
	.id_table	= lis2hh12_i2c_id,
	.probe		= lis2hh12_i2c_probe,
	.remove		= lis2hh12_i2c_remove,
};

void lis2hh12_device_init()
{
	int error;

	error = i2c_smbus_write_byte_data(lis2hh12_client, 0x27, 0);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "lis2hh12: clearing STATUS failed with %d", error);
		return;
	} else {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "lis2hh12: cleared STATUS");
	}
	error = i2c_smbus_write_byte_data(lis2hh12_client, LIS2HH12_CTRL_1, LIS2HH12_ODR_ENABLE(output_data_rate));
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "lis2hh12: setting CTRL_1 failed with %d", error);
		return;
	} else {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "lis2hh12: set CTRL_1 to 0x%x", LIS2HH12_ODR_ENABLE(output_data_rate));
	}
	error = i2c_smbus_write_byte_data(lis2hh12_client, LIS2HH12_CTRL_3, 0xc0);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "lis2hh12: setting CTRL_3 failed with %d", error);
		return;
	} else {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "lis2hh12: set CTRL_3 to 0x%x", 0xc0);
	}
	error = i2c_smbus_write_byte_data(lis2hh12_client, LIS2HH12_CTRL_4, 0xc0);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "lis2hh12: setting CTRL_4 failed with %d", error);
		return;
	} else {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "lis2hh12: set CTRL_4 to 0x%x", 0xc0);
	}
	error = i2c_smbus_write_byte_data(lis2hh12_client, 0x2e, 0x40);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "lis2hh12: setting FIFO_CTRL failed with %d", error);
		return;
	}

	cancel_delayed_work_sync(&poll_work);
	INIT_DELAYED_WORK(&poll_work, lis2hh12_poll);
	schedule_delayed_work(&poll_work, LIS2HH12_POLL_TIME);
}

void lis2hh12_init(void)
{
	int error;
	struct device_node *dev_node;
	char *status_check, chkbuf[10];
	dev_node = of_find_compatible_node(NULL, NULL, "stm,lis2hh12");

	if (!dev_node) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "No lis2hh12 in this device");
		return;
	}

	status_check = chkbuf;
	of_property_read_string(dev_node, "status", (const char **)&status_check);
	if (strncmp(status_check, "okay", 4)) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "lis2hh12 not enabled");
		return;
	}

	error = i2c_add_driver(&lis2hh12_i2c_driver);
	if (error) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12: i2c_add_driver failed with %d", error);
		return;
	}
	if (!lis2hh12_device_ready) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12: i2c device not ready");
		i2c_del_driver(&lis2hh12_i2c_driver);
		return;
	}

	if (of_property_read_bool(dev_node, "orientation-mode")) {
		orientation_mode = 1;
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12 will be used for player orientation");
	}
	if (of_property_read_bool(dev_node, "motion-detection-mode")) {
		motion_detection_mode = 1;
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12 will be used for motion detection");
	}
	if (of_property_read_bool(dev_node, "music-impacts-z-axis")) {
		music_impacts_z_axis = 1;
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12 will ignore z axis for motion detection");
	}
	if (of_property_read_bool(dev_node, "music-impacts-y-axis")) {
		music_impacts_y_axis = 1;
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12 will ignore y axis for motion detection");
	}
	if (of_property_read_bool(dev_node, "reinit-on-resume")) {
		reinit_on_resume = 1;
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12 will re-initialize when resuming from suspend");
	}
	if (of_property_read_bool(dev_node, "supports-thermal-mgmt")) {
		if (thermal_mgmt_map_device(THERMAL_SENSOR_MOTION, "MOTION", "sonos,accelerometer", lis2hh12_get_temp)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "lis2hh12 will be used for thermal management");
		} else {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "lis2hh12 thermal mapping failed");
		}
	}

	lis2hh12_proc_init();

	lis2hh12_device_init();
}

void lis2hh12_exit(void)
{
	if (lis2hh12_device_ready) {
		lis2hh12_proc_remove();
		cancel_delayed_work_sync(&poll_work);
		i2c_del_driver(&lis2hh12_i2c_driver);
		lis2hh12_device_ready = 0;
	}
}
