/*
 * Copyright (c) 2016-2020 Sonos, Inc.
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>
#include <asm/uaccess.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/of_gpio.h>

#include "blackbox.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
static int linein_over_irq = -EINVAL;
static int linein_overvolt_gpio = -EINVAL;

irqreturn_t linein_overvolt_isr(int irq, void *data)
{
	event_queue_send_event_defer(HWEVTQSOURCE_LINEIN, HWEVTQINFO_HW_ERROR);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	hwevtq_send_event_defer(HWEVTQSOURCE_LINEIN, HWEVTQINFO_HW_ERROR);
#endif
	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "over volt IRQ\n");
	return IRQ_HANDLED;
}

static inline void linein_get_gpio(void)
{
	struct device_node *np;

	np = of_find_node_by_name(NULL, "misc-gpio");
	if (np == NULL) {
        bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "sonos misc gpio block missing from DTB");
        return;
    }

    linein_overvolt_gpio = of_get_named_gpio(np, "linein-overvolt", 0);
}

void linein_overvolt_init(void)
{
	int ret = 0;

	linein_get_gpio();
	if (gpio_is_valid(linein_overvolt_gpio)) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "line-in overvolt gpio %d found in DTB", linein_overvolt_gpio);
		ret = gpio_request_one(linein_overvolt_gpio, GPIOF_DIR_IN, "ADC Overvoltage");
		if (ret) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: GPIO %d request failed with error %d.\n", __func__, linein_overvolt_gpio, ret);
			return;
		}
		linein_over_irq = gpio_to_irq(linein_overvolt_gpio);
		ret = request_irq(linein_over_irq, linein_overvolt_isr, IRQF_TRIGGER_RISING, "ADC Overvoltage", NULL);
		if (ret) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: IRQ %d request failed with error %d.\n", __func__, linein_over_irq, ret);
			return;
		}
	}
}

void linein_overvolt_exit(void)
{
	if (gpio_is_valid(linein_overvolt_gpio)) {
		free_irq(linein_over_irq, NULL);
		gpio_free(linein_overvolt_gpio);
	}
}
