/*
 * Copyright (c) 2018 Sonos, Inc.
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/uaccess.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>
#include <linux/of_gpio.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/types.h>

#include "blackbox.h"
#include "gpio_outs.h"
#include "sensors_dev.h"

#define PROCFS_DIR		"driver/gpio_outs"

struct gpio_outs_dtb *gpio_outs_table[GPIO_OUTS_NUM_SUPPORTED] = { 0 };

static struct gpio_outs_dtb gpio_outs_list[] = {
	{	.name  = GPIO_OUTS_NAME_BT_CLASSIC_RES,
		.index = GPIO_OUTS_BT_CLASSIC_RES,
		.filename = GPIO_OUTS_NAME_BT_CLASSIC_RES,
		.init_val = 0,
		.proc_output = "BT-Classic-Reset",
	}
};
#define GPIO_OUTS_DTB_ENTRIES (sizeof(gpio_outs_list) / sizeof(struct gpio_outs_dtb))

static struct proc_dir_entry *gpio_outs_dir = NULL;

static int gpio_outs_proc_show(struct seq_file *m, void *v)
{
	struct gpio_outs_dtb *goe = m->private;
	int val;

	if (goe) {
		val = gpio_get_value(goe->gpio);
		seq_printf(m, "name: %s  value: %d flags: 0x%x gpio_num: %d\n", goe->name, val, goe->gpio_flags, goe->gpio);
	} else {
		seq_printf(m, "Unable to read GPIO\n");
	}

	return 0;
}

static ssize_t gpio_outs_proc_write(struct file *file, const char __user * buffer,
				  size_t count, loff_t *data)
{
	const u8 *on  = "1";
	const u8 *off = "0";
	u8 buf[10];
	struct gpio_outs_dtb *goe = PDE_DATA(file_inode(file));

	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	}
	buf[count] = '\0';

	if (strncmp(buf, on, strlen(on)) == 0) {
		gpio_set_value(goe->gpio, 1);
	} else if (strncmp(buf, off, strlen(off)) == 0) {
		gpio_set_value(goe->gpio, 0);
	} else {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "invalid operation: %s", buf);
	}

	return count;
}

static int gpio_outs_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, gpio_outs_proc_show, PDE_DATA(inode));
}

static const struct file_operations gpio_outs_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= gpio_outs_proc_open,
	.write		= gpio_outs_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

static int gpio_outs_proc_init(struct gpio_outs_dtb *goe)
{
	struct proc_dir_entry *proc;

	proc = proc_create_data(goe->filename, 0666, gpio_outs_dir, &gpio_outs_proc_fops, goe);
	if (proc == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: ERROR: %s file not created\n", __func__, goe->filename);
		return -EIO;
	}
	return 0;
}

static int gpio_outs_get_gpio(enum gpio_outs_pins index)
{
	if (gpio_outs_table[index]) {
		return gpio_outs_table[index]->gpio;
	} else {
		return -1;
	}
}

static const char * gpio_outs_get_label(enum gpio_outs_pins index)
{
	if (gpio_outs_table[index]) {
		return gpio_outs_table[index]->name;
	} else {
		return "";
	}
}

static void gpio_outs_table_init(void)
{
	int idx;
	struct device_node *np;

	for (idx = 0; idx < GPIO_OUTS_NUM_SUPPORTED; idx++) {
		gpio_outs_list[idx].gpio = -EINVAL;
	}

	np = of_find_node_by_name(NULL, "gpio-outs");
	if (np == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "sonos gpio-outs block missing from DTB");
		return;
	}

	for (idx = 0; idx < GPIO_OUTS_DTB_ENTRIES; idx++) {
		struct gpio_outs_dtb *goe = &gpio_outs_list[idx];
		int gpio = of_get_named_gpio_flags(np, goe->name, 0, &goe->gpio_flags);
		if (gpio_is_valid(gpio)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s found in DTB, gpio %d, index %d", goe->name, gpio, goe->index);
			gpio_outs_list[idx].gpio = gpio;
			gpio_outs_table[goe->index] = goe;
		} else {
			bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s not found in DTB", goe->name);
		}
	}
}

void gpio_outs_init(void)
{
	int idx;
	int ret = 0;

	gpio_outs_table_init();

	gpio_outs_dir = proc_mkdir(PROCFS_DIR, NULL);
	if (!gpio_outs_dir) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: failed to create procfs dir %s", __func__, PROCFS_DIR);
		return;
	}

	for (idx = 0; idx < GPIO_OUTS_NUM_SUPPORTED; idx++) {
		int gpio = gpio_outs_get_gpio(idx);
		struct gpio_outs_dtb *goe;

		if (!gpio_is_valid(gpio)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s: idx %d, invalid GPIO %d", __func__, idx, gpio);
			continue;
		}

		goe = gpio_outs_table[idx];
		ret = gpio_request_one(gpio, goe->gpio_flags | GPIOF_DIR_OUT, gpio_outs_get_label(idx));
		if (ret) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: GPIO %d request failed.", __func__, gpio);
			proc_remove(gpio_outs_dir);
			return;
		}

		gpio_set_value(gpio, goe->init_val);
		gpio_outs_proc_init(goe);
	}

	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "gpio-outs init success.");

	return;
}

void gpio_outs_exit(void)
{
	int idx;
	for (idx = 0; idx < GPIO_OUTS_NUM_SUPPORTED; idx++) {
		struct gpio_outs_dtb *goe;
		int gpio = gpio_outs_get_gpio(idx);
		if (gpio_is_valid(gpio)) {
			goe = gpio_outs_table[idx];
			if (goe) {
				remove_proc_entry(goe->filename, gpio_outs_dir);
			}
			gpio_free(gpio);
		}
	}

	proc_remove(gpio_outs_dir);
}
