/*
 * Copyright (c) 2020 Sonos, Inc.
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/jiffies.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>
#include <linux/of_gpio.h>
#include <linux/slab.h>
#include <linux/sonos_kernel.h>
#include <asm/uaccess.h>

#include "blackbox.h"
#include "event_queue_api.h"
#include "hwevent_queue_api.h"
#include "button_inst.h"
#include "gpio_buttons.h"
#include "mdp.h"


#define	FXL6408_DEVID		0x01
#define	FXL6408_IO_DIR		0x03
#define	FXL6408_OUTPUT_STATE	0x05
#define	FXL6408_OUTPUT_HIZ	0x07
#define	FXL6408_INPUT_DEFAULT	0x09
#define	FXL6408_PULL_ENABLE	0x0b
#define	FXL6408_PDNPUP		0x0d
#define	FXL6408_INPUT_STATUS	0x0f
#define	FXL6408_INT_MASK	0x11
#define	FXL6408_INT_STATUS	0x13

#define FXL6408_SW_RST		0x01
#define FXL6408_RST_INT		0x02
#define FXL6408_MF_MASK		0xE0
#define FXL6408_MF_VAL		0xA0

struct i2c_client *fxl6408_client = NULL;
int fxl6408_device_ready = 0;
static int fxl6408_expected = 0;
static struct gpio *fxl6408_int = NULL;
static u32 num_buttons = 0;
static int button_held = 0;
static int button_source[8];
static int likely_wake = 0;

static int io_dir_init;
static int pdnpup_init;
static int pull_enable_init;
static int input_status_mask;
static int input_default;
static int interrupt_mask;

extern struct button_inst buttons_table[];

irqreturn_t fxl6408_isr(int, void*);
void fxl6408_debounce(struct work_struct *);

typedef struct fxl6408_private {
	int irq;
	int irq_trigger;
	uint8_t active_mask;
	uint8_t fxl6408_current;
	uint8_t fxl6408_last;
	uint8_t fxl6408_suspend;
	uint8_t pressed;
} fxl6408_priv;

fxl6408_priv fxl6408;
fxl6408_priv *fp = &fxl6408;

static struct _fxl6408_work_wrapper {
        struct delayed_work fxl6408_work;
        struct gpio *pins;
        char sim;
} fxl6408_work_wrapper;

#define BUTTON_DEFAULT_HOLD_TIME        340
#define BUTTON_DEFAULT_REPEAT_TIME      160

#define FXL6408_WRITE(R, V)     i2c_smbus_write_byte_data(fxl6408_client, R, V)
#define FXL6408_READ(R)         i2c_smbus_read_byte_data(fxl6408_client, R)

void fxl6408_send_hwevtq(enum HWEVTQ_EventSource source, int event)
{
	hwevtq_send_event(source, event);
	if (likely_wake) {
		hwevtq_send_event(source, HWEVTQINFO_WAKEUP);
		likely_wake = 0;
	}
}

static int fxl6408_suspend(struct device *dev)
{
	(void)dev;
	disable_irq_nosync(fp->irq);
	fp->fxl6408_suspend = (input_status_mask & FXL6408_READ(FXL6408_INPUT_STATUS));

	return 0;
}

static int fxl6408_resume(struct device *dev)
{
	int ret;
	int input;
	(void)dev;
	if (fp->fxl6408_suspend != (input = (input_status_mask & FXL6408_READ(FXL6408_INPUT_STATUS)))) {
		likely_wake = 1;
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s: input_status 0x%x/0x%x", __func__, fp->fxl6408_suspend, input);
		cancel_delayed_work(&fxl6408_work_wrapper.fxl6408_work);
		schedule_delayed_work(&fxl6408_work_wrapper.fxl6408_work, msecs_to_jiffies(20));
	} else {
		if ((ret = FXL6408_READ(FXL6408_INT_STATUS)))  {
			int button;
			bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s: int_status 0x%x, input_status 0x%x/0x%x",
				__func__, ret, fp->fxl6408_suspend, input);
			likely_wake = 1;
			for (button = 0;button < num_buttons;button++) {
				if (ret & (1 << button)) {
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
					fxl6408_send_hwevtq((enum HWEVTQ_EventSource)button_source[button], HWEVTQINFO_PRESSED);
#endif
				}
			}
		} else {
			bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s: int_status 0x%x, input_status 0x%x/0x%x",
				__func__, ret, fp->fxl6408_suspend, input);
		}
	}

	free_irq(fp->irq, fp);
	ret = request_irq(fp->irq, fxl6408_isr, fxl6408_int->flags, fxl6408_int->label, (void*)fp);
	if (ret) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Could not re-enable fxl6408 interrrupt...");
	}
	return 0;
}

const struct dev_pm_ops fxl6408_i2c_pm_ops = {
        .suspend = fxl6408_suspend, \
        .resume = fxl6408_resume, \
        .freeze = fxl6408_suspend, \
        .thaw = fxl6408_resume, \
        .poweroff = fxl6408_suspend, \
        .restore = fxl6408_resume,
};

static int fxl6408_proc_read(struct seq_file *m, void *v)
{

	if (!fxl6408_device_ready) {
		seq_printf(m, "fxl6408 device not ready\n");
		return 0;
	}
	if (fxl6408_client == NULL) {
		seq_printf(m, "FXL6408: I2C fxl6408_client not initialized.\n");
		return 0;
	}
#if 0
	for (reg = 0; reg < 0x16; reg++) {
		s32 ret = FXL6408_READ(reg);
		if (ret < 0) {
			int r;
			for (r = reg; r < 0x3f; r++) {
				val[r] = 0;
			}
			seq_printf(m, "FXL6408: i2c read error on reg %02x\n", reg);
			break;
		}
		val[reg] = ret;
	}
	for (reg = 0; reg < 0x16; reg++) {
		if (reg && ((reg%16) == 0))
			seq_printf(m, "\n");
		if (((reg%16) == 0))
			seq_printf(m, "%2x: ", reg);
		if (reg && ((reg%4) == 0) &&  ((reg%16) != 0))
			seq_printf(m, "  ");
		seq_printf(m, "%02x ", val[reg]);
	}
#endif

	seq_printf(m, "dev_id:          0x%x\n", FXL6408_READ(FXL6408_DEVID));
	seq_printf(m, "io_direction:    0x%x\n", FXL6408_READ(FXL6408_IO_DIR));
	seq_printf(m, "output_state:    0x%x\n", FXL6408_READ(FXL6408_OUTPUT_STATE));
	seq_printf(m, "output_hiz:      0x%x\n", FXL6408_READ(FXL6408_OUTPUT_HIZ));
	seq_printf(m, "input_default:   0x%x\n", FXL6408_READ(FXL6408_INPUT_DEFAULT));
	seq_printf(m, "pull_enable:     0x%x\n", FXL6408_READ(FXL6408_PULL_ENABLE));
	seq_printf(m, "pdnpup:          0x%x\n", FXL6408_READ(FXL6408_PDNPUP));
	seq_printf(m, "input_status:    0x%x\n", FXL6408_READ(FXL6408_INPUT_STATUS));
	seq_printf(m, "int_mask:        0x%x\n", FXL6408_READ(FXL6408_INT_MASK));
	seq_printf(m, "int_status:      0x%x\n", FXL6408_READ(FXL6408_INT_STATUS));
	seq_printf(m, "\n");

	return 0;
}

static ssize_t fxl6408_proc_write(struct file *file, const char __user * buffer,
			size_t count, loff_t *data)
{
	int num_regs = 0x31;
	char buf[200];
	char *peq;
	int result = 0, error;
	int temp;
	u32  regnum;
	uint8_t  val;

	if (!fxl6408_device_ready) {
		bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "fxl6408 device not ready");
		result = -EIO;
	} else if (count >= sizeof(buf)) {
		result = -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;

	} else {
		buf[count] = '\0';
		peq = strchr(buf, '=');
		if (peq != NULL) {
			*peq = 0;
			if (kstrtoint(peq+1, 0, &temp)) {
				bb_log_dev(&(fxl6408_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "ksrtol (register address) failed!");
			}
			val = temp;
			if (strncmp(buf, "reg", 3) == 0 && buf[3] != '=') {
				if (kstrtoint(buf+3, 0, &temp)) {
					bb_log_dev(&(fxl6408_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "kstrtol (register value) failed!");
				}
				regnum = temp;
				if ((regnum < num_regs) && (fxl6408_client != NULL)) {
					error = FXL6408_WRITE(regnum, val);
					if (error) {
						bb_log_dev(&(fxl6408_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "register write failed with %d", error);
						return error;
					}
				}
			}
		}
		result = count;
	}

	return result;
}

static int fxl6408_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, fxl6408_proc_read, PDE_DATA(inode));
}

static const struct file_operations fxl6408_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= fxl6408_proc_open,
	.write		= fxl6408_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define FXL6408_PROCFS_FILE "driver/fxl6408"

static int fxl6408_proc_init(void)
{
	struct proc_dir_entry *proc;

	proc = proc_create_data(FXL6408_PROCFS_FILE, 0666, NULL, &fxl6408_proc_fops, NULL);
	if (!proc) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Couldn't create /proc/driver/fxl6408");
		return -EIO;
	}

	return 0;
}

static void fxl6408_proc_remove(void)
{
	remove_proc_entry(FXL6408_PROCFS_FILE, NULL);
}

static void fxl6408_reg_init(void)
{
        FXL6408_WRITE(FXL6408_IO_DIR, io_dir_init);
        FXL6408_WRITE(FXL6408_PDNPUP, pdnpup_init);
        FXL6408_WRITE(FXL6408_PULL_ENABLE, pull_enable_init);
        FXL6408_WRITE(FXL6408_INPUT_STATUS, input_status_mask);
        FXL6408_WRITE(FXL6408_INPUT_DEFAULT, input_default);
        fp->fxl6408_current = fp->fxl6408_last = input_default;
        FXL6408_WRITE(FXL6408_INT_MASK, interrupt_mask);

	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "fxl6408 registers initialized - int_status = 0x%x", FXL6408_READ(FXL6408_INT_STATUS));
}

static int fxl6408_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id)
{
	s32 ret = 0;
	char value = 0;
	fxl6408_client = i2c_client;

	bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "Probe for gpio_expander...");

	FXL6408_WRITE(FXL6408_DEVID, FXL6408_SW_RST);
	ret = FXL6408_READ(FXL6408_DEVID);

	if (ret < 0) {
		bb_log_dev(&(fxl6408_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "Error reading ID register [%d]", ret);
		return -EIO;
	} else {
		value = ret;
		if (((value & FXL6408_MF_MASK) != FXL6408_MF_VAL) || !(value & FXL6408_RST_INT)) {
			bb_log_dev(&(fxl6408_client->dev), BB_MOD_SENSORS, BB_LVL_WARNING, "Invalid response %x, expected %x", value, (FXL6408_MF_VAL | FXL6408_RST_INT));
			return -EINVAL;
		}
	}

	INIT_DELAYED_WORK(&fxl6408_work_wrapper.fxl6408_work, fxl6408_debounce);

	if (gpio_is_valid(fxl6408_int->gpio)) {
                ret = gpio_request_one(fxl6408_int->gpio, GPIOF_DIR_IN, fxl6408_int->label);
                if (ret) {
                        bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Could not get interrupt %d.", fxl6408_int->gpio);
			goto probe_return;
                }
		fp->irq = gpio_to_irq(fxl6408_int->gpio);
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s: claiming irq %d", __func__, fp->irq);
                ret = request_irq(fp->irq, fxl6408_isr, fxl6408_int->flags, fxl6408_int->label, (void*)fp);
                if (ret) {
                        bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Could not get button IRQ %d.", gpio_to_irq(fxl6408_int->gpio));
			goto probe_return;
                }
        }

	fxl6408_reg_init();

	bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "fxl6408 driver loaded");
	fxl6408_device_ready = 1;
probe_return:
	return ret;
}

int fxl6408_i2c_remove(struct i2c_client *i2c_client)
{
	return 0;
}

static const struct i2c_device_id fxl6408_i2c_id[] = {
	{ "fxl6408", 0 },
	{ }
};
MODULE_DEVICE_TABLE(fxl6408_i2c, fxl6408_i2c_id);

static struct of_device_id fxl6408_ids[] = {
	{ .compatible = "fairchild,fxl6408" },
	{ }
};

static struct i2c_driver fxl6408_i2c_driver = {
	.driver = {
		.name		= "fxl6408",
		.owner		= THIS_MODULE,
		.pm             = &fxl6408_i2c_pm_ops,
		.of_match_table	= fxl6408_ids,
	},
	.id_table	= fxl6408_i2c_id,
	.probe		= fxl6408_i2c_probe,
	.remove		= fxl6408_i2c_remove,
};

irqreturn_t fxl6408_isr(int irq, void *data)
{
	(void)data;

	disable_irq_nosync(irq);

        bb_log_dbg(BB_MOD_SENSORS, "Triggered fxl6408 ISR.");
	cancel_delayed_work(&fxl6408_work_wrapper.fxl6408_work);
        schedule_delayed_work(&fxl6408_work_wrapper.fxl6408_work, msecs_to_jiffies(20));
	return IRQ_HANDLED;
}

void fxl6408_debounce(struct work_struct *work)
{
	uint8_t difference;
	int ret;
	int button = 0;
	int new_default;


	ret = FXL6408_READ(FXL6408_INT_STATUS);
	fp->fxl6408_current = (input_status_mask & FXL6408_READ(FXL6408_INPUT_STATUS));
	difference = fp->fxl6408_last ^ fp->fxl6408_current;
	bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "current = 0x%x, last = 0x%x, difference = 0x%x", fp->fxl6408_current, fp->fxl6408_last, difference);
	if (!difference && ((fp->fxl6408_current == interrupt_mask) || likely_wake)) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s, but current and last are the same value [0x%x] - INT_STATUS = 0x%x",
			       	__func__, fp->fxl6408_current, ret);
		difference = ret;
	}
	fp->fxl6408_last = fp->fxl6408_current;
	new_default = FXL6408_READ(FXL6408_INPUT_DEFAULT);

	for (button = 0;button < num_buttons;button++) {
		int button_mask = (1 << button);
		if (difference & (1 << button)) {
			if (fp->fxl6408_current & button_mask) {
				event_queue_send_event((enum HWEVTQ_EventSource)button_source[button], HWEVTQINFO_RELEASED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				fxl6408_send_hwevtq((enum HWEVTQ_EventSource)button_source[button], HWEVTQINFO_RELEASED);
#endif
				new_default |= button_mask;
				button_held &= ~button_mask;
			} else {
				event_queue_send_event((enum HWEVTQ_EventSource)button_source[button], HWEVTQINFO_PRESSED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                fxl6408_send_hwevtq((enum HWEVTQ_EventSource)button_source[button], HWEVTQINFO_PRESSED);
#endif
				new_default &= ~button_mask;
				button_held |= button_mask;
			}
		} else if (button_held & button_mask) {
			event_queue_send_event((enum HWEVTQ_EventSource)button_source[button], HWEVTQINFO_REPEATED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                        fxl6408_send_hwevtq((enum HWEVTQ_EventSource)button_source[button], HWEVTQINFO_REPEATED);
#endif
		}
	}

#if 0
	printk(KERN_ERR "input_default = 0x%x, input_status = 0x%x\n", FXL6408_READ(FXL6408_INPUT_DEFAULT), FXL6408_READ(FXL6408_INPUT_STATUS));
	printk(KERN_ERR "changing default from 0x%x to 0x%x\n", FXL6408_READ(FXL6408_INPUT_DEFAULT), new_default);
#endif
	FXL6408_WRITE(FXL6408_INPUT_DEFAULT, new_default);

	cancel_delayed_work(&fxl6408_work_wrapper.fxl6408_work);
	if (button_held) {
		schedule_delayed_work(&fxl6408_work_wrapper.fxl6408_work, msecs_to_jiffies(BUTTON_DEFAULT_HOLD_TIME));
	}

	free_irq(fp->irq, fp);
	ret = request_irq(fp->irq, fxl6408_isr, fxl6408_int->flags, fxl6408_int->label, (void*)fp);
	if (ret) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Could not re-enable fxl6408 interrrupt...");
	}
}

enum HWEVTQ_EventInfo fxl6408_button_get_value(enum button_pins button)
{
	int current_input_status;
	int button_mask = 0;
	int i;

	for (i = 0;i < BUTTONS_NUM_SUPPORTED;i++) {
		if (button == (button_source[i] - HWEVTQSOURCE_BUTTON_PLAYPAUSE)) {
			button_mask = 1 << i;
			break;
		}
	}

	current_input_status = (input_status_mask & FXL6408_READ(FXL6408_INPUT_STATUS));

	if (buttons_table[button].source == HWEVTQSOURCE_NO_SOURCE) {
		return HWEVTQINFO_RELEASED;
	} else {
		return (!!(current_input_status & button_mask) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED);
	}
}

void fxl6408_button_table_init(void)
{
	int i;

	for (i = 0; i < BUTTONS_NUM_SUPPORTED; i++) {
		buttons_table[i].source = HWEVTQSOURCE_NO_SOURCE;
		buttons_table[i].gpio = -EINVAL;
	}


	for (i = 0; i < num_buttons; i++) {
		int table_entry = button_source[i] - HWEVTQSOURCE_BUTTON_PLAYPAUSE;
		buttons_table[table_entry].button_read = fxl6408_button_get_value;
		buttons_table[table_entry].source = HWEVTQSOURCE_BUTTON_PLAYPAUSE + table_entry;
	}
}

int fxl6408_init(void)
{
	int ret = 0;
	int i;
	struct device_node *dev_node;
	struct device_node *child;
	char *status_check, chkbuf[10];

	dev_node = of_find_compatible_node(NULL, NULL, "fairchild,fxl6408");

	if (!dev_node) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "No fxl6408 in this device");
		ret = -ENODEV;
		goto init_return;
	} else {
		fxl6408_expected = 1;
	}

	status_check = chkbuf;
	of_property_read_string(dev_node, "status", (const char **)&status_check);
	if (strncmp(status_check, "okay", 4)) {
		bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "fxl6408 not enabled");
		ret = -ENODEV;
		goto init_return;
	}

	if (of_property_read_u32(dev_node, "io-dir-init", &io_dir_init)) {
		io_dir_init = 0;
	}
	if (of_property_read_u32(dev_node, "pdnpup-init", &pdnpup_init)) {
		pdnpup_init = 0;
	}
	if (of_property_read_u32(dev_node, "pull-enable-init", &pull_enable_init)) {
		pull_enable_init = 0;
	}
	if (of_property_read_u32(dev_node, "input-status-mask", &input_status_mask)) {
		input_status_mask = 0;
	}
	if (of_property_read_u32(dev_node, "input-default", &input_default)) {
		input_default = 0;
	}
	if (of_property_read_u32(dev_node, "interrupt-mask", &interrupt_mask)) {
		interrupt_mask = 0;
	}

	fxl6408_int = (struct gpio *)kmalloc(sizeof(struct gpio), GFP_KERNEL);
	child = of_get_child_by_name(dev_node, "fxl6408-int");
	if (!child) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "Could not find fxl6408 interrupt gpio!\n");
		ret = -EINVAL;
		goto init_return;
	} else {
		fxl6408_int->gpio = of_get_named_gpio(child, "int-gpio", 0);
		of_property_read_u32(child, "int-flags", (u32*)&(fxl6408_int->flags));
		of_property_read_string(child, "int-label", (const char **)&(fxl6408_int->label));
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "%s - gpio %d", fxl6408_int->label, fxl6408_int->gpio);
	}

	of_property_read_u32(dev_node, "num-buttons", &num_buttons);

	for (i = 0; i < num_buttons; i++) {
		of_property_read_u32_index(dev_node, "button-source", i, &button_source[i]);
	}

	ret = i2c_add_driver(&fxl6408_i2c_driver);

	fxl6408_button_table_init();

	if (ret) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "fxl6408: i2c_add_driver failed with %d", ret);
		goto init_return;
	}
	fxl6408_proc_init();

	bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "fxl6408 driver loaded successfully");

init_return:
	return ret;
}

void fxl6408_exit(void)
{
	if (fxl6408_expected) {
		fxl6408_device_ready = 0;
		fxl6408_proc_remove();
		cancel_delayed_work_sync(&fxl6408_work_wrapper.fxl6408_work);
		i2c_del_driver(&fxl6408_i2c_driver);
	}

	if (fxl6408_int && gpio_is_valid(fxl6408_int->gpio)) {
		gpio_free(fxl6408_int->gpio);
		free_irq(fp->irq, &fxl6408);
	}

	if (fxl6408_int) {
		kfree(fxl6408_int);
	}
}

