/*
 * Copyright (c) 2015 - 2020 Sonos, Inc.
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>
#include <linux/of_gpio.h>

#include "blackbox.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "button_inst.h"
#include "gpio_buttons.h"
#include "mdp.h"

typedef struct button_private {
	int irq;
	int irq_trigger;
	int button;
	struct delayed_work button_work;
	uint8_t pressed;
} button_priv;

#define BUTTON_DEFAULT_HOLD_TIME	340
#define BUTTON_DEFAULT_REPEAT_TIME	160
static int button_repeat_time = BUTTON_DEFAULT_REPEAT_TIME;
button_priv buttons[BUTTONS_NUM_SUPPORTED] = {{ 0 }};

irqreturn_t button_isr(int irq, void *data)
{
	button_priv *bp=(button_priv *)data;
	disable_irq_nosync(irq);
	bb_log_dbg(BB_MOD_SENSORS, "In button ISR %d.", irq);
	cancel_delayed_work(&(bp->button_work));
	schedule_delayed_work(&(bp->button_work), msecs_to_jiffies(20));
	return IRQ_HANDLED;
}

static void button_debounce(struct work_struct *work)
{
	button_priv *bp=container_of((struct delayed_work *)work,button_priv,button_work);
	int repeat_time = BUTTON_DEFAULT_HOLD_TIME;
	enum HWEVTQ_EventSource source = buttons_get_source(bp->button);
	int err = 0;
	if (!gpio_get_value(buttons_get_gpio(bp->button))) {
		if (!bp->pressed) {
			event_queue_send_event(source, HWEVTQINFO_PRESSED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(source, HWEVTQINFO_PRESSED);
#endif
			bp->pressed = 1;
		} else {
			event_queue_send_event(source, HWEVTQINFO_REPEATED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(source, HWEVTQINFO_REPEATED);
#endif
			repeat_time = button_repeat_time;
		}
		schedule_delayed_work((struct delayed_work *)work, msecs_to_jiffies(repeat_time));
	} else {
		if (bp->pressed) {
			event_queue_send_event(source, HWEVTQINFO_RELEASED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(source, HWEVTQINFO_RELEASED);
#endif
			bp->pressed = 0;
		}
	}
	if (repeat_time != button_repeat_time) {
		bb_log_dbg(BB_MOD_SENSORS, "%s: changing irq %d polarity.", __func__, bp->irq);
		free_irq(bp->irq, bp);
		bp->irq_trigger = (bp->irq_trigger == IRQF_TRIGGER_LOW) ? IRQF_TRIGGER_HIGH : IRQF_TRIGGER_LOW;
		err = request_irq(bp->irq, button_isr, bp->irq_trigger,
					 (bp->irq_trigger == IRQF_TRIGGER_LOW) ? "Button trigger low" : "Button trigger high", bp);
		if (err) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: IRQ %d request failed.", __func__, bp->irq);
		}
	}
}

enum HWEVTQ_EventInfo button_get_value(enum button_pins pin)
{
	if (gpio_is_valid(buttons_get_gpio(pin))) {
		return (gpio_get_value(buttons_get_gpio(pin)) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED);
	} else {
		return HWEVTQINFO_RELEASED;
	}
}

void button_set_repeat_time(int new_time)
{
	button_repeat_time = new_time;
}
int button_get_repeat_time(void)
{
	return button_repeat_time;
}

struct sensors_mech_button {
	char *name;
	int index;
};

struct sensors_mech_button sensors_mech_buttons[] = {
	{.name = "play-pause-button", .index = BUTTONS_INDEX_BUTTON_PLAYPAUSE},
	{.name = "volup-button", .index = BUTTONS_INDEX_BUTTON_VOL_UP},
	{.name = "voldn-button", .index = BUTTONS_INDEX_BUTTON_VOL_DN},
	{.name = "connect-button", .index = BUTTONS_INDEX_BUTTON_JOIN},
        {.name = "mode-button", .index = BUTTONS_INDEX_BUTTON_MODE},
	{.name = "power-button", .index = BUTTONS_INDEX_BUTTON_POWER},
};
#define SENSORS_NUM_POSSIBLE_BUTTONS (sizeof(sensors_mech_buttons) / sizeof(struct sensors_mech_button))


void button_table_init(void)
{
	int i;
	struct device_node *np;

	np = of_find_node_by_name(NULL, "misc-gpio");
	if (np && of_property_read_bool(np, "buttons-on-expander")) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "physical buttons are behind a gpio expander");
		return;
	}

	for (i = 0; i < BUTTONS_NUM_SUPPORTED; i++) {
		buttons_table[i].source = HWEVTQSOURCE_NO_SOURCE;
		buttons_table[i].gpio = -EINVAL;
	}

	if (np == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "sonos misc gpio block missing from DTB");
		return;
	}

	for (i = 0; i < SENSORS_NUM_POSSIBLE_BUTTONS; i++) {
		struct sensors_mech_button *smb = &sensors_mech_buttons[i];
		int gpio = of_get_named_gpio(np, smb->name, 0);
		if (of_property_read_bool(np, "no-irq")) {
			buttons[i].irq = -1;
		}
		if (gpio_is_valid(gpio)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s found in DTB, gpio %d, index %d", smb->name, gpio, smb->index);
			buttons_table[smb->index].gpio = gpio;
			buttons_table[smb->index].source = HWEVTQSOURCE_BUTTON_PLAYPAUSE + smb->index;
			buttons_table[smb->index].name = smb->name;
			buttons_table[smb->index].button_read = button_get_value;
		} else {
			bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s not found in DTB", smb->name);
		}
	}
}

void button_init(void)
{
	int ret = 0;
	int x;

	button_table_init();

	for (x=0; x < BUTTONS_NUM_SUPPORTED; x++) {
		int gpio = buttons_get_gpio(x);
		if (!gpio_is_valid(gpio)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "button %d, invalid GPIO %d", x, gpio);
			continue;
		}

		buttons[x].button = x;
		INIT_DELAYED_WORK(&(buttons[x].button_work), button_debounce);
		ret = gpio_request_one(gpio, GPIOF_DIR_IN, buttons_get_label(buttons[x].button));
		if (ret) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: GPIO %d request failed.", __func__, gpio);
			return;
		}

		if (buttons[x].irq != -1) {
			buttons[x].irq = gpio_to_irq(gpio);
			buttons[x].irq_trigger = IRQF_TRIGGER_LOW;
			ret = request_irq(buttons[x].irq, button_isr, buttons[x].irq_trigger, "Button (low-triggered)", &(buttons[x]));
			if (ret) {
				bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: IRQ %d request failed.", __func__, buttons[x].irq);
			}
		}
	}
	return;
}

void button_exit(void)
{
	int x;
	for (x = 0; x < BUTTONS_NUM_SUPPORTED; x++) {
		int gpio = buttons_get_gpio(x);
		if (!gpio_is_valid(gpio)) {
			continue;
		}
		cancel_delayed_work_sync(&(buttons[x].button_work));
		gpio_free(buttons_get_gpio(buttons[x].button));
		free_irq(buttons[x].irq, &(buttons[x]));
	}
}
