/*
 * Copyright (c) 2018-2020 Sonos, Inc.
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>
#include <asm/uaccess.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/of_gpio.h>

#include "blackbox.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "sensors_dev.h"

#define CABLE_DETECT_SPOOF_DETECT      (-1)
#define CABLE_DETECT_SPOOF_DISCONNECTED  0
#define CABLE_DETECT_SPOOF_CONNECTED     1


struct cable_detect_entry {
	char *name;
	char *proc_filename;
	char *proc_output;
	char *keyword;
	int   irq;
	int   spoof;
	int   gpio;
	enum of_gpio_flags  gpio_flags;
	struct delayed_work detect_work;
	enum HWEVTQ_EventSource source;
};

struct cable_detect_entry cable_detect_table[] = {
	{.name = CABLE_DETECT_NAME_LINEIN,
	 .proc_filename = "driver/linein",
	 .proc_output = "Line-in",
	 .keyword = "linedet=",
	 .source = HWEVTQSOURCE_LINEIN},
	{.name = CABLE_DETECT_NAME_SUBWOOFER_OUT,
	 .proc_filename = "driver/subwoof",
	 .proc_output = "Subwoofer-out",
	 .keyword = "subwoofdet=",
	 .source = HWEVTQSOURCE_SUBWOOFER},
	{.name = CABLE_DETECT_NAME_AVTRIGGER,
	 .proc_filename = "driver/avtrigger",
	 .proc_output = "Avtrigger",
	 .keyword = "avtrigger=",
	 .source = HWEVTQSOURCE_AVTRIGGER},
	{.name = CABLE_DETECT_NAME_LINEOUT,
	 .proc_filename = "driver/lineout",
	 .proc_output = "Lineout",
	 .keyword = "lineout=",
	 .source = HWEVTQSOURCE_LINEOUT},
	{.name = CABLE_DETECT_NAME_SPDIF_OUT,
	 .proc_filename = "driver/spdif_out",
	 .proc_output = "Spdif_out",
	 .keyword = "spdif_out=",
	 .source = HWEVTQSOURCE_SPDIF_OUT},
};
#define CABLE_DETECT_TABLE_SIZE (sizeof(cable_detect_table) / sizeof(struct cable_detect_entry))

irqreturn_t cable_detect_isr(int irq, void *data)
{
	struct cable_detect_entry *cde = data;
	disable_irq_nosync(cde->irq);
	schedule_delayed_work(&cde->detect_work, msecs_to_jiffies(500));
	return IRQ_HANDLED;
}

static void cable_detect_work_func(struct work_struct *work)
{
	struct cable_detect_entry *cde = container_of((struct delayed_work *)work, struct cable_detect_entry, detect_work);
	int err = 0;
	int flags = 0;

	free_irq(cde->irq, cde);
	if (gpio_get_value(cde->gpio)) {
		flags = IRQF_TRIGGER_LOW;
		if (cde->spoof == CABLE_DETECT_SPOOF_DETECT) {
			event_queue_send_event(cde->source, HWEVTQINFO_CONNECTED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(cde->source, HWEVTQINFO_CONNECTED);
#endif
		}
	} else {
		flags = IRQF_TRIGGER_HIGH;
		if (cde->spoof == CABLE_DETECT_SPOOF_DETECT) {
			event_queue_send_event(cde->source, HWEVTQINFO_DISCONNECTED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(cde->source, HWEVTQINFO_DISCONNECTED);
#endif
		}
	}
	err = request_irq(cde->irq, cable_detect_isr, flags, cde->name, cde);
	if (err) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: Unable to re-request %s IRQ.\n", __func__, cde->name);
	}
}

static int cable_detect_get_real_value(struct cable_detect_entry *cde)
{
	int connected = gpio_get_value(cde->gpio);
	if (cde->gpio_flags & OF_GPIO_ACTIVE_LOW) {
		connected = !connected;
	}
	return connected;
}

static int cable_detect_get_value(struct cable_detect_entry *cde)
{
	if (cde->spoof == CABLE_DETECT_SPOOF_DETECT) {
		return cable_detect_get_real_value(cde);
	}
	return (cde->spoof == CABLE_DETECT_SPOOF_CONNECTED) ? 1 : 0;
}

static enum HWEVTQ_EventInfo __cable_detect_get_info(struct cable_detect_entry *cde)
{
	return cable_detect_get_value(cde) ? HWEVTQINFO_CONNECTED : HWEVTQINFO_DISCONNECTED;
}

#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
static enum HWEVTQ_EventInfo __cable_detect_get_hwevtq_info(struct cable_detect_entry *cde)
{
        return cable_detect_get_value(cde) ? HWEVTQINFO_CONNECTED : HWEVTQINFO_DISCONNECTED;
}
#endif

enum HWEVTQ_EventInfo cable_detect_get_info(char *name)
{
	int i;
	for (i = 0; i < CABLE_DETECT_TABLE_SIZE; i++) {
		struct cable_detect_entry *cde = &cable_detect_table[i];
		if (gpio_is_valid(cde->gpio) && (strcmp(name, cde->name) == 0)) {
			return __cable_detect_get_info(cde);
		}
	}
	return HWEVTQINFO_DISCONNECTED;
}

int cable_detect_set_spoof(struct cable_detect_entry *cde, int spoof)
{
	enum HWEVTQ_EventInfo info;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	enum HWEVTQ_EventInfo hwevtq_info = HWEVTQINFO_NO_EVENT;
#endif

	switch (spoof) {
	case CABLE_DETECT_SPOOF_DETECT:
		cde->spoof = spoof;
		info = __cable_detect_get_info(cde);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_info = __cable_detect_get_hwevtq_info(cde);
#endif
		break;
	case CABLE_DETECT_SPOOF_CONNECTED:
		cde->spoof = spoof;
		info = HWEVTQINFO_CONNECTED;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_info = HWEVTQINFO_CONNECTED;
#endif
		break;
	case CABLE_DETECT_SPOOF_DISCONNECTED:
		cde->spoof = spoof;
		info = HWEVTQINFO_DISCONNECTED;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_info = HWEVTQINFO_DISCONNECTED;
#endif
		break;
	default:
		return -EINVAL;
	}
	event_queue_send_event(cde->source, info);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	hwevtq_send_event(cde->source, hwevtq_info);
#endif
	return 0;
}

static int
cable_detect_proc_show(struct seq_file *m, void *v)
{
	struct cable_detect_entry *cde = m->private;
	char *actual = cable_detect_get_real_value(cde) ? "connected" : "disconnected";

	if (cde->spoof == CABLE_DETECT_SPOOF_DETECT) {
		seq_printf(m, "%s:  %s\n", cde->name, actual);
	} else {
		char *sim = cable_detect_get_value(cde) ? "connected" : "disconnected";
		seq_printf(m, "%s:  %s (Simulation mode, actual %s)\n", cde->name, sim, actual);
	}
	return 0;
}

static int
cable_detect_proc_write(struct file *file, const char __user * buffer,
				  size_t count, loff_t *data)
{
	char buf[200];
	int len;
	struct cable_detect_entry *cde = PDE_DATA(file_inode(file));

	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	}
	buf[count] = '\0';

	len = strlen(cde->keyword);
	if (strncmp(buf, cde->keyword, len) == 0) {
		char *detect = "detect";
		char *on  = "on";
		char *off = "off";
		if (strncmp(&buf[len], detect, strlen(detect)) == 0) {
			cable_detect_set_spoof(cde, CABLE_DETECT_SPOOF_DETECT);
		} else if (strncmp(&buf[len], on, strlen(on)) == 0) {
			cable_detect_set_spoof(cde, CABLE_DETECT_SPOOF_CONNECTED);
		} else if (strncmp(&buf[len], off, strlen(off)) == 0) {
			cable_detect_set_spoof(cde, CABLE_DETECT_SPOOF_DISCONNECTED);
		} else {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "illegal line-in override: %s", &buf[len]);
		}
	}
	return count;
}

static int cable_detect_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, cable_detect_proc_show, PDE_DATA(inode));
}

static const struct file_operations cable_detect_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= cable_detect_proc_open,
	.write		= cable_detect_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

static int cable_detect_proc_init(char *filename, struct cable_detect_entry *cde)
{
	struct proc_dir_entry *proc;

	proc = proc_create_data(filename, 0666, NULL, &cable_detect_proc_fops, cde);
	if (proc == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "ERROR: %s file not created\n", filename);
		return -EIO;
	}
	return 0;
}

static void cable_detect_proc_remove(char *filename)
{
	remove_proc_entry(filename, NULL);
}

static inline void cable_detect_get_gpio(struct cable_detect_entry *cde)
{
	struct device_node *np;

	np = of_find_node_by_name(NULL, "misc-gpio");
	if (np == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "sonos misc gpio block missing from DTB");
		return;
	}
	cde->gpio = of_get_named_gpio_flags(np, cde->name, 0, &cde->gpio_flags);
}

void cable_detect_init(void)
{
	int i, error;

	for (i = 0; i < CABLE_DETECT_TABLE_SIZE; i++) {
		struct cable_detect_entry *cde = &cable_detect_table[i];

		cde->gpio = -EINVAL;
		cde->spoof = CABLE_DETECT_SPOOF_DETECT;

		cable_detect_get_gpio(cde);
		if (!gpio_is_valid(cde->gpio)) {
			bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "%s gpio not found in DTB", cde->name);
			continue;
		}

		INIT_DELAYED_WORK(&cde->detect_work, cable_detect_work_func);
		error = gpio_request_one(cde->gpio, GPIOF_DIR_IN, cde->name);
		if (error) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: GPIO %d request failed with error %d\n", __func__, cde->gpio, error);
			return;
		}
		cde->irq = gpio_to_irq(cde->gpio);
		error = request_irq(cde->irq, cable_detect_isr,
							(gpio_get_value(cde->gpio) ? IRQF_TRIGGER_LOW : IRQF_TRIGGER_HIGH),
							cde->name, cde);
		if (error) {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, " %s: IRQ %d request failed with error %d\n", __func__, cde->irq, error);
			return;
		}
		cable_detect_proc_init(cde->proc_filename, cde);
	}
}

void cable_detect_exit(void)
{
	int i;
	for (i = 0; i < CABLE_DETECT_TABLE_SIZE; i++) {
		struct cable_detect_entry *cde = &cable_detect_table[i];
		if (gpio_is_valid(cde->gpio)) {
			cable_detect_proc_remove(cde->proc_filename);
			free_irq(cde->irq, cde);
			gpio_free(cde->gpio);
		}
	}
}
