/*
 * Button simulation
 * Copyright (c) 2014 - 2020 Sonos Inc.
 *
 * SPDX-License-Identifier: GPL-2.0
 */

#include <linux/stddef.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/string.h>
#include <asm/uaccess.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/of.h>
#include <linux/sonos_kernel.h>
#include "mdp.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "button_inst.h"
#include "blackbox.h"
#include "sonos_lock.h"

#define MAX_SIMULATED_BUTTONS 9
struct button_sim_inst buttons_simulated[MAX_SIMULATED_BUTTONS];

#define MS_TO_JIFFIES(x) (((x) * HZ) / 1000)
#define REPEAT_TIME_FIRST   340
#define REPEAT_TIME         160
#define MAX_CMD_SIZE         15

struct buttons_sim_data {
	struct button_sim_inst *buttons_match;
	char simulated_cmd[MAX_CMD_SIZE+1];
	struct timer_list sim_timer;
	int msec_remaining;
	int msec_this_interval;
	int no_repeat_sim;
};
struct buttons_sim_data buttons_sim;
int sim_enabled;

static inline int
buttons_sim_cmd_match(char *cmd, struct button_sim_inst *bsm)
{
	return (strncmp(cmd, bsm->cmd, strlen(bsm->cmd)) == 0);
}

static void
buttons_sim_timer_handler(unsigned long x)
{
	struct button_sim_inst *bsm;
	enum HWEVTQ_EventInfo info;

	buttons_sim.msec_remaining -= buttons_sim.msec_this_interval;
        info = (buttons_sim.msec_remaining <= 0) ? HWEVTQINFO_RELEASED : HWEVTQINFO_REPEATED;
	if (buttons_sim.no_repeat_sim) {
		info = (buttons_sim.msec_remaining <= 0) ? HWEVTQINFO_RELEASED : HWEVTQINFO_NO_EVENT;
	}

	bsm = buttons_sim.buttons_match;
	while (bsm->source != HWEVTQSOURCE_NO_SOURCE)  {
		if (buttons_sim_cmd_match(buttons_sim.simulated_cmd, bsm)) {
			if (bsm->source == HWEVTQSOURCE_BUTTON_POWER) {
				if (info != HWEVTQINFO_NO_EVENT) {
#if defined (SONOS_ARCH_ATTR_SOC_IS_A113)
				if (PRODUCT_ID_IS_DHUEZ || PRODUCT_ID_IS_MONACO || PRODUCT_ID_IS_MONACOSL) {
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
					hwevtq_send_event_defer(HWEVTQSOURCE_BUTTON_POWER,
							HWEVTQINFO_RELEASED);
#endif
				}
#endif
				}
			} else {
				if (info != HWEVTQINFO_NO_EVENT) {
					event_queue_send_event_defer(bsm->source, info);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
					hwevtq_send_event_defer(bsm->source, info);
#endif
				}
			}
		}
		bsm++;
	}

	if (buttons_sim.msec_remaining > 0) {
		buttons_sim.msec_this_interval = REPEAT_TIME;
		mod_timer(&buttons_sim.sim_timer, jiffies + MS_TO_JIFFIES(REPEAT_TIME));
	}
}

static unsigned int
buttons_sim_get_duration(char *param)
{
	unsigned long ulongval = 0;
	unsigned long max_duration = 1000 * 15;
	unsigned int duration = 0;

	if (*param == '=') {
		ulongval = simple_strtoul(param+1, NULL, 10);
		if (ulongval < max_duration) {
			duration = ulongval;
		} else {
			bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "duration of %lums exceeds max of %lums, default to 0ms\n",
				   ulongval, max_duration);
		}
	}
	return duration;
}

static int
buttons_proc_show(struct seq_file *m, void *v)
{
	struct button_sim_inst *bsm;

	seq_printf(m, "\nButtons supported\n");
	bsm = buttons_sim.buttons_match;
	while (bsm->source != HWEVTQSOURCE_NO_SOURCE) {
		seq_printf(m, "%s\n", bsm->cmd);
		bsm++;
	}
	return 0;
}

int
buttons_sim_process_cmd(char *cmd)
{
	struct button_sim_inst *bsm, *match_bsm;
	unsigned int duration = 0;
	int msec;

	if (timer_pending(&buttons_sim.sim_timer)) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s: simulated button event already in progress\n", __func__);
		return 0;
	}

	match_bsm = NULL;
	bsm = buttons_sim.buttons_match;
	while (bsm->source != HWEVTQSOURCE_NO_SOURCE) {
		if (buttons_sim_cmd_match(cmd, bsm)) {
			match_bsm = bsm;
			if (match_bsm->source == HWEVTQSOURCE_BUTTON_POWER)  {
#if defined (SONOS_ARCH_ATTR_SOC_IS_A113)
			if (PRODUCT_ID_IS_DHUEZ || PRODUCT_ID_IS_MONACO || PRODUCT_ID_IS_MONACOSL) {
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event(HWEVTQSOURCE_BUTTON_POWER, HWEVTQINFO_PRESSED);
#endif
			}
#endif
			} else {
				event_queue_send_event(bsm->source, HWEVTQINFO_PRESSED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event(bsm->source, HWEVTQINFO_PRESSED);
#endif
			}
		}
		bsm++;
	}
	if (match_bsm != NULL) {
		strncpy(buttons_sim.simulated_cmd, match_bsm->cmd, MAX_CMD_SIZE);
		buttons_sim.simulated_cmd[MAX_CMD_SIZE] = '\0';
		duration = buttons_sim_get_duration(cmd+strlen(match_bsm->cmd));
		if (duration > REPEAT_TIME_FIRST) {
			msec = REPEAT_TIME_FIRST;
		} else {
			msec = duration;
		}
		buttons_sim.msec_remaining = duration;
		buttons_sim.msec_this_interval = msec;
		buttons_sim.no_repeat_sim = match_bsm->no_repeat_sim;
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "Simulating %s button press for %dms", match_bsm->cmd, duration);
		mod_timer(&buttons_sim.sim_timer, jiffies + MS_TO_JIFFIES(msec));
		return 1;
	}
	return 0;
}

static ssize_t
buttons_proc_write(struct file *file, const char __user * buffer,
				  size_t count, loff_t *data)
{
	char buf[200];

	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	}
	buf[count] = '\0';
	buttons_sim_process_cmd(buf);
	return count;
}

static int buttons_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, buttons_proc_show, PDE_DATA(inode));
}

static const struct file_operations buttons_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= buttons_proc_open,
	.write		= buttons_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define BUTTONS_PROCFS_FILE "driver/buttons"

static int buttons_proc_init(void)
{
	struct proc_dir_entry *proc;

	proc = proc_create(BUTTONS_PROCFS_FILE, 0666, NULL, &buttons_proc_fops);
	if (proc == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_ERR, "%s file not created\n", BUTTONS_PROCFS_FILE);
		return -EIO;
	}

	return 0;
}

static void buttons_proc_remove(void)
{
	remove_proc_entry(BUTTONS_PROCFS_FILE, NULL);
}

static void buttons_init_sim_table(void)
{
	int i;
	struct device_node *node, *child;
	struct property *prop;
	struct button_sim_inst *bsi = &buttons_simulated[0];
	struct button_sim_inst *last_bsi = &buttons_simulated[MAX_SIMULATED_BUTTONS-1];

	for (i = 0; i < MAX_SIMULATED_BUTTONS; i++) {
		buttons_simulated[i].source = HWEVTQSOURCE_NO_SOURCE;
	}

	node = of_find_node_by_name(NULL, "simulated-buttons");
	if (node == NULL) {
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "no simulated buttons found in DTB");
		return;
	}

	for_each_available_child_of_node(node, child) {
		const __be32 *srcs = NULL;

		prop = of_find_property(child, "event-sources", NULL);
		if (prop == NULL) {
			bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "event-sources missing from simulated button %s", child->name);
			continue;
		}

		while (bsi <= last_bsi) {
			u32 source_offset;
			srcs = of_prop_next_u32(prop, srcs, &source_offset);
			if (srcs == NULL) {
				break;
			}
			bsi->source = HWEVTQSOURCE_BUTTON_PLAYPAUSE + source_offset;
			strlcpy(bsi->cmd, child->name, sizeof(bsi->cmd));
			bsi->no_repeat_sim = 0;
			if (of_property_read_bool(child, "no-repeat-sim")) {
				bsi->no_repeat_sim = 1;
				bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "button %s, source %d - REPEAT events sim not supported", bsi->cmd, bsi->source);
			} else {
				bb_log(BB_MOD_SENSORS, BB_LVL_DEBUG, "button %s, source %d", bsi->cmd, bsi->source);
			}
			bsi++;
		}

		if (bsi > last_bsi) {
			bb_log(BB_MOD_SENSORS, BB_LVL_WARNING, "simulated buttons overflow");
			break;
		}
	}
}

void
buttons_sim_init(void)
{
	sim_enabled = 0;
	if (is_mdp_authorized(MDP_AUTH_FLAG_BUTTON_DEBUG)) {
		buttons_init_sim_table();
		buttons_sim.buttons_match = buttons_simulated;
		init_timer(&buttons_sim.sim_timer);
		buttons_sim.sim_timer.function = buttons_sim_timer_handler;
		buttons_sim.sim_timer.data = 0;
		buttons_proc_init();
		bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "button simulation enabled");
		sim_enabled = 1;
	}
}

void
buttons_sim_exit(void)
{
	if (sim_enabled) {
		del_timer(&buttons_sim.sim_timer);
		buttons_proc_remove();
	}
}
