/**
 * Copyright (C) 2019 Bosch Sensortec GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * Neither the name of the copyright holder nor the names of the
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDER
 * OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES(INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 * The information provided is believed to be accurate and reliable.
 * The copyright holder assumes no responsibility
 * for the consequences of use
 * of such information nor for any infringement of patents or
 * other rights of third parties which may result from its use.
 * No license is granted by implication or otherwise under any patent or
 * patent rights of the copyright holder.
 *
 * @file	bma420.h
 * @date	2019-12-03
 * @version	v2.12.6
 *
 */

#ifndef BMA420_H_
#define BMA420_H_

/*! CPP guard */
#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************/

/*!             Header files
 ****************************************************************************/
#include "bma4.h"

/***************************************************************************/

/*!               Macro definitions
 ****************************************************************************/

/*! @name Chip ID of BMA420 sensor */
#define BMA420_CHIP_ID                UINT8_C(0x10)

/*! @name Sensor feature size */
#define BMA420_FEATURE_SIZE           UINT8_C(0x0C)

/*! @name Data Lengths */
#define BMA420_RD_WR_MIN_LEN          UINT8_C(0x02)

/*! @name Feature start address */
#define BMA420_ANY_MOT_START_ADDR     UINT8_C(0x00)
#define BMA420_NO_MOT_START_ADDR      UINT8_C(0x04)
#define BMA420_CONFIG_ID_START_ADDR   UINT8_C(0x08)
#define BMA420_AXES_REMAP_START_ADDR  UINT8_C(0x0A)

/*! @name Mask definitions for axes re-mapping */
#define BMA420_X_AXIS_MSK             UINT8_C(0x03)
#define BMA420_X_AXIS_SIGN_MSK        UINT8_C(0x04)
#define BMA420_Y_AXIS_MSK             UINT8_C(0x18)
#define BMA420_Y_AXIS_SIGN_MSK        UINT8_C(0x20)
#define BMA420_Z_AXIS_MSK             UINT8_C(0xC0)
#define BMA420_Z_AXIS_SIGN_MSK        UINT8_C(0x01)

/*! @name Bit position for axes re-mapping */
#define BMA420_X_AXIS_SIGN_POS        UINT8_C(0x02)
#define BMA420_Y_AXIS_POS             UINT8_C(0x03)
#define BMA420_Y_AXIS_SIGN_POS        UINT8_C(0x05)
#define BMA420_Z_AXIS_POS             UINT8_C(0x06)

/*! @name Any/No-motion threshold macros */
#define BMA420_ANY_NO_MOT_THRES_MSK   UINT16_C(0x07FF)

/*! @name Any/No-motion enable macros */
#define BMA420_ANY_NO_MOT_AXIS_EN_MSK UINT16_C(0xE000)
#define BMA420_ANY_NO_MOT_AXIS_EN_POS UINT8_C(0x0D)

/*! @name Any/No-motion duration macros */
#define BMA420_ANY_NO_MOT_DUR_MSK     UINT16_C(0x1FFF)

/*! @name Any/No-motion axis enable macros */
#define BMA420_X_AXIS_EN              UINT8_C(0x01)
#define BMA420_Y_AXIS_EN              UINT8_C(0x02)
#define BMA420_Z_AXIS_EN              UINT8_C(0x04)
#define BMA420_EN_ALL_AXIS            UINT8_C(0x07)
#define BMA420_DIS_ALL_AXIS           UINT8_C(0x00)

/*! @name Interrupt status macros */
#define BMA420_ANY_MOT_INT            UINT8_C(0x20)
#define BMA420_NO_MOT_INT             UINT8_C(0x40)
#define BMA420_ERROR_INT              UINT8_C(0x80)

/******************************************************************************/
/*!  @name         Structure Declarations                                     */
/******************************************************************************/
/*! @name Structure to define any/no-motion configuration */
struct bma420_any_no_mot_config
{
    /*! Expressed in 50 Hz samples (20 ms) */
    uint16_t duration;

    /*! Threshold value for Any-motion/No-motion detection in
     * 5.11g format
     */
    uint16_t threshold;

    /*! To enable selected axes */
    uint8_t axes_en;
};

/*! @name Structure to store the value of re-mapped axis and its sign */
struct bma420_axes_remap
{
    /*! Re-mapped x-axis */
    uint8_t x_axis;

    /*! Re-mapped y-axis */
    uint8_t y_axis;

    /*! Re-mapped z-axis */
    uint8_t z_axis;

    /*! Re-mapped x-axis sign */
    uint8_t x_axis_sign;

    /*! Re-mapped y-axis sign */
    uint8_t y_axis_sign;

    /*! Re-mapped z-axis sign */
    uint8_t z_axis_sign;
};

/***************************************************************************/

/*!                 BMA420 User Interface function prototypes
 ****************************************************************************/

/*!
 * @brief This API is the entry point. Call this API before using all other APIs
 * This API reads the chip-id of the sensor and sets the resolution, feature
 * length and the type of variant.
 *
 * @param[in,out] dev : Structure instance of bma4_dev
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_start(struct bma4_dev *dev);

/*!
 * @brief This API is used to upload the configuration file to enable the
 * features of the sensor.
 *
 * @param[in, out] dev : Structure instance of bma4_dev
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_write_config_file(struct bma4_dev *dev);

/*!
 * @brief This API is used to get the configuration id of the sensor.
 *
 * @param[out] config_id    : Pointer variable used to store the configuration
 *                            id.
 * @param[in] dev           : Structure instance of bma4_dev.
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_get_config_id(uint16_t *config_id, struct bma4_dev *dev);

/*!
 * @brief This API maps/un-maps the user provided interrupt to either interrupt
 * pin1 or pin2 in the sensor.
 *
 * @param[in] int_line  : Variable to select either interrupt pin1 or pin2.
 *
 *  int_line    |   Macros
 *  ------------|-------------------
 *      0       | BMA4_INTR1_MAP
 *      1       | BMA4_INTR2_MAP
 *
 * @param[in] int_map   : Variable to specify the interrupts.
 * @param[in] enable    : Variable to specify mapping or un-mapping of
 *                        interrupts.
 *  enable  |   Macros
 *----------|-------------------
 *   0x01   |  BMA4_EN
 *  0x00    |  BMA4_DIS
 *
 * @param[in] dev      : Structure instance of bma4_dev.
 *
 * @note Below macros specify the feature interrupts.
 *          - BMA420_ANY_MOT_INT
 *          - BMA420_NO_MOT_INT
 *          - BMA420_ERROR_INT
 *
 * @note Below macros specify the hardware interrupts.
 *          - BMA4_FIFO_FULL_INT
 *          - BMA4_FIFO_WM_INT
 *          - BMA4_DATA_RDY_INT
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_map_interrupt(uint8_t int_line, uint16_t int_map, uint8_t enable, struct bma4_dev *dev);

/*!
 * @brief This API reads the bma420 interrupt status from the sensor.
 *
 * @param[out] int_status   : Variable to store the interrupt status
 *                            read from the sensor.
 * @param[in] dev           : Structure instance of bma4_dev.
 *
 * @note Below macros specify the feature interrupts.
 *          - BMA420_ANY_MOT_INT
 *          - BMA420_NO_MOT_INT
 *          - BMA420_ERROR_INT
 *
 * @note Below macros specify the hardware interrupts.
 *          - BMA4_FIFO_FULL_INT
 *          - BMA4_FIFO_WM_INT
 *          - BMA4_DATA_RDY_INT
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_read_int_status(uint16_t *int_status, struct bma4_dev *dev);

/*!
 * @brief This API performs x, y and z axis re-mapping in the sensor.
 *
 * @param[in] remap_data    : Pointer to store axes re-mapping data.
 * @param[in] dev           : Structure instance of bma4_dev
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_set_remap_axes(const struct bma420_axes_remap *remap_data, struct bma4_dev *dev);

/*!
 * @brief This API reads the x, y and z axis re-mapped data from the sensor.
 *
 * @param[out] remap_data   : Pointer to store the read axes re-mapped data.
 * @param[in] dev           : Structure instance of bma4_dev
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_get_remap_axes(struct bma420_axes_remap *remap_data, struct bma4_dev *dev);

/*!
 * @brief This API sets the configuration of any-motion feature in the sensor
 * This API enables/disables the any-motion feature according to the axis set.
 *
 * @param[in] any_mot           : Pointer to structure variable to configure
 *                                any-motion.
 * @verbatim
 * -------------------------------------------------------------------------
 *         Structure parameters    |        Description
 * --------------------------------|----------------------------------------
 *                                 |        Defines the number of
 *                                 |        consecutive data points for
 *                                 |        which the threshold condition
 *         duration                |        must be respected, for interrupt
 *                                 |        assertion. It is expressed in
 *                                 |        50 Hz samples (20 ms).
 *                                 |        Range is 0 to 163sec.
 *                                 |        Default value is 5 = 100ms.
 * --------------------------------|----------------------------------------
 *                                 |        Slope threshold value for
 *                                 |        Any-motion detection
 *         threshold               |        in 5.11g format.
 *                                 |        Range is 0 to 1g.
 *                                 |        Default value is 0xAA = 83mg.
 * --------------------------------|----------------------------------------
 *                                 |        Enables the feature on a per-axis
 *         axis_en                 |        basis.
 * ---------------------------------------------------------------------------
 * @endverbatim
 *
 *  Value    |  axis_en
 *  ---------|-------------------------
 *  0x00     |  BMA420_DIS_ALL_AXIS
 *  0x01     |  BMA420_X_AXIS_EN
 *  0x02     |  BMA420_Y_AXIS_EN
 *  0x04     |  BMA420_Z_AXIS_EN
 *  0x07     |  BMA420_EN_ALL_AXIS
 *
 * @param[in] dev               : Structure instance of bma4_dev
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_set_any_mot_config(const struct bma420_any_no_mot_config *any_mot, struct bma4_dev *dev);

/*!
 * @brief This API gets the configuration of any-motion feature from the
 * sensor.
 *
 * @param[out] any_mot        : Pointer to structure variable to configure
 *                              any-motion.
 * @verbatim
 * -------------------------------------------------------------------------
 *         Structure parameters    |        Description
 * --------------------------------|----------------------------------------
 *                                 |        Defines the number of
 *                                 |        consecutive data points for
 *                                 |        which the threshold condition
 *         duration                |        must be respected, for interrupt
 *                                 |        assertion. It is expressed in
 *                                 |        50 Hz samples (20 ms).
 *                                 |        Range is 0 to 163sec.
 *                                 |        Default value is 5 = 100ms.
 * --------------------------------|----------------------------------------
 *                                 |        Slope threshold value for
 *                                 |        Any-motion detection
 *         threshold               |        in 5.11g format.
 *                                 |        Range is 0 to 1g.
 *                                 |        Default value is 0xAA = 83mg.
 * --------------------------------|-----------------------------------------
 *                                 |        Enables the feature on a per-axis
 *         axis_en                 |        basis.
 * ---------------------------------------------------------------------------
 * @endverbatim
 *
 *  Value    |  axis_en
 *  ---------|-------------------------
 *  0x00     |  BMA420_DIS_ALL_AXIS
 *  0x01     |  BMA420_X_AXIS_EN
 *  0x02     |  BMA420_Y_AXIS_EN
 *  0x04     |  BMA420_Z_AXIS_EN
 *  0x07     |  BMA420_EN_ALL_AXIS
 *
 * @param[in] dev               : Structure instance of bma4_dev
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_get_any_mot_config(struct bma420_any_no_mot_config *any_mot, struct bma4_dev *dev);

/*!
 * @brief This API sets the configuration of no-motion feature in the sensor
 * This API enables/disables the no-motion feature according to the axis set.
 *
 * @param[in] no_mot                : Pointer to structure variable to configure
 *                                  no-motion.
 * @verbatim
 * -------------------------------------------------------------------------
 *         Structure parameters    |        Description
 * --------------------------------|----------------------------------------
 *                                 |        Defines the number of
 *                                 |        consecutive data points for
 *                                 |        which the threshold condition
 *         duration                |        must be respected, for interrupt
 *                                 |        assertion. It is expressed in
 *                                 |        50 Hz samples (20 ms).
 *                                 |        Range is 0 to 163sec.
 *                                 |        Default value is 5 = 100ms.
 * --------------------------------|----------------------------------------
 *                                 |        Slope threshold value for
 *                                 |        No-motion detection
 *         threshold               |        in 5.11g format.
 *                                 |        Range is 0 to 1g.
 *                                 |        Default value is 0xAA = 83mg.
 * --------------------------------|----------------------------------------
 *                                 |        Enables the feature on a per-axis
 *         axis_en                 |        basis.
 * ---------------------------------------------------------------------------
 * @endverbatim
 *
 *  Value    |  axis_en
 *  ---------|-------------------------
 *  0x00     |  BMA420_DIS_ALL_AXIS
 *  0x01     |  BMA420_X_AXIS_EN
 *  0x02     |  BMA420_Y_AXIS_EN
 *  0x04     |  BMA420_Z_AXIS_EN
 *  0x07     |  BMA420_EN_ALL_AXIS
 *
 * @param[in] dev               : Structure instance of bma4_dev
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_set_no_mot_config(const struct bma420_any_no_mot_config *no_mot, struct bma4_dev *dev);

/*!
 * @brief This API gets the configuration of no-motion feature from the
 * sensor.
 *
 * @param[out] no_mot        : Pointer to structure variable to configure
 *                              no-motion.
 * @verbatim
 * -------------------------------------------------------------------------
 *         Structure parameters    |        Description
 * --------------------------------|----------------------------------------
 *                                 |        Defines the number of
 *                                 |        consecutive data points for
 *                                 |        which the threshold condition
 *         duration                |        must be respected, for interrupt
 *                                 |        assertion. It is expressed in
 *                                 |        50 Hz samples (20 ms).
 *                                 |        Range is 0 to 163sec.
 *                                 |        Default value is 5 = 100ms.
 * --------------------------------|----------------------------------------
 *                                 |        Slope threshold value for
 *                                 |        No-motion detection
 *         threshold               |        in 5.11g format.
 *                                 |        Range is 0 to 1g.
 *                                 |        Default value is 0xAA = 83mg.
 * --------------------------------|-----------------------------------------
 *                                 |        Enables the feature on a per-axis
 *         axis_en                 |        basis.
 * ---------------------------------------------------------------------------
 * @endverbatim
 *
 *  Value    |  axis_en
 *  ---------|-------------------------
 *  0x00     |  BMA420_DIS_ALL_AXIS
 *  0x01     |  BMA420_X_AXIS_EN
 *  0x02     |  BMA420_Y_AXIS_EN
 *  0x04     |  BMA420_Z_AXIS_EN
 *  0x07     |  BMA420_EN_ALL_AXIS
 *
 * @param[in] dev               : Structure instance of bma4_dev
 *
 * @return Result of API execution status
 *
 * @retval 0 -> Success
 * @retval Any non zero value -> Fail
 */
uint16_t bma420_get_no_mot_config(struct bma420_any_no_mot_config *no_mot, struct bma4_dev *dev);

#ifdef __cplusplus
}
#endif /*End of CPP guard */

#endif /*End of header guard macro */
