/*
 * Copyright (c) 2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/slab.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/semaphore.h>
#include "blackbox.h"
#include "sdd.h"
#include "touchpad_log_entry.h"


static DEFINE_SEMAPHORE(sdd_touchpad_log_lock);

void sdd_touchpad_log_event(struct seq_file *m, touchpad_log_entry_t *tle)
{
        seq_printf(m, "event %d, ", tle->event_gest);
	if ((tle->x == TOUCHPAD_LOG_XY_RELEASED) && (tle->y == TOUCHPAD_LOG_XY_RELEASED)) {
		seq_printf(m, "released, ");
		if (tle->vector == TOUCHPAD_LOG_VECTOR_NONVOLUME) {
			seq_printf(m, "released,  ");
		} else {
			seq_printf(m, "vect %02x? ", tle->vector);
		}
	} else {
		seq_printf(m, "xy %02x.%02x, ", tle->x, tle->y);
		if (tle->vector == TOUCHPAD_LOG_VECTOR_NONVOLUME) {
			seq_printf(m, "nonvolume, ");
		} else {
			seq_printf(m, "volume %02x, ", tle->vector);
		}
	}
}

void sdd_touchpad_log_gesture(struct seq_file *m, touchpad_log_entry_t *tle)
{
	char *gs;
	switch (tle->event_gest) {
	case TOUCHPAD_LOG_GEST_PLAY_PAUSE:
		gs = "Play/Pause";
		break;
	case TOUCHPAD_LOG_GEST_HOLD:
		gs = "Hold";
		break;
	case TOUCHPAD_LOG_GEST_NEXT:
		gs = "Next";
		break;
	case TOUCHPAD_LOG_GEST_PREV:
		gs = "Prev";
		break;
	case TOUCHPAD_LOG_GEST_VOLUME:
		gs = "Volume";
		break;
	default:
		gs = "?????????";
	}
	seq_printf(m, "Gesture %s, ", gs);
}

int sdd_touchpad_log_show(struct seq_file *m, void *v)
{
	int done;

	down(&sdd_touchpad_log_lock);
	while (1) {
		touchpad_log_entry_t tle;
		done = sdd_read_touchpad_log((u8 *)&tle, sizeof(tle));
		if (done) {
			break;
		}
		seq_printf(m, "PSoC%d ", tle.psoc);
		if (tle.type == TOUCHPAD_LOG_TYPE_EVENT) {
		    sdd_touchpad_log_event(m, &tle);
		} else if (tle.type == TOUCHPAD_LOG_TYPE_GESTURE) {
		    sdd_touchpad_log_gesture(m, &tle);
		}
		seq_printf(m, "%dms\n", tle.msec);
	}
	seq_printf(m, "\n");
	up(&sdd_touchpad_log_lock);
	return 0;
}
