/*
 * Copyright (c) 2014-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/timer.h>
#include <linux/list.h>
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#endif
#include <linux/kthread.h>
#include "blackbox.h"
#include "sdd.h"

void sdd_start_timer(sdd_timer_t *sdt)
{
	unsigned long flags;
	spin_lock_irqsave(&sdt->lock, flags);
	if (sdt->state == SDD_TIMER_STATE_EXPIRED) {
		sdt->restart_count++;
	}
	sdt->state = SDD_TIMER_STATE_RUN;
	mod_timer(&sdt->timer, jiffies + msecs_to_jiffies(sdt->timeout_ms));
	spin_unlock_irqrestore(&sdt->lock, flags);
}

void sdd_stop_timer(sdd_timer_t *sdt)
{
	unsigned long flags;
	spin_lock_irqsave(&sdt->lock, flags);
	del_timer(&sdt->timer);
	sdt->state = SDD_TIMER_STATE_STOP;
	sdt->stop_count++;
	spin_unlock_irqrestore(&sdt->lock, flags);
}

void sdd_change_timer(sdd_timer_t *sdt, unsigned int msecs)
{
	unsigned long flags;
	spin_lock_irqsave(&sdt->lock, flags);
	sdt->timeout_ms = msecs;
	mod_timer(&sdt->timer, jiffies + msecs_to_jiffies(msecs));
	spin_unlock_irqrestore(&sdt->lock, flags);
}

void sdd_inc_timer(sdd_timer_t *sdt, unsigned int msecs)
{
	unsigned long flags;
	spin_lock_irqsave(&sdt->lock, flags);
	sdt->timeout_ms += msecs;
	spin_unlock_irqrestore(&sdt->lock, flags);
}

void sdd_dec_timer(sdd_timer_t *sdt, unsigned int msecs)
{
	unsigned long flags;
	spin_lock_irqsave(&sdt->lock, flags);
	sdt->timeout_ms -= msecs;
	spin_unlock_irqrestore(&sdt->lock, flags);
}

sdd_timer_t *
sdd_create_timer(sdd_data_t *psd, char *name,
				 unsigned int msecs,
				 void (*action)(void *))
{
	sdd_timer_t *sdt;

	sdt = kmalloc(sizeof(sdd_timer_t), GFP_KERNEL);
	if (sdt == NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: alloc error", __func__);
		return NULL;
	}
	memset(sdt, 0, sizeof(*sdt));

	spin_lock_init(&sdt->lock);
	strncpy(sdt->name, name, SDD_TIMER_NAME_LEN-1);
	sdt->sdd_data = psd;
	sdt->name[SDD_TIMER_NAME_LEN-1] = '\0';
	sdt->timeout_ms = msecs;
	sdt->state = SDD_TIMER_STATE_STOP;
	sdt->timer.function = sdd_timer;
	sdt->timer.data = (unsigned long)sdt;
	sdt->handler = action;
	init_timer(&sdt->timer);
	INIT_LIST_HEAD(&sdt->list);
	list_add(&sdt->list, &psd->timers);
	return sdt;
}

void sdd_timer(unsigned long x)
{
	sdd_timer_t *sdt = (sdd_timer_t *)x;
	sdd_data_t *psd = sdt->sdd_data;

	sdt->state = SDD_TIMER_STATE_EXPIRED;
	sdt->timeout_count++;
	complete(&psd->sdd_completion);
}

void sdd_process_timers(sdd_data_t *psd)
{
	sdd_timer_t *sdt;

	list_for_each_entry(sdt, &psd->timers, list)
	{
		if (sdt->state == SDD_TIMER_STATE_EXPIRED) {
			sdt->state = SDD_TIMER_STATE_STOP;
			sdt->handler(psd);
			sdt->processed_count++;
		}
	}
}

static char * sdd_timer_state_strings[] = {"stop", "run", "expired"};

int sdd_proc_read_timers(struct seq_file *m, void *v)
{
	sdd_data_t *psd = (sdd_data_t *)m->private;
	sdd_timer_t *sdt;

	seq_printf(m, "\nTimers\n");
	list_for_each_entry(sdt, &psd->timers, list)
	{
		seq_printf(m, "%s: %s\n", sdt->name, sdd_timer_state_strings[sdt->state]);
		seq_printf(m, "%9d msec timeout\n",sdt->timeout_ms);
		seq_printf(m, "%9d fired\n", sdt->timeout_count);
		seq_printf(m, "%9d processed\n", sdt->processed_count);
		seq_printf(m, "%9d stopped\n", sdt->stop_count);
		seq_printf(m, "%9d restarted\n", sdt->restart_count);
		seq_printf(m, "\n");
	}

	return 0;
}
