/*
 * Copyright (c) 2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */


#include <linux/delay.h>
#include "blackbox.h"
#include "sdd.h"

static struct sge_config_param sdd_legacy_config_table[] = {
	{.func = sdd_set_press_hold_time,         .value = SGE_PRESS_HOLD_TIME},
	{.func = sdd_set_repeat_time,             .value = SGE_REPEAT_TIME},
	{.func = sdd_set_min_samples_per_gesture, .value = SGE_MIN_SAMPLES_PER_GESTURE},
	{.func = sdd_set_min_samples_per_tap,     .value = SGE_MIN_SAMPLES_PER_TAP},
	{.func = sdd_set_move_interval,           .value = SGE_MOVE_INTERVAL},
	{.func = sdd_set_finger_threshold,        .property_name = "finger-threshold"},
	{.func = sdd_set_noise_threshold,         .property_name = "noise-threshold"},
	{.func = sdd_set_finger_hysteresis,       .property_name = "hysteresis"},
	{.func = sdd_set_dts_share_swd_uart,	  .property_name = "share-swd-uart"},
};
#define SGE_CONFIG_COMMON_ENTRIES 1
#define SGE_LEGACY_CONFIG_TABLE_ENTRIES (sizeof(sdd_legacy_config_table) / sizeof(struct sge_config_param))

static struct sge_config_param sdd_button_config_table[] = {
	{.func = sdd_set_finger_threshold,            .property_name = "finger-threshold"},
	{.func = sdd_set_noise_threshold,             .property_name = "noise-threshold"},
	{.func = sdd_set_finger_hysteresis,           .property_name = "hysteresis"},
	{.func = sdd_set_prox_finger_threshold,       .property_name = "prox-finger-threshold"},
	{.func = sdd_set_prox_noise_threshold,        .property_name = "prox-noise-threshold"},
	{.func = sdd_set_prox_finger_hysteresis,      .property_name = "prox-hysteresis"},
};
#define SGE_BUTTON_CONFIG_TABLE_ENTRIES (sizeof(sdd_button_config_table) / sizeof(struct sge_config_param))

static struct sge_config_param sdd_touchpad_config_table[] = {
	{.func = sdd_set_finger_threshold,            .property_name = "finger-threshold"},
	{.func = sdd_set_noise_threshold,             .property_name = "noise-threshold"},
	{.func = sdd_set_finger_hysteresis,           .property_name = "hysteresis"},
	{.func = sdd_set_prox_finger_threshold,       .property_name = "prox-finger-threshold"},
	{.func = sdd_set_prox_noise_threshold,        .property_name = "prox-noise-threshold"},
	{.func = sdd_set_prox_finger_hysteresis,      .property_name = "prox-hysteresis"},
};
#define SGE_TOUCHPAD_CONFIG_TABLE_ENTRIES (sizeof(sdd_touchpad_config_table) / sizeof(struct sge_config_param))

static int sdd_configure_legacy_sge(sdd_data_t *psd)
{
	int i, first_entry, error, attempts, max_attempts=3;
	struct sge_config_param *param;

	param = sdd_legacy_config_table;
	if (psd->dts_config_flags & SDD_DTS_NO_CAPTOUCH) {
		first_entry = SGE_LEGACY_CONFIG_TABLE_ENTRIES - SGE_CONFIG_COMMON_ENTRIES;
		param += first_entry;
	} else {
		first_entry = 0;
	}
	for (i = first_entry; i < SGE_LEGACY_CONFIG_TABLE_ENTRIES; i++) {
		if (param->property_name != NULL) {
			error = of_property_read_u32_array(psd->i2c_client->dev.of_node, param->property_name, &param->value, 1);
			if (error < 0) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "property %s missing", param->property_name);
				return error;
			}
		}
		for (attempts = 0; attempts < max_attempts; attempts++) {
			error = (param->func)(psd, param->value);
			if (!error) {
				break;
			}
			mdelay(2);
		}
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "SGE configure failed");
			return error;
		}
		param++;
	}
	return 0;
}

static int sdd_configure_captouch(sdd_data_t *psd, struct sge_config_param *config_table, u32 num_entries)
{
	int i, error;
	for (i = 0; i < num_entries; i++) {
		struct sge_config_param *param = &config_table[i];
		if (param->property_name != NULL) {
			error = of_property_read_u32_array(psd->i2c_client->dev.of_node, param->property_name, &param->value, 1);
			if (error < 0) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "property %s missing", param->property_name);
				return error;
			}
		}
		error = (param->func)(psd, param->value);
		if (error) {
		    bb_log(BB_MOD_SDD, BB_LVL_ERR, "SGE configure failed on property %s", param->property_name);
		    return error;
		}
	}
	return 0;
}

int sdd_configure_sge(sdd_data_t *psd)
{
	if (of_property_read_bool(psd->i2c_client->dev.of_node, "captouch-style-button")) {
		return sdd_configure_captouch(psd, sdd_button_config_table, SGE_BUTTON_CONFIG_TABLE_ENTRIES);
	}
	if (of_property_read_bool(psd->i2c_client->dev.of_node, "captouch-style-touchpad")) {
		return sdd_configure_captouch(psd, sdd_touchpad_config_table, SGE_TOUCHPAD_CONFIG_TABLE_ENTRIES);
	}
	return sdd_configure_legacy_sge(psd);
}
