/*
 * Copyright (c) 2018-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/export.h>
#include <linux/delay.h>
#include <linux/time.h>
#include <linux/rtc.h>

#include "blackbox.h"
#include "sdd.h"
#include "sdd_rtc_api.h"
#include "sge_common.h"
#include <linux/rtc/sonos_rtc.h>

enum {
	SDD_RTC_TIME = 0,
	SDD_RTC_ALARM,
	SDD_RTC_NUM_CMDS
};

int sdd_rtc_psoc_inst = -1;

int sdd_rtc_set_time(time64_t time)
{
	sdd_data_t *psd = sdd_get_data(sdd_rtc_psoc_inst);
	int error;

	if (!psd) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "SDD data struct was not initialized, index %d", SGE_REG_IND_IDX_SET_RTC_TIME);
		return -ENODEV;
	}
	mutex_lock(&psd->ind_lock);
	if (psd->suspended) {
		bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: SDD/PSoC is suspended", __func__);
		error = -ENODEV;
		goto out;
	}
	error = __sdd_ind_write(psd, SGE_REG_IND_IDX_SET_RTC_TIME, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)&time, sizeof(time));

out:
	mutex_unlock(&psd->ind_lock);
	return error;
}

int sdd_rtc_get_time(time64_t *result)
{
	sdd_data_t *psd = sdd_get_data(sdd_rtc_psoc_inst);
	int error;

	if (!psd) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "SDD data struct was not initialized, index %d", SGE_REG_IND_IDX_GET_RTC_TIME);
		return -ENODEV;
	}
	mutex_lock(&psd->ind_lock);
	if (psd->suspended) {
		bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: SDD/PSoC is suspended", __func__);
		error = -ENODEV;
		goto out;
	}
	error = __sdd_ind_read(psd, SGE_REG_IND_IDX_GET_RTC_TIME, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)result, sizeof(*result));

out:
	mutex_unlock(&psd->ind_lock);
	return error;
}

static int sdd_rtc_alarm_disabled = 0;

int sdd_rtc_set_alarm(time64_t time)
{
	sdd_data_t *psd = sdd_get_data(sdd_rtc_psoc_inst);
	u64 val = time;
	int error;

	if (!psd) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "SDD data struct was not initialized, index %d", SGE_REG_IND_IDX_SET_RTC_ALARM);
		return -ENODEV;
	}
	mutex_lock(&psd->ind_lock);
	if (psd->suspended) {
		bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: SDD/PSoC is suspended", __func__);
		error = -EAGAIN;
		goto out;
	}
	if (sdd_rtc_alarm_disabled && (val == 0)) {
		bb_log_dbg(BB_MOD_SDD, "alarm already disabled");
		error = 0;
		goto out;
	}
	error = __sdd_ind_write(psd, SGE_REG_IND_IDX_SET_RTC_ALARM, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)&time, sizeof(time));
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: unable to set RTC alarm time: error %d", __func__, error);
		goto out;
	}
	if (val == 0) {
		sdd_rtc_alarm_disabled = 1;
	} else {
		sdd_rtc_alarm_disabled = 0;
	}
	printk("%s: 0x%016llx\n", __func__, val);

out:
	mutex_unlock(&psd->ind_lock);
	return error;
}

int sdd_rtc_get_alarm(time64_t *result)
{
	sdd_data_t *psd = sdd_get_data(sdd_rtc_psoc_inst);
	return sdd_ind_read(psd, SGE_REG_IND_IDX_GET_RTC_ALARM, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)result, sizeof(*result));
}

#define RTC_REPROGRAM_ATTEMPTS 200
void sdd_protect_rtc_content(int restore)
{
	static time64_t rtc_time;
	static time64_t rtc_alarm;
	static u64 start_jiffies;
	static int valid_time = 0;
	static int valid_alarm = 0;
	int error;
	int rtc_programming_retry_count = 0;
	time64_t alarm_check;
	u64 elapsed_msecs, elapsed_secs;

	if (!restore) {
		if ((error = sdd_rtc_get_time(&rtc_time)) < 0) {
			valid_time = 0;
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not read rtc time prior to psoc programming");
		} else {
			valid_time = 1;
		}
		if ((error = sdd_rtc_get_alarm(&rtc_alarm)) < 0) {
			valid_alarm = 0;
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not read rtc alarm prior to psoc programming");
		} else {
			valid_alarm = 1;
		}
		start_jiffies = get_jiffies_64();
	} else {
		if (!valid_time) {
			rtc_time = 0;
		}
		elapsed_msecs = (u64)jiffies_to_msecs((get_jiffies_64() - start_jiffies));
		elapsed_secs = (elapsed_msecs / 1000) + (((elapsed_msecs % 1000) > 500) ? 1 : 0);
		while ((error = sdd_rtc_set_time(rtc_time + elapsed_secs)) < 0) {
			mdelay(20);
			elapsed_msecs = (u64)jiffies_to_msecs((get_jiffies_64() - start_jiffies));
			elapsed_secs = (elapsed_msecs / 1000) + (((elapsed_msecs % 1000) > 500) ? 1 : 0);
			if (rtc_programming_retry_count++ >= RTC_REPROGRAM_ATTEMPTS) {
				break;
			}
		}
		if (rtc_programming_retry_count >= RTC_REPROGRAM_ATTEMPTS) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not restore rtc following psoc programming");
		}
		if (!valid_alarm || (rtc_time < (60*60*24*365))) {
			rtc_alarm = 0;
		}
		rtc_programming_retry_count = 0;
		error = sdd_rtc_get_alarm(&alarm_check);
		while (error || alarm_check != rtc_alarm) {
			if ((error = sdd_rtc_set_alarm(rtc_alarm)) < 0) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "Error [%d] attempting to restore alarm after programming psoc", error);
			}
			error = sdd_rtc_get_alarm(&alarm_check);
			if (rtc_programming_retry_count++ > RTC_REPROGRAM_ATTEMPTS) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not program alarm after %d tries: error %d", rtc_programming_retry_count, error);
				break;
			}
		}
	}
}

void sdd_rtc_alarm_recovery()
{
	int rtc_err;
	time64_t rtc_time, alarm_time;
	struct rtc_wkalrm alarm;
	int time_error, alarm_error;

	struct rtc_device *rtc = rtc_class_open("rtc0");
	time_error = sdd_rtc_get_time(&rtc_time);
	alarm_error = sdd_rtc_get_alarm(&alarm_time);
	if (!time_error && !alarm_error && (rtc_time > (60*60*24*365*48)) && (alarm_time > (rtc_time + 10))) {
		alarm.enabled = 1;
		rtc_time64_to_tm(alarm_time, &alarm.time);
		rtc_err = rtc_set_alarm(rtc, &alarm);
	} else {
		alarm_time = 0;
		sdd_rtc_set_alarm(alarm_time);
	}
}

static int rtc_show(struct seq_file *m, void *v)
{
	uintptr_t lopt = (uintptr_t) (m->private);
	u8 option = lopt;
	int error = 0;
	time64_t time;

	if (option == SDD_RTC_TIME) {
		error =  sdd_rtc_get_time(&time);
	} else if (option == SDD_RTC_ALARM) {
		error =  sdd_rtc_get_alarm(&time);
	}

	if (error) {
		seq_printf(m, "RTC access error\n");
		return 0;
	}

	seq_printf(m, "0x%016llx\n", time);

	return 0;
}

static ssize_t rtc_proc_write(struct file *file, const char __user *buffer, size_t count, loff_t *ppos)
{
	char buf[20];
	uintptr_t lopt = (uintptr_t) PDE_DATA(file_inode(file));
	u8 option = (u8) lopt;
	int error = 0;
	time64_t time;

	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	}

	buf[count] = '\0';

	if (option == SDD_RTC_TIME) {
		if ((error = kstrtoull(buf, 0, &time))) {
			bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: parsing error: %d.", __func__, error);
			return error;
		}
		error =  sdd_rtc_set_time(time);
	} else if (option == SDD_RTC_ALARM) {
		if ((error = kstrtoull(buf, 0, &time))) {
			bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: parsing error: %d.", __func__, error);
			return error;
		}
		error =  sdd_rtc_set_alarm(time);
	}

	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: sdd-rtc setter failure.", __func__);
		return -EFAULT;
	}

	return count;
}

static int rtc_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtc_show, PDE_DATA(inode));
}

static struct file_operations sdd_rtc_proc_ops = {
	.owner       = THIS_MODULE,
	.open        = rtc_proc_open,
	.write       = rtc_proc_write,
	.read        = seq_read,
	.llseek      = seq_lseek,
	.release     = single_release,
};

struct rtc_hw_ops sonos_rtc_hw_ops = {
	.sonos_read_time	= sdd_rtc_get_time,
	.sonos_set_time		= sdd_rtc_set_time,
	.sonos_read_alarm	= sdd_rtc_get_alarm,
	.sonos_set_alarm	= sdd_rtc_set_alarm,
};

int sdd_rtc_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create_data("driver/sdd-rtc-time", 0666, NULL, &sdd_rtc_proc_ops, (void *) SDD_RTC_TIME);
	if (entry == NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Unable to create driver/sdd-rtc-time");
		return -EIO;
	}

	entry = proc_create_data("driver/sdd-rtc-alarm", 0666, NULL, &sdd_rtc_proc_ops, (void*) SDD_RTC_ALARM);
	if (entry == NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Unable to create driver/sdd-rtc-alarm");
		return -EIO;
	}

	return 0;
}

int sdd_rtc_init(sdd_data_t *psd)
{
	if (sdd_rtc_psoc_inst == -1) {
		sdd_rtc_psoc_inst = psd->inst;
	} else {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "RTC already assigned to PSoC %d", sdd_rtc_psoc_inst);
		return -EPERM;
	}
	return sonos_rtc_register_ops(&sonos_rtc_hw_ops);
}

void sdd_rtc_remove(void)
{
	sonos_rtc_unregister_ops();
	remove_proc_entry("driver/sdd-rtc-alarm", NULL);
	remove_proc_entry("driver/sdd-rtc-time", NULL);
	sdd_rtc_psoc_inst = -1;
}

