/*
 * Copyright (c) 2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 */
#include <linux/init.h>
#include <linux/io.h>
#include <linux/imx_rpmsg.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/pm_qos.h>
#include <linux/rpmsg.h>
#include <psoc_rpmsg.h>
#include "blackbox.h"
#include "sdd.h"

#define I2C_RPMSG_MAX_BUF_SIZE			16
#define RPMSG_TIMEOUT				1000

#define I2C_RPMSG_CATEGORY			0x09
#define I2C_RPMSG_VERSION			0x0001
#define I2C_RPMSG_TYPE_REQUEST			0x00
#define I2C_RPMSG_TYPE_RESPONSE			0x01
#define I2C_RPMSG_COMMAND_READ			0x00
#define I2C_RPMSG_COMMAND_WRITE			0x01
#define I2C_RPMSG_PRIORITY			0x01

#define I2C_RPMSG_M_STOP			0x0200

struct psoc_rpmsg_info {
	struct rpmsg_device *rpdev;
	struct device *dev;
	struct psoc_rpmsg *msg;
	struct pm_qos_request pm_qos_req;
	struct completion cmd_complete;
	struct mutex lock;
};

static struct psoc_rpmsg_info psoc_rpmsg_info;

int psoc_send_message(struct psoc_rpmsg *msg,
			       struct psoc_rpmsg_info *info)
{
	int err;

	if (!info->rpdev) {
		dev_dbg(info->dev,
			 "rpmsg channel not ready, m4 image ready?\n");
		return -EINVAL;
	}

	mutex_lock(&info->lock);
	pm_qos_add_request(&info->pm_qos_req, PM_QOS_CPU_DMA_LATENCY, 0);

	msg->header.cate = IMX_RPMSG_PSOC;
	msg->header.major = IMX_RMPSG_MAJOR;
	msg->header.minor = IMX_RMPSG_MINOR;
	msg->header.type = 0;

	reinit_completion(&info->cmd_complete);

	err = rpmsg_send(info->rpdev->ept, (void *)msg,
			    sizeof(struct psoc_rpmsg));
	if (err) {
		dev_err(&info->rpdev->dev, "rpmsg_send failed: %d\n", err);
		goto err_out;
	}

	err = wait_for_completion_timeout(&info->cmd_complete,
					  msecs_to_jiffies(RPMSG_TIMEOUT));

	if (!err) {
		dev_err(&info->rpdev->dev, "rpmsg_send timeout!\n");
		err = -ETIMEDOUT;
		goto err_out;
	}

	err = 0;

err_out:
	pm_qos_remove_request(&info->pm_qos_req);
	mutex_unlock(&info->lock);

	dev_dbg(&info->rpdev->dev, "cmd:%d, resp:%d.\n",
		  msg->header.cmd, msg->ret_val);

	return err;
}

int sdd_psoc_program_notify(bool enter)
{
	struct psoc_rpmsg msg;
	int rc;

	msg.header.cmd = PSOC_RPMSG_PROGRAM;
	msg.data.byte = enter ? PSOC_RPMSG_PROGRAM_START : PSOC_RPMSG_PROGRAM_STOP;

	rc = psoc_send_message(&msg, &psoc_rpmsg_info);
	if (rc) {
		return -EINVAL;
	}
	return psoc_rpmsg_info.msg->ret_val;
}

int sdd_psoc_program_check(void)
{
	struct psoc_rpmsg msg;
	int rc;

	msg.header.cmd = PSOC_RPMSG_CHECK;
	msg.data.byte = 0;

	rc = psoc_send_message(&msg, &psoc_rpmsg_info);
	if (rc) {
		return -EINVAL;
	}
	return psoc_rpmsg_info.msg->data.rsp.data[0];
}

int sdd_read_touchpad_log(u8 *data, u32 len)
{
	struct psoc_rpmsg msg;
	int rc;

	msg.header.cmd = PSOC_RPMSG_TOUCHPAD_LOG;
	msg.data.byte = 0;

	rc = psoc_send_message(&msg, &psoc_rpmsg_info);
	if (rc) {
		return -EINVAL;
	}
	if (!psoc_rpmsg_info.msg->data.log.empty) {
	    memcpy(data, &psoc_rpmsg_info.msg->data.log.tle, len);
	}
	return psoc_rpmsg_info.msg->data.log.empty;
}

int sdd_disable_psoc_interrupt(sdd_data_t *psd)
{
	struct psoc_rpmsg msg;
	int rc;

	msg.header.cmd = PSOC_RPMSG_VAR_SET;
	msg.data.var.id = PSOC_RPMSG_VAR_ID_DISABLE_INTERRUPT;
	msg.data.var.psoc = psd->inst;
	rc = psoc_send_message(&msg, &psoc_rpmsg_info);
	if (rc) {
		return -EINVAL;
	}
	if (psoc_rpmsg_info.msg->ret_val != PSOC_RPMSG_SUCCESS) {
		return -ENXIO;
	}
	return 0;
}

int sdd_enable_psoc_interrupt(sdd_data_t *psd)
{
	struct psoc_rpmsg msg;
	int rc;

	msg.header.cmd = PSOC_RPMSG_VAR_SET;
	msg.data.var.id = PSOC_RPMSG_VAR_ID_ENABLE_INTERRUPT;
	msg.data.var.psoc = psd->inst;
	rc = psoc_send_message(&msg, &psoc_rpmsg_info);
	if (rc) {
		return -EINVAL;
	}
	if (psoc_rpmsg_info.msg->ret_val != PSOC_RPMSG_SUCCESS) {
		return -ENXIO;
	}
	return 0;
}

static int sdd_read_m33_var(sdd_data_t *psd, u32 id, u32 *data)
{
	struct psoc_rpmsg msg;
	int rc;

	msg.header.cmd = PSOC_RPMSG_VAR_GET;
	msg.data.var.id = id;
	msg.data.var.psoc = psd->inst;
	rc = psoc_send_message(&msg, &psoc_rpmsg_info);
	if (rc) {
		return -EINVAL;
	}
	if (psoc_rpmsg_info.msg->ret_val != PSOC_RPMSG_SUCCESS) {
		return -ENXIO;
	}
	*data = psoc_rpmsg_info.msg->data.var.val;
	return 0;
}

static int sdd_write_m33_var(sdd_data_t *psd, u32 id, u32 val)
{
	struct psoc_rpmsg msg;
	int rc;
	msg.header.cmd = PSOC_RPMSG_VAR_SET;
	msg.data.var.id = id;
	msg.data.var.psoc = psd->inst;
	msg.data.var.val = val;
	rc = psoc_send_message(&msg, &psoc_rpmsg_info);
	if (rc) {
		return -EINVAL;
	}
	if (psoc_rpmsg_info.msg->ret_val != PSOC_RPMSG_SUCCESS) {
		return -ENXIO;
	}
	return 0;
}

int sdd_read_swipe_distance(sdd_data_t *psd, u32 *dist)
{
	return sdd_read_m33_var(psd, PSOC_RPMSG_VAR_ID_SWIPE_DISTANCE, dist);
}

int sdd_write_swipe_distance(sdd_data_t *psd, u32 dist)
{
	return sdd_write_m33_var(psd, PSOC_RPMSG_VAR_ID_SWIPE_DISTANCE, dist);
}

int sdd_read_press_hold(sdd_data_t *psd, u32 *msec)
{
	return sdd_read_m33_var(psd, PSOC_RPMSG_VAR_ID_PRESS_HOLD_TIME, msec);
}

int sdd_write_press_hold(sdd_data_t *psd, u32 msec)
{
	return sdd_write_m33_var(psd, PSOC_RPMSG_VAR_ID_PRESS_HOLD_TIME, msec);
}

int sdd_read_double_tap(sdd_data_t *psd, u32 *msec)
{
	return sdd_read_m33_var(psd, PSOC_RPMSG_VAR_ID_DOUBLE_TAP_TIME, msec);
}

int sdd_write_double_tap(sdd_data_t *psd, u32 msec)
{
	return sdd_write_m33_var(psd, PSOC_RPMSG_VAR_ID_DOUBLE_TAP_TIME, msec);
}

int sdd_ind_read_rpmsg(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len)
{
	struct psoc_rpmsg msg;
	struct psoc_indirect_cmd_msg *icm = &msg.data.cmd;
	int rc;

	msg.header.cmd = PSOC_RPMSG_INDIRECT_READ;
	icm->psoc = psd->inst;
	icm->index = index;
	icm->flags = flags;
	icm->len = len;
	icm->param = param;
	rc = psoc_send_message(&msg, &psoc_rpmsg_info);
	if (rc) {
		return -EINVAL;
	}
	if (psoc_rpmsg_info.msg->ret_val != PSOC_RPMSG_SUCCESS) {
		return -ENXIO;
	}
	memcpy(data, psoc_rpmsg_info.msg->data.rsp.data, len);
	return 0;
}

int sdd_ind_write_rpmsg(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len)
{
	struct psoc_rpmsg msg;
	struct psoc_indirect_cmd_msg *icm = &msg.data.cmd;
	int rc;

	if (len > sizeof(icm->wrdata)) {
		return -EINVAL;
	}
	msg.header.cmd = PSOC_RPMSG_INDIRECT_WRITE;
	icm->psoc = psd->inst;
	icm->index = index;
	icm->flags = flags;
	icm->len = len;
	icm->param = param;
	memcpy(icm->wrdata, data, len);
	rc = psoc_send_message(&msg, &psoc_rpmsg_info);
	if (rc) {
		return -EINVAL;
	}
	if (psoc_rpmsg_info.msg->ret_val != PSOC_RPMSG_SUCCESS) {
		return -ENXIO;
	}
	return 0;
}

struct sdd_indirect_ops sdd_indirect_ops_rpmsg = {
	.read = sdd_ind_read_rpmsg,
	.write = sdd_ind_write_rpmsg,
	.read_special = sdd_ind_read_rpmsg
};

void sdd_ind_ops_rpmsg(sdd_data_t *psd)
{
	psd->indirect_ops = &sdd_indirect_ops_rpmsg;
}

static struct rpmsg_device_id psoc_rpmsg_id_table[] = {
	{ .name	= "rpmsg-psoc-channel" },
	{ },
};

static int psoc_rpmsg_probe(struct rpmsg_device *rpdev)
{
	int ret = 0;

	dev_info(&rpdev->dev, "new channel: 0x%x -> 0x%x!\n",
		rpdev->src, rpdev->dst);

	psoc_rpmsg_info.rpdev = rpdev;
	mutex_init(&psoc_rpmsg_info.lock);

	init_completion(&psoc_rpmsg_info.cmd_complete);

	return ret;
}

static int psoc_rpmsg_cb(struct rpmsg_device *rpdev, void *data, int len,
			void *priv, u32 src)
{
	struct psoc_rpmsg *msg = (struct psoc_rpmsg *)data;

	dev_dbg(&rpdev->dev, "get from:%d: cmd:%d, resp:%d.\n",
			src, msg->header.cmd, msg->ret_val);

	psoc_rpmsg_info.msg = msg;

	complete(&psoc_rpmsg_info.cmd_complete);

	return 0;
}

static void psoc_rpmsg_remove(struct rpmsg_device *rpdev)
{
	dev_info(&rpdev->dev, "psoc rpmsg driver is removed\n");
}

static struct rpmsg_driver psoc_rpmsg_driver = {
	.drv.name	= "psoc_rpmsg",
	.drv.owner	= THIS_MODULE,
	.id_table	= psoc_rpmsg_id_table,
	.probe		= psoc_rpmsg_probe,
	.callback	= psoc_rpmsg_cb,
	.remove		= psoc_rpmsg_remove,
};

int sdd_rpmsg_init_done = 0;

int sdd_rpmsg_init(void)
{
	int error = 0;
	if (!sdd_rpmsg_init_done) {
		memset(&psoc_rpmsg_info, 0, sizeof(psoc_rpmsg_info));
		error = register_rpmsg_driver(&psoc_rpmsg_driver);
		sdd_rpmsg_init_done = !error;
	}
	return error;
}

void sdd_rpmsg_exit(void)
{
    if (sdd_rpmsg_init_done) {
	unregister_rpmsg_driver(&psoc_rpmsg_driver);
        sdd_rpmsg_init_done = 0;
    }
}
