/*
 * Copyright (c) 2014-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include "linux/module.h"
#include <linux/moduleparam.h>
#include <asm/irq.h>
#include <linux/kernel.h>
#include <linux/jiffies.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include "mdp.h"
#include <asm/uaccess.h>
#include <linux/i2c.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/types.h>
#include "blackbox.h"
#include "sdd.h"
#include "sge_common.h"

#define MAX_WRITE_BUF_LEN  4096
#define REC_MAX_CSV_LINE_LEN 32
#define CMD_STR_LEN_MAX 40
#define IN_BUFFER_LEN (MAX_WRITE_BUF_LEN+REC_MAX_CSV_LINE_LEN)
#define SDD_REC_SEND_SAMPS      "send\n"
#define SDD_REC_GET_SAMPS   "receive\n"
#define SDD_REC_RUN_PLAYBACK    "play\n"
#define SDD_REC_ON_TOUCH        "record\n"
#define SDD_AUTO_XFER_PREF      "auto_xfer="
#define SDD_AUTO_TRIG_PREF      "auto_trig="

enum tag_token_labels {
        CSV_TOKEN_REC_FRM = 0,
        CSV_TOKEN_REC_IDX,
        CSV_TOKEN_REC_LPOS,
        CSV_TOKEN_REC_RPOS,
        CSV_TOKEN_REC_AC,
        CSV_TOKEN_REC_HC,
        CSV_NUM_TOKENS
};

static int rec_open(struct inode *inode, struct file *file);
static int rec_cmd(sdd_data_t *psd, u32 rec_cmd);
static int rec_show(struct seq_file *m, void *v);
static void rec_show_status(struct seq_file *m, sdd_data_t *psd);
static int rec_put_psoc_buffer(sdd_data_t *psd);
static int rec_get_psoc_buffer(sdd_data_t *psd);
static void rec_show_local_buffer(struct seq_file *m, sdd_data_t *psd) ;
static ssize_t rec_csvparse_hndlr(struct file *file, const char __user * buffer, size_t count, loff_t *data);
static ssize_t rec_control_wr_hndlr(struct file *file, const char __user * buffer, size_t count, loff_t *data);

static struct file_operations rec_control_ops = {
        .owner       = THIS_MODULE,
        .open        = rec_open,
        .write       = rec_control_wr_hndlr,
        .read        = seq_read,
        .llseek      = seq_lseek,
        .release     = single_release,
};

static struct file_operations rec_cap_ops = {
        .owner       = THIS_MODULE,
        .open        = rec_open,
        .write       = rec_csvparse_hndlr,
        .read        = seq_read,
        .llseek      = seq_lseek,
        .release     = single_release,
};

static struct sdd_rec_file {
        char *filename;
        void *option;
        struct file_operations *ops;
} sdd_rec_files[] = {
        {"capture", (void*) SGE_REG_REC_SHOW_LOCAL_BUF, &rec_cap_ops},
        {"control", (void*) SGE_REG_REC_SHOW_STATUS, &rec_control_ops},
        {NULL, 0, NULL}
};

struct proc_dir_entry *sdd_rec_parent_proc;
static fsm_input rd_buffer[NUM_REC_SRAM_SAMPLES];
static u16 wr_samp_idx = 0;
static char in_buffer[IN_BUFFER_LEN];
static u8 auto_xfer = 0;

static int rec_cmd(sdd_data_t *psd, u32 rec_cmd)
{
        int error, poll_count;
        u8 byte, cmd=rec_cmd;

        error = sdd_write_sge_byte(psd, SGE_REG_REC_CONTROL, cmd);
        if (!error) {
                for (poll_count = 0; poll_count <= 99; poll_count++) {
                        mdelay(1);
                        error = sdd_read_sge_byte(psd, SGE_REG_REC_CONTROL, &byte);
                        if (!error && (byte == SGE_REG_REC_NONE)) {
                                return 0;
                        }
                }
        }
        return -EIO;
}

static int rec_put_psoc_buffer(sdd_data_t *psd)
{
        int error = 0;
        int i, frm_idx;
        u8* rd_sample;
        u8 offset = 0;

        error = sdd_write_sge_byte(psd, SGE_REG_REC_ON_TOUCH, 0);
        if (error) {
                bb_log(BB_MOD_SDD, BB_LVL_INFO, "unable to write REC_ON_TOUCH reg.\n");
                return error;
        }

        for (frm_idx = 0; frm_idx < NUM_REC_FRAMES; frm_idx++) {
                error = sdd_write_sge_byte(psd, SGE_REG_REC_FRM_IDX, frm_idx);
                if (error) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "sdd_rec: playback, unable to write frame idx\n");
                        return error;
                }

                offset = SGE_REG_REC_FSM_DATA;
                for (i = 0; i < NUM_REC_SAMPS_PER_FRAME; i++, offset += sizeof(fsm_input)) {
                        rd_sample = (u8 *) &rd_buffer[i+(frm_idx*NUM_REC_SAMPS_PER_FRAME)];
                        error = sdd_write_sge_block(psd, offset, sizeof(fsm_input), (const u8*) rd_sample);
                        if (error) {
                                bb_log(BB_MOD_SDD, BB_LVL_INFO, "sdd_rec: read block error\n");
                                return error;
                        }

                }

                error = rec_cmd(psd, SGE_REG_REC_SET_FRAME);
                if (error) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "sdd_rec: unable to issue REC_SET_FRAME\n");
                        return error;
                }
        }

        return 0;
}

static void rec_show_status(struct seq_file *m, sdd_data_t *psd)
{
        int error = 0;
        u8 rec_stats;
        u8 rec_playing;
        u8 rec_control;
        u8 auto_trig;

        error = sdd_read_sge_byte(psd, SGE_REG_REC_CONTROL, &rec_control);
        if (error) {
                seq_printf(m, "Unable to read register: SGE_REG_REC_CONTROL\n");
                return;
        }

        error = sdd_read_sge_byte(psd, SGE_REG_REC_ON_TOUCH, &rec_stats);
        if (error) {
                seq_printf(m, "Unable to read register: SGE_REG_REC_ON_TOUCH\n");
                return;
        }

        error = sdd_read_sge_byte(psd, SGE_REG_REC_PLAYING, &rec_playing);
        if (error) {
                seq_printf(m, "Unable to read register: SGE_REG_REC_PLAYING\n");
                return;
        }

        auto_trig = (rec_stats == 2) ? 1 : 0;

        seq_printf(m, "PSoC Register Values:\n");
        seq_printf(m, "REC_CONTROL:\t%2x\n", rec_control);
        seq_printf(m, "REC_ON_TOUCH:\t%2x\n", rec_stats);
        seq_printf(m, "REC_PLAYING:\t%2x\n", rec_playing);
        seq_printf(m, "Local Mode Settings:\n");
        seq_printf(m, "auto_trig:\t%2x\n", auto_trig);
        seq_printf(m, "auto_xfer:\t%2x\n", auto_xfer);
}

static void rec_show_local_buffer(struct seq_file *m, sdd_data_t *psd)
{
        int i;
        int frm_idx = 0;
        int subsamp_idx = 0;

        seq_printf(m, "frm-idx,sample-idx,lpos,rpos,touch-bitmap,halfcap-bitmap\n");

        for (i = 0; i < NUM_REC_SRAM_SAMPLES; i++) {
                frm_idx = i / NUM_REC_SAMPS_PER_FRAME;
                subsamp_idx = i % NUM_REC_SAMPS_PER_FRAME;
                seq_printf(m, "%2d,%2d,%2x,%2x,%04x,%04x\n", frm_idx, subsamp_idx, rd_buffer[i].spos_left, rd_buffer[i].spos_right,rd_buffer[i].ac_bmp, rd_buffer[i].hc_bmp);
        }
}

static int rec_get_psoc_buffer(sdd_data_t *psd)
{
        int error, i, frm_idx;
        fsm_input rd_sample;
        u8 offset = 0;

        memset((void*)&rd_buffer, 0, sizeof(rd_buffer));


        for (frm_idx = 0; frm_idx < NUM_REC_FRAMES; frm_idx++) {
                error = sdd_write_sge_byte(psd, SGE_REG_REC_FRM_IDX, frm_idx);
                if (error) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "unable to write frame idx\n");
                        return error;
                }

                error = rec_cmd(psd, SGE_REG_REC_GET_FRAME);
                if (error) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "unable to issue REC_GET_FRAME\n");
                        return error;
                }


                offset = SGE_REG_REC_FSM_DATA;
                for (i = 0; i < NUM_REC_SAMPS_PER_FRAME; i++, offset += sizeof(rd_sample)) {
                        error = sdd_read_sge_block(psd, offset, sizeof(rd_sample), (char *)&rd_sample);
                        if (error) {
                                bb_log(BB_MOD_SDD, BB_LVL_INFO, "read block error\n");
                                return error;
                        }

                        rd_buffer[i+(frm_idx*NUM_REC_SAMPS_PER_FRAME)] = rd_sample;
                }
        }

        return 0;
}

static int rec_show(struct seq_file *m, void *v)
{
        sdd_data_t *psd = sdd_get_data(0);
        uintptr_t lopt = (uintptr_t)(m->private);
        u8 option = lopt;
        int error;

        if (option == SGE_REG_REC_NONE) {
        } else if (option == SGE_REG_REC_SHOW_LOCAL_BUF) {
                if (auto_xfer) {
                        error = rec_get_psoc_buffer(psd);
                        if (error) {
                                bb_log(BB_MOD_SDD, BB_LVL_INFO, "unable to get samples from PSOC SRAM.\n");
                                return error;
                        }
                }

                rec_show_local_buffer(m, psd);
        } else if (option == SGE_REG_REC_SHOW_STATUS) {
                rec_show_status(m, psd);
        } else {
                error = rec_cmd(psd, option);
                if (error) {
                        seq_printf(m, "write REC_CONTROL error\n");
                        return error;
                }
        }
        return 0;
}

static ssize_t rec_csvparse_hndlr(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
        int error;
        char *start;
        char *pch;
        char *col_start;
        char *rec_start;
        char *tok_ptrs[CSV_NUM_TOKENS];
        static char remainder[REC_MAX_CSV_LINE_LEN] = {0};

        sdd_data_t *psd = sdd_get_data(0);
        u32 chars_parsed = 0;
        u32 frm_idx = 0;
        u32 rec_idx = 0;
        u32 parsed_val = 0;
        u8 remainder_len = 0;
        u16 samp_idx = 0;
        u8 tok_idx = 0;

        if (!wr_samp_idx) {
                bb_log(BB_MOD_SDD, BB_LVL_INFO, "parsing playback csv file\n");
                memset(remainder, 0, REC_MAX_CSV_LINE_LEN);
                memset(in_buffer, 0, IN_BUFFER_LEN);
        } else {
                remainder_len = strlen(remainder);
                if (remainder_len) {
                        memcpy(in_buffer, remainder, remainder_len);
                }
        }

        pch = in_buffer + remainder_len;

        if (count > (sizeof(in_buffer)) ) {
                bb_log(BB_MOD_SDD, BB_LVL_INFO, "rec_csvparse_hndlr: count >= buffer size\n");
                return -EIO;
        }

        if (copy_from_user(pch, buffer, count)) {
                bb_log(BB_MOD_SDD, BB_LVL_INFO, "rec_csvparse_hndlr: unable to copy from user space\n");
                return -EFAULT;
        }

        pch = in_buffer;
        rec_start = strsep(&pch, "\n");
        start = rec_start;

        if (!pch) {
                bb_log(BB_MOD_SDD, BB_LVL_INFO, "no delimiter found\n");
                return -EFAULT;
        }

        if (!wr_samp_idx) {
                rec_start = strsep(&pch, "\n");
        }

        while (pch != NULL) {
                col_start = strsep(&rec_start, ",");
                tok_idx = 0;
                tok_ptrs[tok_idx++] = strim(col_start);
                while (rec_start != NULL && tok_idx < CSV_NUM_TOKENS) {
                        col_start = strsep(&rec_start, ",");
                        tok_ptrs[tok_idx++] = strim(col_start);
                }

                if ((error = kstrtouint(tok_ptrs[CSV_TOKEN_REC_FRM], 10, &frm_idx))) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "Unable to parse record: %d, token: %d val: %s\n", wr_samp_idx,
                               CSV_TOKEN_REC_FRM, tok_ptrs[CSV_TOKEN_REC_FRM]);
                        return error;
                }

                if ((error = kstrtouint(tok_ptrs[CSV_TOKEN_REC_IDX], 10, &rec_idx))) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "Unable to parse record: %d, token: %d val: %s\n", wr_samp_idx,
                               CSV_TOKEN_REC_IDX, tok_ptrs[CSV_TOKEN_REC_IDX]);
                        return error;
                }

                if (frm_idx >= NUM_REC_FRAMES) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "Invalid Frame Index specified: %u\n", frm_idx);
                        return -EIO;
                }

                if (rec_idx >= NUM_REC_SAMPS_PER_FRAME) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "Invalid Record Index specified: %u\n", rec_idx);
                        return -EIO;
                }

                if (!wr_samp_idx && ((rec_idx != 0) || frm_idx != 0)) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "first record specified was not zero: rec_idx: %u\n", rec_idx);
                        return -EIO;
                }

                if (wr_samp_idx > 299) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "csv data contains too many records: rec_idx: %u frm_idx: %u\n", rec_idx, frm_idx);
                        return -EIO;
                }

                samp_idx = rec_idx+(frm_idx*NUM_REC_SAMPS_PER_FRAME);

                if ((error = kstrtouint(tok_ptrs[CSV_TOKEN_REC_LPOS], 16, &parsed_val))) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "Invalid lpos specified: %s rec_idx: %u frm_idx: %u\n", tok_ptrs[CSV_TOKEN_REC_LPOS], rec_idx, frm_idx);
                        return error;
                }
                rd_buffer[samp_idx].spos_left = (uint8_t)parsed_val;

                if ((error = kstrtouint(tok_ptrs[CSV_TOKEN_REC_RPOS], 16, &parsed_val))) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "Invalid rpos specified: %s rec_idx: %u frm_idx: %u\n", tok_ptrs[CSV_TOKEN_REC_RPOS], rec_idx, frm_idx);
                        return error;
                }
                rd_buffer[samp_idx].spos_right = (uint8_t)parsed_val;

                if ((error = kstrtouint(tok_ptrs[CSV_TOKEN_REC_AC], 16, &parsed_val))) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "Invalid AC bitmap specified: %s rec_idx: %u frm_idx: %u\n", tok_ptrs[CSV_TOKEN_REC_AC], rec_idx, frm_idx);
                        return error;
                }
                rd_buffer[samp_idx].ac_bmp = (uint16_t) parsed_val;

                if ((error = kstrtouint(tok_ptrs[CSV_TOKEN_REC_HC], 16, &parsed_val))) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "Invalid HC bitmap specified: %s rec_idx: %u frm_idx: %u\n", tok_ptrs[CSV_TOKEN_REC_HC], rec_idx, frm_idx);
                        return error;
                }
                rd_buffer[samp_idx].hc_bmp = (uint16_t) parsed_val;

                rec_start = strsep(&pch, "\n");
                wr_samp_idx++;
        }

        chars_parsed = rec_start - start;
        if (count > chars_parsed) {
                memcpy(remainder, in_buffer+chars_parsed, (size_t)(count - chars_parsed));
        } else if (count <= chars_parsed) {
                memset(remainder, 0, REC_MAX_CSV_LINE_LEN);
        }

        if (wr_samp_idx == 300) {
                bb_log(BB_MOD_SDD, BB_LVL_INFO, "parsing playback file succeeded.\n");
                if (auto_xfer) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "auto_xfer mode is enabled, sending samples to PSoC.");
                        error = rec_put_psoc_buffer(psd);
                        if (error) {
                                bb_log(BB_MOD_SDD, BB_LVL_INFO, "unable to send samples to PSoC SRAM.\n");
                                return error;
                        }

                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "buffer send success.\n");
                }
        }

        return count;
}

static ssize_t rec_control_wr_hndlr(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
        char buf[CMD_STR_LEN_MAX];
        sdd_data_t *psd = sdd_get_data(0);
        int error = 0;

        memset(buf, 0, CMD_STR_LEN_MAX);

        if (count >= sizeof(buf))
                return -EIO;
        if (copy_from_user(buf, buffer, count))
                return -EFAULT;

        if (strncmp(buf, SDD_REC_SEND_SAMPS, CMD_STR_LEN_MAX) == 0) {

                error = rec_put_psoc_buffer(psd);
                if (error) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "unable to send samples to PSoC SRAM.\n");
                        return -EIO;
                }

                bb_log(BB_MOD_SDD, BB_LVL_INFO, "buffer send success.\n");

        } else if (strncmp(buf, SDD_REC_GET_SAMPS, CMD_STR_LEN_MAX) == 0) {

                error = rec_get_psoc_buffer(psd);
                if (error) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "unable to get samples from PSOC SRAM.\n");
                        return -EIO;
                }

                bb_log(BB_MOD_SDD, BB_LVL_INFO, "buffer receive success.\n");

        } else if (strncmp(buf, SDD_REC_RUN_PLAYBACK, CMD_STR_LEN_MAX) == 0) {

                error = rec_cmd(psd,SGE_REG_REC_RUN_PLAYBACK);
                if (error) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "unable to run playback\n");
                        return -EIO;
                }

                bb_log(BB_MOD_SDD, BB_LVL_INFO, "playback starting, %d samples\n", NUM_REC_SRAM_SAMPLES);

        } else if (strncmp(buf, SDD_REC_ON_TOUCH, CMD_STR_LEN_MAX) == 0) {

                u8 rec_on_touch;

                error = sdd_read_sge_byte(psd, SGE_REG_REC_ON_TOUCH, &rec_on_touch);
                if (error) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO,  "Unable to read register: SGE_REG_REC_ON_TOUCH\n");
                        return -EIO;
                }

                if (rec_on_touch == 0) {
                        error = sdd_write_sge_byte(psd, SGE_REG_REC_ON_TOUCH, 1);
                        if (error) {
                                bb_log(BB_MOD_SDD, BB_LVL_INFO, "unable to write SGE_REG_REC_ON_TOUCH");
                                return -EIO;
                        }
                } else if (rec_on_touch == 2) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "auto trigger mode is already enabled\n");
                }

                bb_log(BB_MOD_SDD, BB_LVL_INFO, "recording on touch, %d samples\n", NUM_REC_SRAM_SAMPLES);

        } else if (strncmp(buf, SDD_AUTO_XFER_PREF, strlen(SDD_AUTO_XFER_PREF)) == 0) {
                u8 pref_len = strlen(SDD_AUTO_XFER_PREF);
                u32 enabled = 0;

                error = kstrtouint(buf+pref_len, 10, &enabled);

                if (error || (enabled != 0 && enabled != 1)) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "Unable to parse value for: %s%s",SDD_AUTO_XFER_PREF, buf+pref_len);
                        return -EIO;
                }

                if (enabled) {
                        auto_xfer = 1;
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "auto xfer mode is now enabled.\n");
                } else {
                        auto_xfer = 0;
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "auto xfer mode is now disabled.\n");
                }
        } else if (strncmp(buf, SDD_AUTO_TRIG_PREF, strlen(SDD_AUTO_TRIG_PREF)) == 0) {
                u8 pref_len = strlen(SDD_AUTO_TRIG_PREF);
                u32 enabled = 0;
                char *result;

                error = kstrtouint(buf+pref_len, 10, &enabled);

                if (error || (enabled != 0 && enabled != 1)) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "Unable to parse value for: %s%s",SDD_AUTO_TRIG_PREF, buf+pref_len);
                        return -EIO;
                }

                enabled = enabled ? 2 : 0;
                result = enabled ? "enabled" : "disabled";

                error = sdd_write_sge_byte(psd, SGE_REG_REC_ON_TOUCH, enabled);
                if (error) {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "unable to write REC_ON_TOUCH reg.\n");
                        return -EIO;
                } else {
                        bb_log(BB_MOD_SDD, BB_LVL_INFO, "auto trig mode is now %s.\n", result);
                }
        } else {
                bb_log(BB_MOD_SDD, BB_LVL_INFO, "no command received: %s\n", buf);
        }

        return count;
}

static int rec_open(struct inode *inode, struct file *file)
{
        wr_samp_idx = 0;
        return single_open(file, rec_show, PDE_DATA(inode));
}

int sdd_rec_init(sdd_data_t *psd)
{
        struct proc_dir_entry *entry;
        int i;

        sdd_rec_parent_proc = proc_mkdir("driver/sdd-rec",0);
        if (!sdd_rec_parent_proc) {
                bb_log(BB_MOD_SDD, BB_LVL_ERR, "Couldn't create /proc/driver/sdd-rec\n");
                return -EIO;
        }

        for (i = 0; sdd_rec_files[i].filename != NULL; i++) {
                entry = proc_create_data(sdd_rec_files[i].filename, 0666, sdd_rec_parent_proc,
                                         sdd_rec_files[i].ops, sdd_rec_files[i].option);
                if (!entry) {
                        bb_log(BB_MOD_SDD, BB_LVL_ERR, "unable to create REC file %s\n", sdd_rec_files[i].filename);
                        return -EIO;
                }
        }

        if( !entry ) {
                bb_log(BB_MOD_SDD, BB_LVL_ERR, "unable to create REC file driver/sdd-rec/playback\n");
                return( -EIO );
        }
        return 0;
}

void sdd_rec_remove(void)
{
        int i;
        for (i = 0; sdd_rec_files[i].filename != NULL; i++) {
                remove_proc_entry(sdd_rec_files[i].filename, sdd_rec_parent_proc);
        }
        remove_proc_entry("driver/sdd-rec", NULL);
}

