/*
 * Copyright (c) 2015-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/slab.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/semaphore.h>
#include "blackbox.h"
#include "sdd.h"

#define SDD_LOG_NUM_ENTRIES   49

static DEFINE_SEMAPHORE(sdd_log_lock);

static inline u32 sdd_log_next(struct sdd_log *sl, u32 current_index)
{
    return ((current_index + 1) % SDD_LOG_NUM_ENTRIES);
}

static inline u32 sdd_log_next_head(struct sdd_log *sl)
{
    return sdd_log_next(sl, sl->head);
}

static inline u32 sdd_log_next_tail(struct sdd_log *sl)
{
    return sdd_log_next(sl, sl->tail);
}

static inline int sdd_log_empty(struct sdd_log *sl)
{
    return (sl->head == sl->tail);
}

static inline int sdd_log_full(struct sdd_log *sl)
{
    return (sdd_log_next_head(sl) == sl->tail);
}

void sdd_log_event(struct sdd_log *sl, struct sdd_grr_change_entry *sgce)
{
    if (sl != NULL) {
        down(&sdd_log_lock);
        sl->entries[sl->head] = *sgce;
        if (sdd_log_full(sl)) {
            sl->tail = sdd_log_next_tail(sl);
        }
        sl->head = sdd_log_next_head(sl);
        up(&sdd_log_lock);
    }
}

int sdd_log_show(struct seq_file *m, void *v)
{
    sdd_data_t *psd = (sdd_data_t *)(m->private);
    struct sdd_log *sl = psd->grr_event_log;

    down(&sdd_log_lock);
    while (!sdd_log_empty(sl)) {
        struct sdd_grr_change_entry *sgce = &sl->entries[sl->tail];
        seq_printf(m, "GRR: changed from %04x to %04x (%ums) (intr %ums, i2c %ums)\n",
                   sgce->prev_grr, sgce->new_grr, sgce->prev_msec, sgce->intr_msec, sgce->i2c_msec);
        sl->tail = sdd_log_next_tail(sl);
    }
    seq_printf(m, "\n");
    up(&sdd_log_lock);
    return 0;
}

static int sdd_log_open(struct inode *inode, struct file *file)
{
    return single_open(file, sdd_log_show, PDE_DATA(inode));
}

static struct file_operations sdd_log_proc_operations = {
    .owner       = THIS_MODULE,
    .open        = sdd_log_open,
    .write       = NULL,
    .read        = seq_read,
    .llseek      = seq_lseek,
    .release     = single_release,
};

struct sdd_log * sdd_init_log(sdd_data_t *psd)
{
    struct sdd_log *sl;
    u32 size = sizeof(struct sdd_log) + (sizeof(struct sdd_grr_change_entry) * (SDD_LOG_NUM_ENTRIES-1));
    struct proc_dir_entry *entry;

    sl = kmalloc(size, GFP_KERNEL);
    if (sl == NULL) {
        bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: memory alloc failed", __func__);
        return NULL;
    }

    sl->head = sl->tail = 0;
    entry = proc_create_data("driver/grr", 0666, NULL, &sdd_log_proc_operations, psd);
    if (entry == NULL) {
        kfree(sl);
        return NULL;
    }
    return sl;
}

void sdd_log_remove(void)
{
    remove_proc_entry("driver/grr", NULL);
}
