/*
 * Copyright (c) 2014-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include "linux/module.h"
#include <linux/moduleparam.h>
#include <asm/irq.h>
#include <linux/kernel.h>
#include <linux/jiffies.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include "mdp.h"
#include <asm/uaccess.h>
#include <linux/i2c.h>
#include <linux/delay.h>
#include <linux/types.h>
#include "blackbox.h"
#include "sdd.h"
#include "sge_common.h"

static int sdd_ind_cmd(sdd_data_t *psd, u8 index, u8 flags)
{
	int error, poll_count;
	u8 reg;
	const size_t max_read_attempts = 99;

	error = sdd_write_sge_byte(psd, SGE_REG_IND_FLAGS, flags);
	if (error) {
		goto out;
	}

	error = sdd_write_sge_byte(psd, SGE_REG_IND_IDX, index);
	if (error) {
		goto out;
	}

	for (poll_count = 0; poll_count <= max_read_attempts; poll_count++) {
		mdelay(1);
		error = sdd_read_sge_byte(psd, SGE_REG_IND_IDX, &reg);
		if (!error) {
			if (reg == SGE_REG_IND_IDX_NONE) {
				break;
			}

			if (reg == SGE_REG_IND_IDX_UNSUPPORTED) {
				error = -EINVAL;
				break;
			}

			if  (max_read_attempts == poll_count) {
				error = -EIO;
				break;
			}
		}
	}

out:
	return error;
}

static int sdd_ind_read_nocop(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len)
{
	int error;

	if (flags & SGE_REG_IND_FLAGS_SENSOR) {
		u8 sensor = param;
		error = sdd_write_sge_byte(psd, SGE_REG_IND_DATA, sensor);
		if (error) {
		    bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: i2c write failed, error = %d", __func__, error);
		    goto out;
		}
	}
	error = sdd_ind_cmd(psd, index, flags);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: cmd error %d, index %d", __func__, error, index);
		goto out;
	}

	error = sdd_read_sge_block(psd, SGE_REG_IND_DATA, len, data);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: data error %d, index %d", __func__, error, index);
		goto out;
	}

out:
	return error;
}

static int sdd_ind_write_nocop(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len)
{
	int error;

	error = sdd_write_sge_block(psd, SGE_REG_IND_DATA, len, data);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: data error %d, index %d", __func__, error, index);
		goto out;
	}

	error = sdd_ind_cmd(psd, index, flags | SGE_REG_IND_FLAGS_WRITE);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: cmd error %d, index %d", __func__, error, index);
		goto out;
	}

out:
	return error;
}

static int sdd_ind_read_special_nocop(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len)
{
	int error, prev_touch;

	error = sdd_disable_touch(psd, &prev_touch);
	if (error) {
		goto out;
	}

	error = sdd_ind_cmd(psd, index, flags);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: IND command failed error = %d", __func__, error);
		goto out;
	}

	mdelay(100);
	error = sdd_update_sge_baselines(psd);
	if (error) {
		goto out;
	}
	error = sdd_set_touch(psd, prev_touch);
	if (error) {
		goto out;
	}

	error = sdd_read_sge_block(psd, SGE_REG_IND_DATA, len, (u8 *)data);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: unable to read CMOD cap value.", __func__);
		goto out;
	}

out:
	return error;
}

int __sdd_ind_read(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len)
{
	if (psd->suspended) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "SDD/PSoC is suspended, index %d", index);
		return -EAGAIN;
	}
	if ((psd->indirect_ops == NULL) || (psd->indirect_ops->read == NULL)) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: psoc%d missing indirect ops %p, index %d",
			__func__, psd->inst, psd->indirect_ops, index);
		return -EFAULT;
	}
	return psd->indirect_ops->read(psd, index, flags, param, data, len);
}

int sdd_ind_read(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len)
{
	int error;
	if (!psd) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "SDD data struct was not initialized, index %d", index);
		return -ENODEV;
	}

	mutex_lock(&psd->ind_lock);
	error = __sdd_ind_read(psd, index, flags, param, data, len);
	mutex_unlock(&psd->ind_lock);
	return error;
}

int __sdd_ind_write(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len)
{
	if (psd->suspended) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "SDD/PSoC is suspended, index %d", index);
		return -EAGAIN;
	}
	if ((psd->indirect_ops == NULL) || (psd->indirect_ops->write == NULL)) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: psoc%d missing indirect ops, index %d", __func__, psd->inst, index);
		return -EFAULT;
	}
	return psd->indirect_ops->write(psd, index, flags, param, data, len);
}

int sdd_ind_write(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len)
{
	int error;
	if (!psd) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "SDD data struct was not initialized, index %d", index);
		return -ENODEV;
	}

	mutex_lock(&psd->ind_lock);
	error = __sdd_ind_write(psd, index, flags, param, data, len);
	mutex_unlock(&psd->ind_lock);
	return error;
}

int sdd_ind_read_special(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len)
{
	int error;
	if (!psd) {
	    bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: SDD data struct was not initialized, index %d", __func__, index);
		return -ENODEV;
	}

	mutex_lock(&psd->ind_lock);
	if (psd->suspended) {
	    bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: SDD/PSoC is suspended, index %d", __func__, index);
		error = -EAGAIN;
		goto out;
	}
	if ((psd->indirect_ops == NULL) || (psd->indirect_ops->read_special == NULL)) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: psoc%d missing indirect ops, index %d", __func__, psd->inst, index);
		error = -EFAULT;
		goto out;
	}
	error = psd->indirect_ops->read_special(psd, index, flags, param, data, len);

out:
	mutex_unlock(&psd->ind_lock);
	return error;
}

static void ind_stats_show(struct seq_file *m, sdd_data_t *psd)
{
	seq_printf(m, "\nStats\n");
	seq_printf(m, "%09u ZoneA presses\n", psd->zonea_press_count);
	seq_printf(m, "%09u ZoneA repeats\n", psd->zonea_repeat_count);
	seq_printf(m, "%09u ZoneA releases\n", psd->zonea_release_count);
	seq_printf(m, "%09u ZoneB presses\n", psd->zoneb_press_count);
	seq_printf(m, "%09u ZoneB repeats\n", psd->zoneb_repeat_count);
	seq_printf(m, "%09u ZoneB releases\n", psd->zoneb_release_count);
	seq_printf(m, "%09u ZoneC presses\n", psd->zonec_press_count);
	seq_printf(m, "%09u ZoneC repeats\n", psd->zonec_repeat_count);
	seq_printf(m, "%09u ZoneC releases\n", psd->zonec_release_count);
	if (psd->gesture_match->zonem != 0) {
		seq_printf(m, "%09u ZoneM presses\n", psd->zonem_press_count);
		seq_printf(m, "%09u ZoneM repeats\n", psd->zonem_repeat_count);
		seq_printf(m, "%09u ZoneM releases\n", psd->zonem_release_count);
	}
	seq_printf(m, "%09u Swipes\n", psd->swipe_count);
	seq_printf(m, "\n");
}

static int ind_show(struct seq_file *m, void *v)
{
	sdd_data_t *psd = sdd_get_data(0);
	uintptr_t lopt = (uintptr_t) (m->private);
	u8 option = lopt;
	u32 val = 0;
	int i, error;

	switch (option) {
	case SGE_REG_IND_IDX_STATS:
		ind_stats_show(m, psd);
		break;
	case SGE_REG_IND_IDX_PARASITIC_CAP:
		for (i = 0; i < psd->num_sensors; i++) {
			error = sdd_ind_get_sensor32(psd, option, i, &val);
			if (error) {
				seq_printf(m, "error reading index 0x%x: %d\n", option, error);
				return 0;
			} else {
				seq_printf(m, "%s%04x", (i == 0) ? "" : ",", val);
			}
		}
		seq_printf(m, "\n");
		break;
	case SGE_REG_IND_IDX_FINGER_HYSTERESIS:
		error = sdd_ind_get_finger_hysteresis(psd, (u8 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_FINGER_THRESHOLD:
		error = sdd_ind_get_finger_threshold(psd, (u8 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_REPEAT_TIME:
		error = sdd_ind_get_repeat_time(psd, (u16 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_HOLD_TIME:
		error = sdd_ind_get_hold_time(psd, (u16 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_NOISE_THRESHOLD:
		error = sdd_ind_get_noise_threshold(psd, (u8 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_MOVE_INTERVAL:
		error = sdd_ind_get_move_interval(psd, (u8 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_DIE_TEMP:
		error = sdd_ind_get_die_temp(psd, (u8 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_CRC32:
		error = sdd_ind_get_crc32(psd, &val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_NUM_SENSORS:
		error = sdd_ind_get_num_sensors(psd, (u8 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_BASELINE_INITS:
		error = sdd_ind_get_baseline_inits(psd, (u8 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_NO_DIFFS:
		error = sdd_ind_get_no_diffs(psd, (u8 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_PMU_PULSES:
		error = sdd_ind_get_pmu_pulses(psd, (u16 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_DEEP_SLEEPS:
		error = sdd_ind_get_deep_sleeps(psd, (u16 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case  SGE_REG_IND_IDX_RAW_CAP:
	case  SGE_REG_IND_IDX_MOD_IDAC:
	case  SGE_REG_IND_IDX_COMP_IDAC:
	case  SGE_REG_IND_IDX_SENSE_CLK_DIV:
	case  SGE_REG_IND_IDX_MOD_CLK_DIV:
	case  SGE_REG_IND_IDX_SIGNAL:
	case  SGE_REG_IND_IDX_BASELINE:
	case  SGE_REG_IND_IDX_MAX_SIG:
		for (i = 0; i < psd->num_sensors; i++) {
			val = 0;
			error = sdd_ind_get_sensor16(psd, option, i, (u16 *)&val);
			if (error) {
				seq_printf(m, "error reading index 0x%x: %d\n", option, error);
				return 0;
			} else {
				seq_printf(m, "%s%04x", (i == 0) ? "" : ",", val);
			}
		}
		seq_printf(m, "\n");
		break;
	case SGE_REG_IND_IDX_CMOD_CAP:
		error = sdd_ind_get_cmod_cap(psd, &val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	case SGE_REG_IND_IDX_BLE_CARRIER_CHAN:
		error = sdd_ind_get_ble_carrier_chan(psd, (u8 *)&val);
		if (error) {
			seq_printf(m, "error reading index 0x%x: %d\n", option, error);
		} else {
			seq_printf(m, "%x\n", val);
		}
		break;
	default:
		seq_printf(m, "unknown option %d", option);
	}

	return 0;
}

static int ind_open(struct inode *inode, struct file *file)
{
	return single_open(file, ind_show, PDE_DATA(inode));
}

static ssize_t ind_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	uintptr_t lopt = (uintptr_t) (PDE_DATA(file_inode(filp)));
	u16 option = lopt;
	u32 val;
	sdd_data_t *psd = sdd_get_data(0);
	int error;
	bool log = false;

	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (kstrtoint(buf, 16, &val)) {
		bb_log(BB_MOD_SDD, BB_LVL_WARNING, "Cannot parse register value %s.", buf);
		goto out;
	}

	switch (option) {
	case SGE_REG_IND_IDX_FINGER_HYSTERESIS:
		error = sdd_ind_set_finger_hysteresis(psd, (u8)val);
		if (!error) {
			log = true;
		}
		break;
	case SGE_REG_IND_IDX_FINGER_THRESHOLD:
		error = sdd_ind_set_finger_threshold(psd, (u8)val);
		if (!error) {
			log = true;
		}
		break;
	case SGE_REG_IND_IDX_REPEAT_TIME:
		error = sdd_ind_set_repeat_time(psd, (u16)val);
		if (!error) {
			log = true;
		}
		break;
	case SGE_REG_IND_IDX_HOLD_TIME:
		error = sdd_ind_set_hold_time(psd, (u16)val);
		if (!error) {
			log = true;
		}
		break;
	case SGE_REG_IND_IDX_NOISE_THRESHOLD:
		error = sdd_ind_set_noise_threshold(psd, (u8)val);
		if (!error) {
			log = true;
		}
		break;
	case SGE_REG_IND_IDX_MOVE_INTERVAL:
		error = sdd_ind_set_move_interval(psd, (u8)val);
		if (!error) {
			log = true;
		}
		break;

	default:
		printk(KERN_ERR "write for 0x%x not supported", option);
	}

	if (log) {
		printk(KERN_ERR "set index 0x%x to 0x%x\n", option, val);
	}

out:
	return count;
}

static struct file_operations ind_proc_operations = {
	.owner       = THIS_MODULE,
	.open        = ind_open,
	.write       = ind_write,
	.read        = seq_read,
	.llseek      = seq_lseek,
	.release     = single_release,
};

struct sdd_ind_file {
	char *filename;
	void *option;
} sdd_ind_files[] = {
	{"raw-cap", (void*) SGE_REG_IND_IDX_RAW_CAP},
	{"baseline", (void*) SGE_REG_IND_IDX_BASELINE},
	{"modulation-idac", (void*) SGE_REG_IND_IDX_MOD_IDAC },
	{"compensation-idac", (void*) SGE_REG_IND_IDX_COMP_IDAC},
	{"sense-clk-divider", (void*) SGE_REG_IND_IDX_SENSE_CLK_DIV},
	{"modulation-clk-divider", (void*) SGE_REG_IND_IDX_MOD_CLK_DIV},
	{"signal", (void*) SGE_REG_IND_IDX_SIGNAL},
	{"stats", (void*) SGE_REG_IND_IDX_STATS},
	{"parasitic-cap", (void*) SGE_REG_IND_IDX_PARASITIC_CAP},
	{"max-signal", (void*) SGE_REG_IND_IDX_MAX_SIG},
	{"finger-hysteresis", (void*) SGE_REG_IND_IDX_FINGER_HYSTERESIS},
	{"finger-threshold", (void*) SGE_REG_IND_IDX_FINGER_THRESHOLD},
	{"repeat-time", (void*) SGE_REG_IND_IDX_REPEAT_TIME},
	{"hold-time", (void*) SGE_REG_IND_IDX_HOLD_TIME},
	{"noise-threshold", (void*) SGE_REG_IND_IDX_NOISE_THRESHOLD},
	{"move-interval", (void*) SGE_REG_IND_IDX_MOVE_INTERVAL},
	{"die-temp", (void*) SGE_REG_IND_IDX_DIE_TEMP},
	{"crc32", (void*) SGE_REG_IND_IDX_CRC32},
	{"num-sensors", (void*) SGE_REG_IND_IDX_NUM_SENSORS},
	{"baseline-inits", (void*) SGE_REG_IND_IDX_BASELINE_INITS},
	{"no-diffs", (void*) SGE_REG_IND_IDX_NO_DIFFS},
	{"pmu-pulses", (void*) SGE_REG_IND_IDX_PMU_PULSES},
	{"deep-sleeps", (void*) SGE_REG_IND_IDX_DEEP_SLEEPS},
	{"cmod-cap", (void*) SGE_REG_IND_IDX_CMOD_CAP},
	{"ble-carrier-channel", (void*) SGE_REG_IND_IDX_BLE_CARRIER_CHAN},
	{NULL, (void*) 0}};

struct proc_dir_entry *sdd_parent_proc;

struct sdd_indirect_ops sdd_indirect_ops_nocop = {
	.read = sdd_ind_read_nocop,
	.write = sdd_ind_write_nocop,
	.read_special = sdd_ind_read_special_nocop
};

void sdd_ind_ops_nocop(sdd_data_t *psd)
{
	psd->indirect_ops = &sdd_indirect_ops_nocop;
}

int sdd_ind_init(sdd_data_t *psd)
{
	struct proc_dir_entry *entry;
	int i;

	sdd_parent_proc = proc_mkdir("driver/sdd-ind",0);
	if (sdd_parent_proc == NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Couldn't create /proc/driver/sdd-ind");
		return -EIO;
	}

	for (i = 0; sdd_ind_files[i].filename != NULL; i++) {
		entry = proc_create_data(sdd_ind_files[i].filename, 0666, sdd_parent_proc,
					&ind_proc_operations, sdd_ind_files[i].option);
		if (entry == NULL) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Unable to create IND file %s", sdd_ind_files[i].filename);
			return -EIO;
		}
	}
	return 0;
}

void sdd_ind_remove(void)
{
	int i;
	for (i = 0; sdd_ind_files[i].filename != NULL; i++) {
		remove_proc_entry(sdd_ind_files[i].filename, sdd_parent_proc);
	}
	remove_proc_entry("driver/sdd-ind", NULL);
}

void sdd_ind_update_stats(sdd_data_t *psd, u16 old, u16 new)
{
	sdd_gesture_match_t *gest = psd->gesture_match;

	if (!(old & SGE_REG_GRR_TOUCH) && (new & SGE_REG_GRR_TOUCH)) {
		psd->touch_count++;
	}
	if (!(old & gest->zonea) && (new & gest->zonea)) {
		psd->zonea_press_count++;
	} else if ((old & gest->zonea) && !(new & gest->zonea)) {
		psd->zonea_release_count++;
	}
	if (!(old & gest->zonec) && (new & gest->zonec)) {
		psd->zonec_press_count++;
	} else if ((old & gest->zonec) && !(new &gest->zonec )) {
		psd->zonec_release_count++;
	}
	if (!(old & gest->zoneb) && (new & gest->zoneb)) {
		psd->zoneb_press_count++;
	} else if ((old & gest->zoneb) && !(new & gest->zoneb)) {
		psd->zoneb_release_count++;
	}
	if (!(old & gest->zonem) && (new & gest->zonem)) {
		psd->zonem_press_count++;
	} else if ((old & gest->zonem) && !(new & gest->zonem)) {
		psd->zonem_release_count++;
	}

	if (new & gest->zonea_hold) {
		psd->zonea_repeat_count++;
	}
	if (new & gest->zoneb_hold) {
		psd->zoneb_repeat_count++;
	}
	if (new & gest->zonec_hold) {
		psd->zonec_repeat_count++;
	}
	if (new & gest->zonem_hold) {
		psd->zonem_repeat_count++;
	}
	if (!(old & SGE_REG_GRR_PP_SWIPE) && (new & SGE_REG_GRR_PP_SWIPE)) {
		psd->swipe_count++;
	}
}


void sdd_ind_clear_max_signal(sdd_data_t *psd)
{
	u8 z=0, sensor;
	for (sensor = 0; sensor < psd->num_sensors; sensor++) {
		sdd_ind_read(psd, SGE_REG_IND_IDX_CLEAR_MAX_SIG, SGE_REG_IND_FLAGS_SENSOR, sensor, &z, 1);
	}
}

void sdd_ind_check_rawcap(sdd_data_t *psd)
{
	int error, sensor;
	u16 val;

	for (sensor = 0; sensor < psd->num_sensors; sensor++) {
		error = sdd_ind_get_sensor16(psd, SGE_REG_IND_IDX_RAW_CAP, sensor, (u16 *)&val);
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: read raw error=%d sensor=%d", __func__, error, sensor);
		}
		if (val < 0x40) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "sensor%d raw capacitance (%04x) invalid", sensor, val);
		}
	}
}

int sdd_ind_set_finger_hysteresis(sdd_data_t *psd, u8 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_FINGER_HYSTERESIS, 0, 0, &val, 1);
}

int sdd_ind_get_finger_hysteresis(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_FINGER_HYSTERESIS, 0, 0, val, 1);
}

int sdd_ind_set_finger_threshold(sdd_data_t *psd, u8 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_FINGER_THRESHOLD, 0, 0, &val, 1);
}

int sdd_ind_get_finger_threshold(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_FINGER_THRESHOLD, 0, 0, val, 1);
}

int sdd_ind_get_repeat_time(sdd_data_t *psd, u16 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_REPEAT_TIME, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_set_repeat_time(sdd_data_t *psd, u16 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_REPEAT_TIME, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)&val, sizeof(val));
}

int sdd_ind_get_hold_time(sdd_data_t *psd, u16 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_HOLD_TIME, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_set_hold_time(sdd_data_t *psd, u16 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_HOLD_TIME, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)&val, sizeof(val));
}

int sdd_ind_get_noise_threshold(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_NOISE_THRESHOLD, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_set_noise_threshold(sdd_data_t *psd, u8 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_NOISE_THRESHOLD, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)&val, sizeof(val));
}

int sdd_ind_get_move_interval(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_MOVE_INTERVAL, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_set_move_interval(sdd_data_t *psd, u8 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_MOVE_INTERVAL, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)&val, sizeof(val));
}

int sdd_ind_get_die_temp(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_DIE_TEMP, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_get_crc32(sdd_data_t *psd, u32 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_CRC32, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_get_num_sensors(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_NUM_SENSORS, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_get_baseline_inits(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_BASELINE_INITS, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_get_no_diffs(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_NO_DIFFS, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_get_pmu_pulses(sdd_data_t *psd, u16 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_PMU_PULSES, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_get_deep_sleeps(sdd_data_t *psd, u16 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_DEEP_SLEEPS, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_get_sensor16(sdd_data_t *psd, u8 index, u8 sensor, u16 *val)
{
	return sdd_ind_read(psd, index, SGE_REG_IND_FLAGS_SENSOR, sensor, (u8 *)val, sizeof(*val));
}

int sdd_ind_get_sensor32(sdd_data_t *psd, u8 index, u8 sensor, u32 *val)
{
	return sdd_ind_read(psd, index, SGE_REG_IND_FLAGS_SENSOR, sensor, (u8 *)val, sizeof(*val));
}

int sdd_ind_get_cmod_cap(sdd_data_t *psd, u32 *val)
{
	return sdd_ind_read_special(psd, SGE_REG_IND_IDX_CMOD_CAP, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_set_ble_carrier_chan(sdd_data_t *psd, u8 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_BLE_CARRIER_CHAN, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)&val, sizeof(val));
}

int sdd_ind_get_ble_carrier_chan(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_BLE_CARRIER_CHAN, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_transmit_ble_carrier(sdd_data_t *psd, u8 enable)
{
	int error;

	if (psd && psd->btle_enabled) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: unable to continue because the BLE stack is running",
			__func__);
		return -EACCES;
	}
	error = sdd_ind_write(psd, SGE_REG_IND_IDX_TX_BLE_CARRIER, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)&enable, sizeof(enable));
	if (!error) {
		psd->ble_carrier_tx_enabled = enable;
	}
	return error;
}

int sdd_ind_set_eco_trim_cap(sdd_data_t *psd, u32 trim)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_ECO_TRIM_CAP, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)&trim, sizeof(trim));
}

int sdd_ind_get_eco_trim_cap(sdd_data_t *psd, u32 *trim)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_ECO_TRIM_CAP, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)trim, sizeof(*trim));
}

int sdd_ind_get_tpintr_delta(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_TP_INTR_DELTA, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)val, sizeof(*val));
}

int sdd_ind_set_tpintr_delta(sdd_data_t *psd, u8 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_TP_INTR_DELTA, SGE_REG_IND_FLAGS_NONE, 0, (u8 *)&val, sizeof(val));
}

int sdd_ind_set_prox_finger_hysteresis(sdd_data_t *psd, u8 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_FINGER_HYSTERESIS, 0, 0, &val, 1);
}

int sdd_ind_get_prox_finger_hysteresis(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_FINGER_HYSTERESIS, 0, 0, val, 1);
}

int sdd_ind_set_prox_finger_threshold(sdd_data_t *psd, u8 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_PROX_FINGER_THRESH, 0, 0, &val, 1);
}

int sdd_ind_get_prox_finger_threshold(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_PROX_FINGER_THRESH, 0, 0, val, 1);
}

int sdd_ind_set_prox_noise_threshold(sdd_data_t *psd, u8 val)
{
	return sdd_ind_write(psd, SGE_REG_IND_IDX_PROX_NOISE_THRESH, 0, 0, &val, 1);
}

int sdd_ind_get_prox_noise_threshold(sdd_data_t *psd, u8 *val)
{
	return sdd_ind_read(psd, SGE_REG_IND_IDX_PROX_NOISE_THRESH, 0, 0, val, 1);
}

