/*
 * Copyright (c) 2018-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */


#include <linux/kernel.h>
#include <linux/of_address.h>
#include <asm/io.h>

#include "sdd.h"
#include "blackbox.h"
#include "sdd_rt_pmux.h"

#define IOMUXC_PARAMS_DTS_STR		"iomuxc-params"
#define IOMUXC_ADDR_DTS_STR		"iomuxc-ofs"
#define IOMUXC_INP_SEL_FUNC_ALT3	(0x0003)
#define IOMUXC_INP_SEL_FUNC_ALT5	(0x0005)
#define IOMUXC_GPIO_DRIVE_MODE1		(0xa0b1)
#define IOMUXC_GPIO_DRIVE_MODE2		(0x30b1)
#define PAD_SD3D6_ALT3__UART3RTS	(0x0002)

static void __iomem *iomuxc_base;

enum {
	IOMUXC_PARAM_BASE = 0,
	IOMUXC_PARAM_LEN,
	IOMUXC_PARAM_MUX_S3D6,
	IOMUXC_PARAM_MUX_S3D7,
	IOMUXC_PARAM_CTL_S3D6,
	IOMUXC_PARAM_CTL_S3D7,
	IOMUXC_PARAM_UART3_RTSSEL,
	IOMUXC_NUM_REGS
};

struct iomuxc_dts_param {
	const char *param;
	u32 value;
};

static struct iomuxc_dts_param param_list[IOMUXC_NUM_REGS] = {
	[IOMUXC_PARAM_BASE] = { .param = "base-addr" },
	[IOMUXC_PARAM_LEN]  = { .param = "len" },
	[IOMUXC_PARAM_MUX_S3D6] = { .param = "mux-s3d6" },
	[IOMUXC_PARAM_MUX_S3D7] = { .param = "mux-s3d7" },
	[IOMUXC_PARAM_CTL_S3D6] = { .param = "ctl-s3d6" },
	[IOMUXC_PARAM_CTL_S3D7] = { .param = "ctl-s3d7" },
	[IOMUXC_PARAM_UART3_RTSSEL] = { .param = "uart3-rts-sel" },
};

int sdd_rt_pmux_used = 0;

int sdd_rt_pmux_init(sdd_data_t *psd)
{
	int reg_idx, ofs_idx, ret;
	struct device_node *node = psd->i2c_client->dev.of_node;

	if (sdd_rt_pmux_used) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: data structure in use", __func__);
		return -1;
	}

	if (!node) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: device_node pointer is null.", __func__);
		return -1;
	}

	for (reg_idx = 0; reg_idx < IOMUXC_NUM_REGS; reg_idx++) {
		ofs_idx = of_property_match_string(node, IOMUXC_PARAMS_DTS_STR, param_list[reg_idx].param);
		if (ofs_idx < 0) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: failure to call of_property_match_string. err: %d", __func__, ofs_idx);
			return -1;
		}

		ret = of_property_read_u32_index(node, IOMUXC_ADDR_DTS_STR, ofs_idx, &param_list[reg_idx].value);
		if (ret) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: failed to read %s property at index %d: %d", __func__, IOMUXC_ADDR_DTS_STR, ofs_idx, ret);
			return -1;
		}
	}

	iomuxc_base = ioremap(param_list[IOMUXC_PARAM_BASE].value, param_list[IOMUXC_PARAM_LEN].value);
	if (!iomuxc_base) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: iomuxc_base pointer is null.", __func__);
		return -1;
	}

	sdd_rt_pmux_used = 1;
	return 0;
}


void sdd_rt_pmux_disable_uart(void)
{
	void __iomem *iomuxc_mux_sd3_data6;
	void __iomem *iomuxc_mux_sd3_data7;
	void __iomem *iomuxc_ctl_sd3_data6;
	void __iomem *iomuxc_ctl_sd3_data7;

	iomuxc_mux_sd3_data6 = iomuxc_base + param_list[IOMUXC_PARAM_MUX_S3D6].value;
	iomuxc_mux_sd3_data7 = iomuxc_base + param_list[IOMUXC_PARAM_MUX_S3D7].value;
	iomuxc_ctl_sd3_data6 = iomuxc_base + param_list[IOMUXC_PARAM_CTL_S3D6].value;
	iomuxc_ctl_sd3_data7 = iomuxc_base + param_list[IOMUXC_PARAM_CTL_S3D7].value;

	iowrite32(IOMUXC_INP_SEL_FUNC_ALT5, iomuxc_mux_sd3_data6);
	iowrite32(IOMUXC_INP_SEL_FUNC_ALT5, iomuxc_mux_sd3_data7);

	iowrite32(IOMUXC_GPIO_DRIVE_MODE1, iomuxc_ctl_sd3_data6);
	iowrite32(IOMUXC_GPIO_DRIVE_MODE1, iomuxc_ctl_sd3_data7);
}

void sdd_rt_pmux_enable_uart(void)
{
	void __iomem *iomuxc_mux_sd3_data6;
	void __iomem *iomuxc_mux_sd3_data7;
	void __iomem *iomuxc_ctl_sd3_data6;
	void __iomem *iomuxc_ctl_sd3_data7;
	void __iomem *iomuxc_uart3_rts_selinp;

	iomuxc_mux_sd3_data6 = iomuxc_base + param_list[IOMUXC_PARAM_MUX_S3D6].value;
	iomuxc_mux_sd3_data7 = iomuxc_base + param_list[IOMUXC_PARAM_MUX_S3D7].value;
	iomuxc_ctl_sd3_data6 = iomuxc_base + param_list[IOMUXC_PARAM_CTL_S3D6].value;
	iomuxc_ctl_sd3_data7 = iomuxc_base + param_list[IOMUXC_PARAM_CTL_S3D7].value;
	iomuxc_uart3_rts_selinp = iomuxc_base + param_list[IOMUXC_PARAM_UART3_RTSSEL].value;

	iowrite32(IOMUXC_INP_SEL_FUNC_ALT3, iomuxc_mux_sd3_data6);
	iowrite32(IOMUXC_INP_SEL_FUNC_ALT3, iomuxc_mux_sd3_data7);

	iowrite32(IOMUXC_GPIO_DRIVE_MODE2, iomuxc_ctl_sd3_data6);
	iowrite32(PAD_SD3D6_ALT3__UART3RTS, iomuxc_uart3_rts_selinp);
}

void sdd_rt_pmux_exit(void)
{
	iounmap(iomuxc_base);
	sdd_rt_pmux_used = 0;
}
